import os
import posixpath
import errno
import messages
from ftplib import FTP, error_perm
from ftp_migrator.tools import safe_format


def log(message):
    print message


class Ftp(object):
    def __init__(self, host, username, password):
        log(safe_format(messages.DEBUG_FTP_CONNECTION, host=host, username=username))
        self._ftp = FTP(host)
        self._ftp.login(username, password)

    def download_path(self, remote_path, local_path):
        try:
            self._ftp.cwd(remote_path)
            # change directory command was executed successfully so given path is a directory
            # create corresponding directory on target
            mkdir_p(local_path)
            # retrieve list of nested items (both files and directories) and process it
            for item in self._ftp.nlst():
                if item.startswith('plesk-migrator-agent'):
                    # skip migrator agent files when transferring - they are garbage for the target server
                    # now simply match by name pattern, better implementation should pass list of excluded
                    # directories as argument to the FTP migrator
                    continue

                self.download_path(
                    posixpath.join(remote_path, item),
                    os.path.join(local_path, item)
                )
        except error_perm:
            # change directory command was failed with 5xx error code so consider that given path is a file
            self.download_file(remote_path, local_path)
        except Exception as e:
            log(safe_format(messages.FAILED_TO_RETRIEVE_PATH, remote_path=remote_path, message=e))

    def download_file(self, source_path, target_path):
        log(safe_format(messages.DEBUG_FTP_DOWNLOAD_FILE, source_path=source_path, target_path=target_path))
        try:
            with open(target_path, 'wb') as fp:
                def _write_target_file_content(content):
                    fp.write(content)
                self._ftp.retrbinary('RETR %s' % source_path, _write_target_file_content)
        except error_perm as e:
            log(safe_format(
                messages.DEBUG_FTP_DOWNLOAD_FILE_FAILED, source_path=source_path,
                target_path=target_path, message=e
            ))
        except Exception as e:
            log(safe_format(
                messages.DEBUG_FTP_DOWNLOAD_FILE_FAILED, source_path=source_path,
                target_path=target_path, message=e
            ))


def mkdir_p(path):
    """Create directory recursive

    :rtype: str | unicode
    :rtype: None
    """
    try:
        os.makedirs(path)
    except OSError as exc:
        if exc.errno == errno.EEXIST and os.path.isdir(path):
            pass
        else:
            raise
