from parallels.core import messages
import logging

from parallels.core.actions.base.subscription_action import SubscriptionAction
from parallels.core.utils.plesk_api_utils import check_if_subscription_exists
from parallels.api.plesk import operator as plesk_ops
from parallels.core.utils.common import format_list

logger = logging.getLogger(__name__)


class VerifyHostingSettings(SubscriptionAction):
	def get_description(self):
		return messages.VERIFY_THAT_HOSTING_SETTINGS_WERE_RESTORED

	def is_critical(self):
		"""If action is critical or not

		If action is critical and it failed for a subscription, migration tool
		won't run the next operations for the subscription.

		:rtype: bool
		"""
		# We explicitly completely fail subscriptions for which hosting settings were not restored.
		# For all other failures, consider them not critical.
		return False

	def filter_subscription(self, global_context, subscription):
		"""
		:type global_context: parallels.core.global_context.GlobalMigrationContext
		:type subscription: parallels.core.migrated_subscription.MigratedSubscription
		"""
		return True

	def get_failure_message(self, global_context, subscription):
		"""
		:type global_context: parallels.core.global_context.GlobalMigrationContext
		:type subscription: parallels.core.migrated_subscription.MigratedSubscription
		"""
		return messages.FAILED_VERIFY_THAT_HOSTING_SETTINGS_WERE % subscription.name

	def run(self, global_context, subscription):
		"""
		:type global_context: parallels.core.global_context.GlobalMigrationContext
		:type subscription: parallels.core.migrated_subscription.MigratedSubscription
		"""
		safe = global_context.safe

		logger.debug(messages.CHECK_IF_SUBSCRIPTION_WAS_RESTORED_IT)
		if not check_if_subscription_exists(subscription.panel_target_server.plesk_api(), subscription.name):
			safe.fail_subscription(
				subscription.name,
				messages.SUBSCRIPTION_WAS_NOT_RESTORED_IT_DOES)

		logger.debug(messages.CHECK_THAT_EACH_SITE_SUBSCRIPTION_WAS)
		missing_sites = self._get_missing_sites(subscription)
		if len(missing_sites) > 0:
			safe.fail_subscription(
				subscription.name,
				messages.FOLLOWING_SITES_SUBSCRIPTION_WERE_NOT_RESTORED % (
					format_list(missing_sites)
				)
			)

	@staticmethod
	def _get_missing_sites(subscription):
		"""
		:type subscription: parallels.core.migrated_subscription.MigratedSubscription
		"""
		site_names = set(
			site.name for site in subscription.converted_backup.iter_sites()
		)
		missing_sites = set()
		for site_name in site_names:
			request = plesk_ops.SiteOperator.Get(
				filter=plesk_ops.SiteOperator.FilterByName([site_name]),
				dataset=[plesk_ops.SiteOperator.Dataset.GEN_INFO]
			)
			results = subscription.panel_target_server.plesk_api().send(request)
			site_missing = True
			for result in results:
				if not hasattr(result, 'code') or result.code != 1013:
					if result.data[1].gen_info.name == site_name:
						site_missing = False
			if site_missing:
				missing_sites.add(site_name)
		return missing_sites
