from parallels.core.utils.paths import web_paths
from parallels.core.utils.paths.copy_web_content import CopyWebContentItem
from parallels.core.utils.paths.web_files import BaseWebFiles


class HelmWebFiles(BaseWebFiles):
    """List web files and directories to be copied from custom panel"""
    def need_to_copy_files(self, global_context, subscription):
        """Check whether we need to copy files of that subscription

        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :type subscription: parallels.core.migrated_subscription.MigratedSubscription
        :rtype: bool
        """
        if subscription.is_fake:
            return any(site.is_virtual_hosting for site in subscription.raw_dump.iter_sites())
        else:
            return subscription.raw_dump.is_virtual_hosting

    def list_files_to_copy(self, global_context, subscription):
        """Make a list of source server directories and files to be transferred.

        :type global_context: parallels.plesk.source.helm3.global_context.Helm3GlobalMigrationContext
        :type subscription: parallels.core.migrated_subscription.MigratedSubscription
        :rtype: list[parallels.core.utils.paths.copy_web_content.CopyWebContentItem]
        """
        vhost_name = subscription.name_idn

        tocopy = []
        exclude = []

        for site in subscription.raw_dump.iter_addon_domains():
            if site.hosting_type != 'phosting':
                continue
            vhost_site_name = site.name.encode('idna')
            tocopy.append(
                CopyWebContentItem(
                    domain_name=site.name,
                    source_path=web_paths.VirtualHostsPath(vhost_site_name + '/wwwroot/'),
                    target_path=web_paths.SiteDocumentRoot(subscription.converted_dump, site),
                    exclude=[],
                    # FTP only subscription may have no wwwroot
                    skip_if_source_not_exists=True
                ),
            )
            tocopy.append(
                CopyWebContentItem(
                    domain_name=site.name,
                    source_path=web_paths.VirtualHostsPath(vhost_site_name + '/'),
                    target_path=web_paths.SitePrivate(subscription.converted_dump, site),
                    exclude=['/wwwroot/'],
                    skip_if_source_not_exists=False
                ),
            )
            exclude.append(vhost_site_name)

        if not subscription.is_fake:
            tocopy.append(
                CopyWebContentItem(
                    domain_name=subscription.name,
                    source_path=web_paths.VirtualHostsPath(vhost_name + '/wwwroot/'),
                    target_path=web_paths.WebspaceDocumentRoot(subscription.converted_dump),
                    exclude=exclude,
                    # FTP only subscription may have no wwwroot
                    skip_if_source_not_exists=True
                ),
            )
            tocopy.append(
                CopyWebContentItem(
                    domain_name=subscription.name,
                    source_path=web_paths.VirtualHostsPath(vhost_name + '/'),
                    target_path=web_paths.WebspaceMainDomainPrivate(subscription.converted_dump),
                    exclude=['/wwwroot/', '/SSL/'],
                    skip_if_source_not_exists=False
                ),
            )
            tocopy.append(
                CopyWebContentItem(
                    domain_name=subscription.name,
                    source_path=web_paths.VirtualHostsPath(vhost_name + '/SSL/'),
                    target_path=web_paths.WebspaceSSLDocumentRoot(subscription.converted_dump),
                    exclude=[],
                    skip_if_source_not_exists=True
                ),
            )

        return tocopy

    def get_source_vhosts_dir(self, global_context, source_server):
        """Get directory where virtual hosts data is located.

        This function could return None when we don't know where virtual hosts directory is located,
        for example if we don't have direct access to the source server, but have only manually configured
        rsync server.

        :type source_server: parallels.core.connections.source_server.SourceServer
        :type global_context: parallels.plesk.source.helm3.global_context.Helm3GlobalMigrationContext
        :rtype: str | unicode | None
        """
        return global_context.helm3_agent.get_vhosts_dir_source(source_server)