import logging
import os
import posixpath
import threading
from collections import namedtuple

from parallels.core import messages
from parallels.core.utils import yaml_utils
from parallels.core.utils.common import is_empty, default
from parallels.core.utils.common.threading_utils import synchronized_by_args

logger = logging.getLogger(__name__)

KeyInfo = yaml_utils.pretty_yaml(
    namedtuple('KeyInfo', ('key_pathname', 'public_key'))
)


DEFAULT_AUTHORIZED_KEYS_LOCATIONS = [
    # On certain configurations, there is only the second file enabled,
    # so configure keys in both of the files
    '/root/.ssh/authorized_keys',
    '/root/.ssh/authorized_keys2'
]


class SSHKeyPool(object):
    def __init__(self, filename):
        self.keys = dict()
        self.lock = threading.Lock()
        self.filename = filename
        self.readonly = False

    @synchronized_by_args
    def get(self, source_server, target_server):
        """
        :return: parallels.core.utils.ssh_key_pool.KeyInfo
        """
        with self.lock:
            if self.readonly:
                raise Exception(messages.SSH_KEY_POOL_IS_READ_ONLY)

            if target_server.is_local() and source_server.ssh_key() is not None:
                return KeyInfo(
                    key_pathname=source_server.ssh_key(),
                    # we have no information about public key, we don't require to specify it in configuration file
                    # and actually it is not necessary
                    public_key=None
                )

            if (source_server, target_server,) not in self.keys:
                logger.debug(
                    messages.DEBUG_CONFIGURE_SSH_KEY_AUTH,
                    target_server.description(), source_server.description()
                )
                key_pathname, public_key = self._set_up_keys(
                    source_server, target_server,
                    authorized_keys_paths=self._get_server_authorized_keys_paths(source_server)
                )
                self.keys[(source_server, target_server)] = KeyInfo(
                    key_pathname, public_key
                )
                self._save_keys()

            return self.keys[(source_server, target_server)]

    def remove_all(self):
        for (source_server, target_server), key_info in self.keys.iteritems():
            logger.debug(
                messages.DEBUG_REMOVE_SSH_KEY_AUTH,
                target_server.description(),
                source_server.description()
            )
            try:
                self._remove_keys(
                    source_server, target_server,
                    key_pathname=key_info.key_pathname,
                    public_key=key_info.public_key,
                    authorized_keys_paths=self._get_server_authorized_keys_paths(source_server)
                )
            except Exception as e:
                logger.debug(messages.LOG_EXCEPTION, exc_info=True)
                logger.error(
                    messages.FAILED_TO_REMOVE_SSH_KEY,
                    target_server.description(), source_server.description(), e
                )
        self.keys = {}
        self._save_keys()
        self.readonly = True

    def _save_keys(self):
        keys = {}
        for (source_server, target_server), key_info in self.keys.iteritems():
            keys[source_server.ip(), target_server.ip()] = key_info
        yaml_utils.write_yaml(self.filename, keys)

    @staticmethod
    def _get_server_authorized_keys_paths(server):
        if (
            hasattr(server, 'node_settings') and
            hasattr(server.node_settings, 'ssh_authorized_keys_file') and
            not is_empty(server.node_settings.ssh_authorized_keys_file)
        ):
            return [server.node_settings.ssh_authorized_keys_file]
        else:
            with server.runner() as runner:
                root_home_directory = runner.sh("echo ~").strip()

            return [
                # On certain configurations, there is only the second file enabled,
                # so configure keys in both of the files
                posixpath.join(root_home_directory, '.ssh/authorized_keys'),
                posixpath.join(root_home_directory, '.ssh/authorized_keys2')
            ]

    @staticmethod
    def _set_up_keys(source_server, target_server, authorized_keys_paths=None):
        authorized_keys_paths = default(authorized_keys_paths, DEFAULT_AUTHORIZED_KEYS_LOCATIONS)
        target_ssh_keys_dir = target_server.get_session_file_path("ssh-keys")
        with target_server.runner() as runner:
            runner.mkdir(target_ssh_keys_dir)
            private_key_filename = os.path.join(target_ssh_keys_dir, 'id_rsa.%s' % source_server.ip())
            public_key_filename = '%s.pub' % private_key_filename
            runner.sh(
                'echo y | ssh-keygen -q -t rsa -P "" -f {private_key_filename}',
                dict(
                    private_key_filename=private_key_filename
                )
            )
            public_key_contents = runner.get_file_contents(public_key_filename).strip()

        for authorized_keys_path in authorized_keys_paths:
            with source_server.runner() as runner:
                runner.mkdir(posixpath.dirname(authorized_keys_path))
                # Don't use upload_file_content function of runner, as rsync requires keys
                # we're trying to upload here. Use shell echo instead.
                runner.sh('echo {public_key_contents} >> {authorized_keys_path}', dict(
                    public_key_contents=public_key_contents, authorized_keys_path=authorized_keys_path
                ))

        return private_key_filename, public_key_contents

    @staticmethod
    def _remove_keys(source_server, target_server, key_pathname, public_key, authorized_keys_paths=None):
        authorized_keys_paths = default(authorized_keys_paths, DEFAULT_AUTHORIZED_KEYS_LOCATIONS)

        for authorized_keys_path in authorized_keys_paths:
            with source_server.runner() as runner:
                sed_expr = '\|{public_key}|d'.format(public_key=public_key)
                runner.sh(
                    "sed -i -e {sed_expr} {authorized_keys_path}", dict(
                        sed_expr=sed_expr,
                        authorized_keys_path=authorized_keys_path
                    )
                )

        with target_server.runner() as runner:
            runner.remove_file(key_pathname)
            runner.remove_file("%s.pub" % key_pathname)
