import os
import posixpath
import urlparse
import urllib
import urllib2
import logging

from parallels.core import messages
from parallels.core.runners.web.exceptions import WebRpcAgentException

from parallels.core.utils.agent_pool import AgentPool
from parallels.core.utils.common import cached
from parallels.core.utils.ftp import Ftp
from parallels.core.utils.json_utils import encode_json, decode_json
from parallels.core.utils.migrator_utils import download_thirdparty_zip
from parallels.plesk.source.web.connections import WebSourceConfig

logger = logging.getLogger(__name__)


class WebAgentPool(AgentPool):

    @staticmethod
    @cached
    def get_instance():
        """Return pool of transfer agents

        :rtype: WebAgentPool
        """
        return WebAgentPool()

    def get(self, server_config, proxy_to=None, allow_autodeploy=True):
        """Retrive instance of web RPC agent for given server

        :type server_config: parallels.plesk.source.web.config.WebSourceConfig
        :type proxy_to: str | None
        :type allow_autodeploy: bool
        :rtype: parallels.core.runners.web.utils.WebAgent
        """
        return super(WebAgentPool, self).get(server_config, proxy_to, allow_autodeploy)

    def _create_agent(self, server_config, proxy_to, allow_autodeploy):
        assert (isinstance(server_config, WebSourceConfig))
        agent = WebAgent(
            server_config.ftp_host,
            server_config.ftp_username,
            server_config.ftp_password,
            server_config.document_root,
            server_config.base_url
        )
        if allow_autodeploy:
            agent.deploy()
        return agent


class WebAgent(object):

    def __init__(self, ftp_host, ftp_username, ftp_password, document_root_path, base_url):
        self._ftp = Ftp(ftp_host, ftp_username, ftp_password)
        self._document_root_path = document_root_path
        self._base_url = base_url

    @property
    def url(self):
        return urlparse.urljoin(self._base_url, 'panel-migrator-web-rpc-agent.php')

    @property
    def script_path(self):
        return posixpath.join(self._document_root_path, 'panel-migrator-web-rpc-agent.php')

    @property
    def base_ftp_path(self):
        return self._document_root_path

    def get_ftp_path(self, path):
        return posixpath.join(self.base_ftp_path, path)

    def deploy(self):
        # retrieve path to web RPC agent on server where Plesk Migrator is running
        web_rpc_agent_dir = download_thirdparty_zip(
            'panel-migrator-web-rpc-agent_1.0.zip', 'panel-migrator-web-rpc-agent', messages.WEB_RPC_AGENT_DOWNLOAD
        )
        agent_local_path = os.path.join(web_rpc_agent_dir, 'panel-migrator-web-rpc-agent.php')

        # upload web RPC agent on remote server via FTP
        self._ftp.upload_file(agent_local_path, self.script_path)

    def is_windows(self):
        return self._process_request('is_windows')

    def is_file_exists(self, file_path):
        return self._process_request('is_file_exists', {'file_path': file_path})

    def get_files_list(self, dir_path):
        return self._process_request('get_files_list', {'dir_path': dir_path})

    def mkdir(self, dir_path):
        self._process_request('mkdir', {'dir_path': dir_path})

    def download_file(self, remote_file_path, local_file_path):
        self._ftp.download_file_local(self.get_ftp_path(remote_file_path), local_file_path)

    def upload_file(self, local_file_path, remote_file_path):
        self._ftp.upload_file(local_file_path, self.get_ftp_path(remote_file_path))

    def upload_file_content(self, file_path, content):
        self._process_request('upload_file_content', {
            'file_path': file_path,
            'content': content
        })

    def get_file_contents(self, file_path):
        return self._process_request('get_file_contents', {'file_path': file_path})

    def sh_unchecked(self, cmd_str, stdin_content, env, log_output, working_dir, redirect_output_file):
        result = self._process_request('exec', {
            'cmd': cmd_str,
            'stdin_content': stdin_content,
            'env': env,
            'log_output': log_output,
            'working_dir': working_dir,
            'redirect_output_file': redirect_output_file
        })

        if not isinstance(result, dict) or {'exit_code', 'stdout', 'stderr'} != set(result):
            raise WebRpcAgentException(messages.WEB_RPC_AGENT_EXCEPTION_TASK_SH_UNCHECKED_WRONG_DATA)

        return result.get('exit_code'), result.get('stdout'), result.get('stderr')

    def _process_request(self, task, data=None):
        """Send request to web RPC agent to process given task with given data

        :type task: str
        :type data: dict | None
        :rtype: int, dict
        """
        request = urllib2.Request('{url}?task={task}'.format(url=self.url, task=task))
        json_encoded_data = encode_json(data)
        request.add_data(urllib.urlencode({'data': json_encoded_data}))

        logger.debug(messages.WEB_RPC_AGENT_REQUEST.format(url=request.get_full_url(), data=json_encoded_data))
        response = urllib2.urlopen(request)
        logger.debug(messages.WEB_RPC_AGENT_RESPONSE.format(code=response.getcode()))

        if response.getcode() != 200:
            raise WebRpcAgentException(messages.WEB_RPC_AGENT_EXCEPTION_RESPONSE_WRONG_HTTP_CODE.format(
                task=task, code=response.getcode()
            ))

        try:
            result = decode_json(response.read())
            logger.debug(messages.WEB_RPC_AGENT_RESPONSE_DATA.format(data=repr(result)))
        except Exception as e:
            logger.debug(messages.LOG_EXCEPTION, exc_info=True)
            raise WebRpcAgentException(messages.WEB_RPC_AGENT_EXCEPTION_RESPONSE_WRONG_DATA.format(
                task=task, error=e
            ))

        return result
