from parallels.core import messages
from parallels.core.registry import Registry
from parallels.core.reports.model.affected_object import AffectedObject
from parallels.core.utils.common import is_empty, default, format_multiline_list, all_equal
from parallels.core.utils.entity import Entity


class Issue(Entity):
    SEVERITY_INFO = 'info'
    SEVERITY_WARNING = 'warning'
    SEVERITY_ERROR = 'error'

    def __init__(
        self, issue_id, severity, problem_text, solution_text=None,
        affected_objects=None, solution_type=None,
        solution_url=None, solution_url_text=None, solution_component_name=None,
        solution_download_rpc_agent=False, execution_command=None, execution_date=None
    ):
        self._issue_id = issue_id
        self._severity = severity
        self._problem_text = problem_text
        self._solution_text = default(solution_text, '')
        self._affected_objects = affected_objects
        self._solution_type = solution_type
        self._solution_url = solution_url
        self._solution_url_text = solution_url_text
        self._solution_component_name = solution_component_name
        self._solution_download_rpc_agent = solution_download_rpc_agent
        self._execution_command = default(execution_command, Registry.get_instance().get_command_name())
        self._execution_date = default(execution_date, Registry.get_instance().get_execution_date())

    @property
    def issue_id(self):
        return self._issue_id

    @property
    def severity(self):
        return self._severity

    @property
    def problem_text(self):
        return self._problem_text

    @property
    def solution_text(self):
        return self._solution_text

    @property
    def affected_objects(self):
        """List of objects affected by the issue

        :rtype: list[parallels.core.reports.model.affected_object.AffectedObject] | None
        """
        return self._affected_objects

    @property
    def solution_type(self):
        return self._solution_type

    @property
    def solution_url(self):
        return self._solution_url

    @property
    def solution_url_text(self):
        return self._solution_url_text

    @property
    def solution_component_name(self):
        return self._solution_component_name

    @property
    def solution_download_rpc_agent(self):
        return self._solution_download_rpc_agent

    @property
    def execution_command(self):
        return self._execution_command

    @property
    def execution_date(self):
        return self._execution_date

    def get_issue_text(self):
        """Get issue as text that could be printed to console or text file

        :rtype: str | unicode
        """
        result = self.problem_text
        if self.affected_objects is not None and len(self.affected_objects) > 0:
            if all_equal([o.object_type for o in self.affected_objects]):
                object_type = self.affected_objects[0].object_type
                if object_type == AffectedObject.TYPE_SUBSCRIPTION:
                    type_title = messages.AFFECTED_TITLE_SUBSCRIPTIONS
                elif object_type == AffectedObject.TYPE_MSSQL_SERVER:
                    type_title = messages.AFFECTED_TITLE_MSSQL_SERVERS
                else:
                    type_title = messages.AFFECTED_TITLE_GENERIC
            else:
                type_title = messages.AFFECTED_TITLE_GENERIC

            result += "\n\n" + type_title + "\n" + format_multiline_list([o.name for o in self.affected_objects])
        if not is_empty(self.solution_text):
            result += "\n\n" + self.solution_text
        return result
