import itertools

from parallels.core import messages
from parallels.core.actions.base.subscription_action import SubscriptionAction
from parallels.core.actions.post_migration_checks.utils import should_run_post_migration_checks
from parallels.core.hosting_check.run_checks import run_entities
from parallels.core.hosting_check.utils.runner_adapter import HostingCheckerRunnerAdapter
from parallels.core.utils import plesk_api_utils
from parallels.core.utils.common import group_by, cached
from parallels.core.utils.common.logging import create_safe_logger
from parallels.core.utils.database_utils import get_mssql_host_for_target_server
from parallels.hosting_check import DomainMSSQLDatabases, DomainMySQLDatabases, DomainPostgreSQLDatabases, \
    TargetPanelDatabase, MySQLDatabase, MySQLDatabaseAccess, User, MSSQLDatabase, MSSQLDatabaseSourceAccess, \
    MSSQLDatabaseTargetAccess, MSSQLDatabaseScriptRunner, PostgreSQLDatabase, PostgreSQLDatabaseAccess
from parallels.hosting_check.checkers.mysql_database_checker import UnixMySQLClientCLI, WindowsMySQLClientCLI

logger = create_safe_logger(__name__)


class TestDatabasesAction(SubscriptionAction):
    def get_description(self):
        """Get short description of action as string

        :rtype: str | unicode
        """
        return messages.ACTION_TEST_DATABASES_DESCRIPTION

    def get_failure_message(self, global_context, subscription):
        """
        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :type subscription: parallels.core.migrated_subscription.MigratedSubscription
        :rtype: str | unicode
        """
        return messages.ACTION_TEST_DATABASES_FAILURE

    def is_critical(self):
        """If action is critical or not

        If action is critical and it failed for a subscription, migration tool
        won't run the next operations for the subscription.

        :rtype: bool
        """
        return False

    def filter_subscription(self, global_context, subscription):
        """Check if we should run this action on given subscription or not

        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :type subscription: parallels.core.migrated_subscription.MigratedSubscription
        :rtype: bool
        """
        return should_run_post_migration_checks(global_context, 'databases')

    def run(self, global_context, subscription):
        """
        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :type subscription: parallels.core.migrated_subscription.MigratedSubscription
        """
        subscription_report = subscription.get_report(global_context.post_migration_check_report_writer)
        database_service_report = subscription_report.subtarget(u'Database service', subscription.name)

        database_service_entities = self._get_database_check_entities(global_context, subscription)
        run_entities(global_context, database_service_entities, database_service_report)

    def _get_database_check_entities(self, global_context, subscription):
        """
        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :type subscription: parallels.core.migrated_subscription.MigratedSubscription
        """
        result = []

        # initialize entities to put information about databases to check
        mysql_databases_entity = DomainMySQLDatabases(
            domain_name=subscription.name,
            target_panel_databases=[],
            databases=[]
        )
        result.append(mysql_databases_entity)
        mssql_databases_entity = DomainMSSQLDatabases(
            domain_name=subscription.name,
            target_panel_databases=[],
            databases=[]
        )
        result.append(mssql_databases_entity)
        pgsql_databases_entity = DomainPostgreSQLDatabases(
            domain_name=subscription.name,
            target_panel_databases=[],
            databases=[]
        )
        result.append(pgsql_databases_entity)

        database_info_source = DatabasesInfoSource(global_context)

        # fill information about databases that exist on target panel
        for entity, db_type in (
            (mysql_databases_entity, 'mysql'),
            (mssql_databases_entity, 'mssql'),
            (pgsql_databases_entity, 'postgresql')
        ):
            target_databases = database_info_source.list_target_panel_databases(
                subscription.name, db_type
            )
            for name, users in target_databases:
                entity.target_panel_databases.append(
                    TargetPanelDatabase(
                        name=name,
                        users=users
                    )
                )

        # fill information about databases that are expected to be restored
        subscription_databases = database_info_source.list_databases_to_copy().get(
            subscription.name, []
        )
        for db_info in subscription_databases:
            db_type = db_info.source_database_server.type()
            if db_type == 'mysql':
                converted_db = self._get_mysql_database_object(database_info_source, subscription, db_info)
                if converted_db is not None:
                    mysql_databases_entity.databases.append(converted_db)
            elif db_type == 'mssql':
                mssql_databases_entity.databases.append(
                    self._get_mssql_database_object(database_info_source, subscription, db_info)
                )
            elif db_type == 'postgresql':
                pgsql_databases_entity.databases.append(
                    self._get_pgsql_database_object(database_info_source, subscription, db_info)
                )

        return result

    @staticmethod
    def _get_mysql_database_object(database_info_source, subscription, database_info):
        """Initialize and return MySQLDatabase object"""

        source = database_info.source_database_server
        target = database_info.target_database_server

        if source.is_windows() == target.is_windows():

            source_mysql_client_cli = database_info_source.get_source_mysql_client_cli(database_info)

            if not target.is_windows():
                target_mysql_client_cli = UnixMySQLClientCLI(
                    HostingCheckerRunnerAdapter(target.runner)
                )
            else:
                target_mysql_client_cli = WindowsMySQLClientCLI(
                    HostingCheckerRunnerAdapter(target.runner),
                    target.mysql_bin
                )

            users = database_info_source.get_db_users(
                subscription.name,
                source.panel_server.node_id,
                database_info.database_name,
                source.type()
            )

            return MySQLDatabase(
                source_access=MySQLDatabaseAccess(
                    mysql_client_cli=source_mysql_client_cli,
                    host=source.host(),
                    port=source.port(),
                    admin_user=User(
                        source.user(),
                        source.password()
                    ),
                ),
                target_access=MySQLDatabaseAccess(
                    mysql_client_cli=target_mysql_client_cli,
                    host=target.host(),
                    port=target.port(),
                    admin_user=User(
                        target.user(),
                        target.password()
                    ),
                ),
                name=database_info.database_name,
                users=users,
            )
        else:
            # we do not copy MySQL databases from Windows to Linux and vice
            # versa
            return None

    @staticmethod
    def _get_mssql_database_object(database_info_source, subscription, database_info):
        """Initialize and return MSSQLDatabase object"""

        source = database_info.source_database_server
        target = database_info.target_database_server

        def get_session_file_path(filename, target_node=target):
            return target_node.get_session_file_path(filename)

        users = database_info_source.get_db_users(
            subscription.name,
            source.panel_server.node_id,
            database_info.database_name,
            source.type()
        )

        return MSSQLDatabase(
            name=database_info.database_name,
            source_access=MSSQLDatabaseSourceAccess(
                host=get_mssql_host_for_target_server(source),
                admin_user=User(source.user(), source.password()),
            ),
            target_access=MSSQLDatabaseTargetAccess(
                host=target.host(),
                admin_user=User(target.user(), target.password()),
            ),
            source_script_runner=MSSQLDatabaseScriptRunner(
                runner=HostingCheckerRunnerAdapter(source.panel_server.runner),
                get_session_file_path=get_session_file_path,
            ),
            target_script_runner=MSSQLDatabaseScriptRunner(
                runner=HostingCheckerRunnerAdapter(target.panel_server.runner),
                get_session_file_path=get_session_file_path,
            ),
            users=users,
        )

    @staticmethod
    def _get_pgsql_database_object(database_info_source, subscription, database_info):
        """Initialize and return PostgreSQLDatabase object"""

        source = database_info.source_database_server
        target = database_info.target_database_server

        users = database_info_source.get_db_users(
            subscription.name,
            source.panel_server.node_id,
            database_info.database_name,
            source.type()
        )

        return PostgreSQLDatabase(
            name=database_info.database_name,
            source_access=PostgreSQLDatabaseAccess(
                runner=HostingCheckerRunnerAdapter(source.runner),
                host=source.host(),
                port=source.port(),
                admin_user=User(source.user(), source.password()),
            ),
            target_access=PostgreSQLDatabaseAccess(
                runner=HostingCheckerRunnerAdapter(target.runner),
                host=target.host(),
                port=target.port(),
                admin_user=User(target.user(), target.password()),
            ),
            users=users,
        )


class DatabasesInfoSource(object):
    def __init__(self, global_context):
        """
        :type global_context: parallels.core.global_context.GlobalMigrationContext
        """
        self._global_context = global_context

    def list_databases_to_copy(self):
        return group_by(
            self._global_context.migrator._list_databases_to_copy(),
            lambda d: d.subscription_name
        )

    @staticmethod
    def get_source_mysql_client_cli(database_info):
        source = database_info.source_database_server
        if source.is_windows():
            return WindowsMySQLClientCLI(
                HostingCheckerRunnerAdapter(source.panel_server.runner),
                source.panel_server.get_path_to_mysql(),
                skip_secure_auth=source.panel_server.mysql_use_skip_secure_auth()
            )
        else:
            return UnixMySQLClientCLI(HostingCheckerRunnerAdapter(source.runner))

    @cached
    def list_target_panel_databases(self, subscription_name, database_type):
        plesk_api = self._global_context.target_panel_obj.get_subscription_plesk_node(
            self._global_context,
            subscription_name
        ).plesk_api()

        return plesk_api_utils.list_databases(plesk_api, subscription_name, database_type)

    def get_db_users(self, subscription_name, source_id, db_name, db_type):
        subscription = self._global_context.get_subscription(subscription_name).raw_dump
        users = []
        for user in itertools.chain(
            [x for x in subscription.iter_per_database_users() if x.database_name == db_name and x.dbtype == db_type],
            [x for x in subscription.iter_overall_database_users() if x.dbtype == db_type]
        ):
            users.append(User(user.name, user.password, user.password_type))
        return users