import logging

from parallels.core.utils.common import is_empty_iterator
from parallels.plesk import messages
from parallels.core.actions.base.subscription_action import SubscriptionAction

logger = logging.getLogger(__name__)


class DeployDomains(SubscriptionAction):
    def get_description(self):
        return messages.ACTION_DEPLOY_DOMAINS

    def get_failure_message(self, global_context, subscription):
        return messages.ACTION_DEPLOY_DOMAINS_FAILED.format(subscription_name=subscription.name)

    def is_critical(self):
        return False

    def filter_subscription(self, global_context, subscription):
        return not is_empty_iterator(subscription.converted_dump.iter_addon_domains())

    def run(self, global_context, subscription):
        """Perform creation of add-on domains for given subscription on target Plesk

        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :type subscription: parallels.core.migrated_subscription.MigratedSubscription
        """
        filter_domain_name = subscription.converted_dump.get_domain_names()

        for domain_dump in subscription.converted_dump.iter_addon_domains():
            if global_context.hosting_repository.domain.is_exists(domain_dump.name, domain_dump.guid):
                logger.info(messages.ACTION_DEPLOY_DOMAINS_CREATE_DOMAIN_EXISTS.format(
                    domain_name=domain_dump.name,
                    domain_guid=domain_dump.guid
                ))
                continue

            if global_context.target_panel_obj.is_remove_coinciding_dns_records():
                # if target panel is unable to create add-on domain which name matching one of existing
                # DNS records, remove such DNS records first
                self._remove_coinciding_dns_records(
                    global_context,
                    subscription.name,
                    domain_dump.name,
                    filter_domain_name
                )

            logger.info(messages.ACTION_DEPLOY_DOMAINS_CREATE_DOMAIN.format(
                domain_name=domain_dump.name,
                subscription_name=subscription.name
            ))

            # invalidate cache associated with this domain
            global_context.cache_state_controllers.domain.invalidate_cache_states(domain_dump.name)

            try:
                global_context.hosting_repository.domain.create_from_dump(domain_dump, subscription.name)
            except Exception:
                logger.debug(messages.LOG_EXCEPTION, exc_info=True)
                # place error into report and proceed to next domain
                global_context.safe.fail_subscription(
                    subscription.name,
                    messages.ACTION_DEPLOY_DOMAINS_CREATE_DOMAIN_ERROR.format(
                        domain_name=domain_dump.name,
                        subscription_name=subscription.name
                    ),
                    is_critical=False
                )
                continue

            if domain_dump.is_forwarding:
                logger.info(messages.ACTION_DEPLOY_DOMAINS_SET_FORWARDING.format(
                    domain_name=domain_dump.name,
                    subscription_name=subscription.name
                ))
                forwarding_url = domain_dump.get_forwarding_url()
                if forwarding_url is None:
                    global_context.safe.fail_subscription(
                        subscription.name,
                        messages.ACTION_DEPLOY_DOMAINS_SET_FORWARDING_ERROR_MISSED_URL.format(
                            domain_name=domain_dump.name,
                            subscription_name=subscription.name
                        ),
                        is_critical=False
                    )
                try:
                    global_context.hosting_repository.domain.set_forwarding(
                        domain_dump.name, forwarding_url, domain_dump.is_frame_forwarding
                    )
                except Exception:
                    logger.debug(messages.LOG_EXCEPTION, exc_info=True)
                    # place error into report and proceed to next domain
                    global_context.safe.fail_subscription(
                        subscription.name,
                        messages.ACTION_DEPLOY_DOMAINS_SET_FORWARDING_ERROR.format(
                            domain_name=domain_dump.name,
                            subscription_name=subscription.name
                        ),
                        is_critical=False
                    )
                    continue
            elif domain_dump.is_virtual_hosting:
                logger.info(messages.ACTION_DEPLOY_DOMAINS_SET_PHYSICAL_HOSTING.format(
                    domain_name=domain_dump.name,
                    subscription_name=subscription.name
                ))
                try:
                    global_context.hosting_repository.domain.set_physical_hosting(
                        domain_dump.name, domain_dump.www_root, domain_dump.is_enable_ssl
                    )
                except Exception:
                    logger.debug(messages.LOG_EXCEPTION, exc_info=True)
                    # place error into report and proceed to next domain
                    global_context.safe.fail_subscription(
                        subscription.name,
                        messages.ACTION_DEPLOY_DOMAINS_SET_PHYSICAL_HOSTING_ERROR.format(
                            domain_name=domain_dump.name,
                            subscription_name=subscription.name
                        ),
                        is_critical=False
                    )
                    continue

    @staticmethod
    def _remove_coinciding_dns_records(context, subscription_name, domain_name, filter_domain_name):
        """Remove DNS records, which match given domain name

        :type context: parallels.core.global_context.GlobalMigrationContext
        :type subscription_name: str
        :type domain_name: str
        :type filter_domain_name: list[str]
        """
        # retrieve DNS records coinciding with the domain name
        dns_records = []
        try:
            dns_records = context.hosting_repository.dns_record.get_list(
                filter_domain_name=filter_domain_name,
                filter_name=['%s.' % domain_name]
            )
        except Exception:
            # unable to retrieve DNS records for some reason, so if such records actually exists,
            # they will not be removed and further creation of add-on domain will fail;
            # anyway, log an exception and go ahead
            logger.debug(messages.LOG_EXCEPTION, exc_info=True)

        if len(dns_records) < 1:
            return

        logger.info(messages.ACTION_DEPLOY_DOMAINS_REMOVE_COINCIDING_DNS_RECORDS.format(
            subscription_name=subscription_name,
            domain_name=domain_name
        ))
        for dns_record in dns_records:
            try:
                logger.info(messages.ACTION_DEPLOY_DOMAINS_REMOVE_COINCIDING_DNS_RECORDS_REMOVE.format(
                    subscription_name=subscription_name,
                    dns_record_domain_name=dns_record.domain_name,
                    dns_record_pretty_name=dns_record.pretty_name
                ))
                context.hosting_repository.dns_record.remove(dns_record)
            except Exception:
                logger.debug(messages.LOG_EXCEPTION, exc_info=True)
                logger.warning(messages.ACTION_DEPLOY_DOMAINS_REMOVE_COINCIDING_DNS_RECORDS_REMOVE_ERROR.format(
                    subscription_name=subscription_name,
                    domain_name=domain_name,
                    dns_record_domain_name=dns_record.domain_name,
                    dns_record_pretty_name=dns_record.pretty_name
                ))
