"""Model with high-level objects that represents our expectations after migration

It includes objects already existing on target panel, and objects that should
be created on target panel.
"""
from collections import namedtuple

from parallels.common.utils.yaml_utils import pretty_yaml
from parallels.utils import mutable_namedtuple

YAML_PREFIX = 'target'


def _nt(*args):
	"""Namedtuple with pretty YAML serialization"""
	return pretty_yaml(namedtuple(*args), prefix=YAML_PREFIX)


def yaml_prettify(cls):
	return pretty_yaml(cls, prefix=YAML_PREFIX)


@yaml_prettify
class Model(namedtuple('Model', ('plans', 'resellers', 'clients'))):
	def iter_all_subscriptions(self):
		for client in self.iter_all_clients():
			for subscription in client.subscriptions:
				yield subscription

	def iter_all_clients(self):
		for client in self.clients.itervalues():
			yield client
		for reseller in self.resellers.itervalues():
			for client in reseller.clients:
				yield client


@yaml_prettify
class HostingPlan(namedtuple('HostingPlan', ('name', 'is_active', 'properties', 'source'))):
	type = 'hosting'

	@property
	def is_windows(self):
		return self.properties.get('service-iis', 0) != 0

	@property
	def is_unix(self):
		return self.properties.get('service-hosting', 0) != 0

	@property
	def platforms(self):
		platforms = set()
		if self.is_windows:
			platforms.add('Windows')
		if self.is_unix:
			platforms.add('Unix')
		return platforms


@yaml_prettify
class ResellerPlan(namedtuple('ResellerPlan', ('name', 'is_active', 'properties', 'source'))):
	type = 'reseller'

	@property
	def is_windows(self):
		return self.properties.get('service-iis', 0) != 0

	@property
	def is_unix(self):
		return self.properties.get('service-hosting', 0) != 0

	@property
	def platforms(self):
		platforms = set()
		if self.is_windows:
			platforms.add('Windows')
		if self.is_unix:
			platforms.add('Unix')
		return platforms

OtherTypePlan = _nt('OtherTypePlan', ('name', 'is_active', 'properties', 'source', 'type'))
AuxiliaryUserRole = _nt('AuxiliaryUserRole', ('name', 'permissions'))
Reseller = _nt('Reseller', (
	'login', 'password', 'plan_name', 'clients', 'company', 
	'personal_info', 'auxiliary_user_roles', 'auxiliary_users', 'is_enabled', 'plans', 'source',
	# object that can be used by import API to create and configure the reseller
	# so it depends on target panel type
	'settings'
))

AuxiliaryUser = pretty_yaml(
	mutable_namedtuple(
		'AuxiliaryUser', (
			'login', 'password', 'name', 'roles', 'personal_info',
			'is_active', 'subscription_name', 'is_domain_admin'
		)
	),
	prefix=YAML_PREFIX
)

Subscription = pretty_yaml(
	mutable_namedtuple('Subscription', (
		'name', 
		'plan_name', 'plan_id', 'addon_plan_ids', 
		'web_ip', 'web_ip_type', 'web_ipv6', 'web_ipv6_type', 'is_enabled', 
		'source', 'is_windows', 'mail_is_windows', 'sub_id', 'group_name', 'group_id',
		'required_resources', 'additional_resources',
		'sysuser_login',
	)),
	prefix=YAML_PREFIX
)


@yaml_prettify
class Client(mutable_namedtuple('Client', (
	'login', 'password', 'subscriptions', 'company', 'personal_info',
	'auxiliary_user_roles', 'auxiliary_users', 'is_enabled', 'source',
	'target_client'
))):
	def is_admin(self):
		return self.login is None

PersonalInfo = _nt('PersonalInfo', (
	'first_name', 'last_name', 'email', 'preferred_email_format', 
	'address_line_1', 'address_line_2', 'city', 'county', 'state', 'postal_code', 'language_code', 
	'locale', 'country_code', 'primary_phone', 'additional_phone', 'fax', 'mobile_phone'))

Plan = _nt('Plan', (
	'name', 
	'is_addon', 
	'source', 
	# object that can be used by import API to create and configure the service template
	# so it depends on target panel type
	'settings',
))
