from parallels.common import messages
import logging

from parallels.common.actions.base.subscription_action import SubscriptionAction
from parallels.common import migrator_config
from parallels.common.actions.utils.multithreading_properties import MultithreadingProperties
from parallels.common.utils import migrator_utils
from parallels.common.utils import subscription_filter
from parallels.common import MigrationError
from parallels.common.utils.hosting_analyser_utils import apply_hosting_analyser_strategy

logger = logging.getLogger(__name__)


class CopyUnixWebContentBase(SubscriptionAction):
	"""Base class to copy web content for Unix servers"""

	def get_description(self):
		return messages.COPY_WEB_FILES_FROM_UNIX_SERVERS

	def get_failure_message(self, global_context, subscription):
		"""
		:type global_context: parallels.common.global_context.GlobalMigrationContext
		:type subscription: parallels.common.migrated_subscription.MigratedSubscription
		"""
		return messages.FAILED_COPY_WEB_FILES_FOR_SUBSCRIPTION_1 % subscription.name

	def is_critical(self):
		"""If action is critical or not

		If action is critical and it failed for a subscription, migration tool
		won't run the next operations for the subscription.

		:rtype: bool
		"""
		return False

	def get_multithreading_properties(self):
		"""
		:rtype: parallels.common.actions.utils.multithreading_properties.MultithreadingProperties
		"""
		return MultithreadingProperties(can_use_threads=True)

	def filter_subscription(self, global_context, subscription):
		"""
		:type global_context: parallels.common.global_context.GlobalMigrationContext
		:type subscription: parallels.common.migrated_subscription.MigratedSubscription
		"""
		return subscription_filter.unix_with_virtual_hosting(
			subscription
		)

	def run(self, global_context, subscription):
		"""
		:type global_context: parallels.common.global_context.GlobalMigrationContext
		:type subscription: parallels.common.migrated_subscription.MigratedSubscription
		"""
		source_server = subscription.web_source_server
		target_server = subscription.web_target_server

		user_mapping = {}
		group_mapping = {}

		if source_server.apache_user != target_server.apache_user:
			user_mapping[source_server.apache_user] = target_server.apache_user
		if source_server.apache_group != target_server.apache_group:
			group_mapping[source_server.apache_group] = target_server.apache_group

		tocopy = self._list_files_to_copy(global_context, subscription)
		rsync_additional_args = migrator_config.read_rsync_additional_args(global_context.config)
		apply_hosting_analyser_strategy(global_context, subscription, rsync_additional_args)

		key_info = global_context.ssh_key_pool.get(source_server, target_server)
		with \
			source_server.runner() as runner_source, \
			target_server.runner() as runner_target:
				try:
					for item in tocopy:
						migrator_utils.copy_directory_content_unix(
							source_server.ip(),
							source_server.user(),
							runner_source,
							runner_target,
							item.source_subdir,
							target_server.vhosts_dir + '/' + item.target_subdir,
							key_info.key_pathname,
							item.exclude,
							item.skip_if_source_not_exists,
							rsync_additional_args=rsync_additional_args,
							user_mapping=user_mapping,
							group_mapping=group_mapping,
							source_rsync_bin=source_server.rsync_bin
						)
				except Exception as e:
					logger.debug(u"Exception: ", exc_info=e)
					raise MigrationError((
						messages.RSYNC_FAILED_COPY_FILES_FROM_SOURCE_1) % (source_server.description(), target_server.description(), str(e)))

	def _list_files_to_copy(self, global_context, subscription):
		"""Make a list of source server directories to be transferred.

		Return a list of (source directory -> destination directory) mappings.
		Override in child classes.
		"""
		raise NotImplementedError()
