import logging
from xml.etree import ElementTree
from parallels.common.utils.migrator_utils import normalize_domain_name

from parallels.common.utils.pmm.pmmcli import PMMCLICommandUnix
from parallels.common.utils.pmm.pmmcli import PMMCLICommandWindows
from parallels.plesk_api.core import PleskError
from parallels.common.utils.pmm.restore_result import RestoreResult
from parallels.common.utils.windows_utils import cmd_command
from parallels.common.utils.steps_profiler import sleep

logger = logging.getLogger(__name__)

RESTORE_TASK_POLL_INTERVAL = 1
RESTORE_TASK_POLL_LIMIT = 1800


def restore_hosting_settings(
	target_conn, backup, backup_path, domains, safe,
	disable_apsmail_provisioning=None, target_backup_path='/tmp/plesk.backup'
):
	_restore_hosting_settings(
		target_conn, backup, backup_path, domains, safe,
		disable_apsmail_provisioning, u"hosting settings", target_backup_path=target_backup_path
	)


def restore_aps_application_settings(
	target_conn, backup, backup_path, domains, safe,
	disable_apsmail_provisioning=None,
	target_backup_path='/tmp/plesk.backup'
):
	_restore_hosting_settings(
		target_conn, backup, backup_path, domains, safe,
		disable_apsmail_provisioning, u"APS applications",
		extra_env=dict(PLESK_RESTORE_SITE_APPS_ONLY=u'true'),  # special environment variable for APS restoration
		target_backup_path=target_backup_path
	)


def _restore_hosting_settings(
	target_conn, backup, source_backup_path, domains, safe,
	disable_apsmail_provisioning, settings_description, extra_env=None,
	target_backup_path='/tmp/plesk.backup'
):
	"""Restore hosting settings of subscriptions

	disable_apsmail_provisioning - dictionary with
		keys - domain names,
		values - if we should not provision APS mail for that domain
		for example: {'a.tld': True, 'b.tld': False}
			- for 'a.tld' web hosting will be provisioned,
			but APS mail won't (mail settings will appear in the hosting panel,
			but nothing will happen on mail server;
			that is necessary for SmarterMail assimilation scenario)
			- for 'b.tld' both web and mail hosting will be provisioned

	disable_apsmail_provisioning could be None - that means that mail service will be provisioned for every domain
	"""

	if len(domains) == 0:
		logger.debug(u"Restore %s skipped as there are no subscriptions specified", settings_description)
		return

	# STEP #1: create utility objects
	server = target_conn.plesk_server

	if target_conn.is_windows:
		restore_hosting = RestoreHostingUtilsWindows(target_conn.plesk_server)
	else:
		restore_hosting = RestoreHostingUtilsUnix(target_conn.plesk_server)

	pmmcli_cmd = restore_hosting.create_pmmcli()

	# STEP #2: remove backup from target panel's repository if it already exists
	# If backup with such name already exists in target panel's repository
	# (for example, if migrator was interrupted before it cleans up backup file)
	# PMM will refuse to import it again
	# and we could get outdated backup, so remove backup file from target Plesk before import
	_remove_plesk_backup(target_conn, pmmcli_cmd, backup)

	# STEP #3: upload backup
	logger.debug(u"Upload backup dump '%s' to target node" % source_backup_path)
	with server.runner() as runner:
		runner.upload_file(source_backup_path, target_backup_path)

	# STEP #4: import backup to PPA repository
	logger.debug(u"Import backup dump to target panel's repository")
	backup_id = _import_backup(pmmcli_cmd, target_backup_path)

	# STEP #5: index imported XML domain backup files
	# Index backup files after backup is imported: create dict with key - domain name and value - backup XML file name.
	logger.debug(u"Create mapping of domain name to backup XML file name")
	domain_to_backup_file = restore_hosting.index_imported_domain_backup_files(backup_id)
	logger.debug("Mapping of backup XML files: %r", domain_to_backup_file)

	# STEP #6: restore each domain backup XML file
	# Only main subscription domains are required, their addon sites will be restored automatically.
	# Restore subscriptions one by one separately, to avoid them affecting others
	restore_domains_backup_xmls(
		disable_apsmail_provisioning, domain_to_backup_file,
		domains, extra_env, pmmcli_cmd,  restore_hosting,
		safe, server, settings_description
	)

	# STEP #7: remove backup file from target Plesk, so it does not bother customers
	# (they could see parts of the backup in their control panels if we don't remove it)
	_remove_plesk_backup(target_conn, pmmcli_cmd, backup)


def _import_backup(pmmcli_cmd, target_backup_path):
	import_result = pmmcli_cmd.import_dump(target_backup_path)
	if import_result.error_code not in {'0', '116'}:
		raise Exception(
			u"Failed to import backup XML: "
			u"errcode must be '0' (no errors) or '116' (backup sign error). "
			u"Output of pmmcli utility:\n%s" % (
				import_result.raw_stdout
			)
		)
	backup_id = import_result.backup_id
	if backup_id is None:
		raise Exception(
			u"Failed to import backup XML: can not find backup id. "
			u"Output of pmmcli utility:\n%s" % (
				import_result.raw_stdout
			)
		)
	logger.debug(u"Imported backup id is '%s'", backup_id)
	return backup_id


def restore_domains_backup_xmls(
	disable_apsmail_provisioning, domain_to_backup_file, domains, extra_env, pmmcli_cmd,
	restore_hosting, safe, server, settings_description
):
	for i, domain_name in enumerate(domains, 1):
		logger.info(
			u"Restore %s for subscription '%s' (#%d out of %d)", settings_description, domain_name, i, len(domains)
		)
		with safe.try_subscription(
			domain_name, u"Failed to restore %s of subscription with Plesk backup/restore." % settings_description
		):
			domain_name_normalized = normalize_domain_name(domain_name)
			if domain_name_normalized in domain_to_backup_file:
				full_env = restore_hosting.get_restore_env(
					disable_apsmail_provisioning, domain_name, extra_env
				)
				_restore_single_domain(
					domain_name, domain_to_backup_file[domain_name_normalized], full_env,
					pmmcli_cmd, safe, server, settings_description
				)
			else:
				safe.fail_subscription(domain_name, u"Failed to find imported backup XML file for subscription")


def _restore_single_domain(
	domain_name, domain_dump_xml, env, pmmcli_cmd,
	safe, server, settings_description
):
	restore_result = pmmcli_cmd.restore(domain_dump_xml, env)

	if restore_result.task_id is None:
		raise Exception(
			u'Restoration of subscription failed: no task ID was returned by pmmcli. '
			u'Check debug.log and PMM restoration logs for more details.'
			u"Output of pmmcli utility:\n%s" % (
				restore_result.raw_stdout
			)
		)

	# Workaround for race condition in PMM:
	# when we start restoration, it starts task, and in that task it starts
	# deployer. If deployer was never running, PMM will report "failed"
	# status for restoration task to handle situation with absent/broken
	# deployer binary. So if we request task status *before*
	# deployer was started, we get "failed" task status, which is not correct.
	# The most simple way to fix - add 2 second wait to ensure that deployer
	# was started before our first task status request. Considering that
	# restoration practically never takes less than 5 seconds, that should be ok.
	sleep(2, 'Initially wait for PMM deployer to start')

	logger.debug(
		u'Waiting for restore task to finish, '
		u'check interval: %s seconds, '
		u'maximum check attempts %s',
		RESTORE_TASK_POLL_INTERVAL, RESTORE_TASK_POLL_LIMIT
	)

	status = None

	for attempt in xrange(RESTORE_TASK_POLL_LIMIT):
		logger.debug(
			u'Poll Plesk for restoration task status, attempt #%s', attempt
		)
		status = pmmcli_cmd.get_task_status(restore_result.task_id)
		if status.is_running:
			sleep(RESTORE_TASK_POLL_INTERVAL, 'Waiting for Plesk restore task to finish')
		else:
			break

	assert status is not None  # loop executed at least once

	if status.is_running:  # if still running - raise exception
		raise Exception(
			u'Restoration of subscription failed: timed out '
			u'waiting for the restore task. '
			u'Check debug.log and PMM restoration logs for more details'
		)

	result_log = _download_result_xml(server, status.log_location)
	restore_result = RestoreResult(domain_name, result_log)
	succeeded = not restore_result.has_errors()
	if not succeeded:
		safe.fail_subscription(
			domain_name, u"An error occurred, when restoring {settings}:\n{text}".format(
				settings=settings_description,
				text=restore_result.get_error_messages(),
			),
			None, is_critical=False
		)


def _download_result_xml(server, log_location):
	"""Download restoration result XML contents."""
	if log_location is None:
		raise Exception(
			u'Restoration of subscription failed: no restoration '
			u'status log is available. '
			u'Check debug.log and PMM restoration logs for more'
			u'details'
		)
	logger.debug(u'Retrieving the restoration log "%s"', log_location)
	with server.runner() as runner:
		return runner.get_file_contents(log_location)


def _remove_plesk_backup(target_conn, pmmcli_cmd, backup):
	try:
		pmmcli_cmd.delete_dump(backup.backup_info_file)
	except PleskError:
		# silently skip if we failed to remove backup
		logger.debug(u'Failed to remove backup file from target Plesk, exception:', exc_info=True)


class RestoreHostingUtilsBase(object):
	"""Base class for OS-specific hosting restoration functions"""

	def __init__(self, server):
		self._server = server

	def create_pmmcli(self):
		"""Create PMMCLICommand object able to import backup, restore backup, etc"""
		raise NotImplementedError()

	def index_imported_domain_backup_files(self, backup_id):
		"""Create mapping {domain: backup file path} for backup imported to repository

		We find all XML files with corresponding backup_id and then check and
		if it is a domain backup XML, then we parse it and take domain name from <domain> node
		ATM it is impossible to reliably get file name where domain information is
		stored by domain name in case of long domain names, so we index files in such way.
		"""
		raise NotImplementedError()

	def get_restore_env(self, disable_apsmail_provisioning, domain_name, extra_env):
		"""Get environment variables necessary for restoration"""
		raise NotImplementedError()

	def _get_backup_xml_domain(self, filename):
		with self._server.runner() as runner:
			file_contents = runner.get_file_contents(filename)
		try:
			# the file_contents that we have here is in unicode() format
			# but parser (XMLParser, called from et.fromstring) needs the string in its original encoding
			# so decode it back from unicode to utf-8
			file_xml = ElementTree.fromstring(file_contents.encode('utf-8'))
			domain_node = file_xml.find('domain')
			if domain_node is not None:
				domain_name = normalize_domain_name(
					domain_node.attrib.get('name')
				)
				return domain_name
		except Exception:
			# Ignore all parse errors
			logger.debug(u"Exception while parsing XML (most probably could be ignored):", exc_info=True)
			return None

class RestoreHostingUtilsUnix(RestoreHostingUtilsBase):
	"""Unix-specific hosting restoration functions"""

	def create_pmmcli(self):
		"""Create PMMCLICommand object able to import backup, restore backup, etc"""
		return PMMCLICommandUnix(self._server)

	def index_imported_domain_backup_files(self, backup_id):
		"""Create mapping {domain: backup file path} for backup imported to repository

		See comment in base class for details.
		"""
		domain_to_backup_file = {}
		with self._server.runner() as runner:
			file_names = runner.run(
				'/usr/bin/find', [
					'-L',  # follow symlinks, useful if /var/lib/psa is a symlink to a directory on another partition
					self._server.dump_dir,
					'-type',
					'f',
					'-name',
					u'*_%s.xml' % (backup_id,)
				]
			).splitlines()
		for filename in file_names:
			if filename != '':
				domain_name = self._get_backup_xml_domain(filename)
				if domain_name is not None:
					domain_to_backup_file[domain_name] = filename

		return domain_to_backup_file

	def get_restore_env(self, disable_apsmail_provisioning, domain_name, extra_env):
		"""Get environment variables necessary for restoration

		Notes on environment variables:

		PLESK_DISABLE_APSMAIL_PROVISIONING must stay defined, as it indicates to
		PMM that it should work despite heterogeneous mode

		PLESK_DISABLE_PROVISIONING=false must stay defined for the same reasons
		when migrating to PPA 11.5 which does not have latest micro updates
		installed, and does not know about PLESK_DISABLE_APSMAIL_PROVISIONING (this
		could be cleaned up on the next major update when migrator requires new
		version of PPA)

		PLESK_RESTORE_DO_NOT_CHANGE_APACHE_RESTART_INTERVAL is set to 'true' as we
		control Apache restart interval ourselves. Reasons:

		- we need to set it once at the beginning of restore hosting step and
		revert it back at the end of the step, no need to pull it for each
		subscription
		- in case of critical failure (which sometimes happens) of PMM it
		leaves Apache restart interval equal to 999999, and there is no
		idea how to fix it now in a reasonable time
		- we need to leave an ability to customize the interval to customer

		PLESK_MIGRATION_MODE is used to specify PMM that database assimilation
		scenario is possible, and if database with such name already exists on the
		server PMM will try to just register it in Plesk (security is considered
		too - check PMM code for more details)
		"""
		if disable_apsmail_provisioning is None:
			disable_domain_apsmail = False
		elif domain_name in disable_apsmail_provisioning:
			disable_domain_apsmail = disable_apsmail_provisioning[domain_name]
		else:
			disable_domain_apsmail = False
		full_env = dict(
			PLESK_MIGRATION_MODE=u'1',
			PLESK_DISABLE_PROVISIONING=u'false',
			PLESK_DISABLE_APSMAIL_PROVISIONING=(u'true' if disable_domain_apsmail else u'false'),
			PLESK_RESTORE_DO_NOT_CHANGE_APACHE_RESTART_INTERVAL=u'true'
		)
		if extra_env:
			full_env.update(extra_env)
		return full_env


class RestoreHostingUtilsWindows(RestoreHostingUtilsBase):
	"""Windows-specific hosting restoration commands"""

	def create_pmmcli(self):
		"""Create PMMCLICommand object able to import backup, restore backup, etc"""
		return PMMCLICommandWindows(self._server)

	def index_imported_domain_backup_files(self, backup_id):
		"""Create mapping {domain: backup file path} for backup imported to repository

		See comment in base class for details.
		"""
		with self._server.runner() as runner:
			file_names = runner.sh(cmd_command(
				ur'dir "{backup_dir}\*info_{backup_id}.xml" /b /s'.format(
					backup_dir=self._server.dump_dir,
					backup_id=backup_id
				)
			)).split("\n")
		backup_xml_files = {}
		for filename in file_names:
			filename = filename.strip()
			if filename != '' and not '.discovered' in filename:
				domain_name = self._get_backup_xml_domain(filename)
				if domain_name is not None:
					backup_xml_files[domain_name] = filename
		return backup_xml_files

	def get_restore_env(self, disable_apsmail_provisioning, domain_name, extra_env):
		"""Get environment variables necessary for restoration"""
		if (
			disable_apsmail_provisioning is not None
			and
			any(disable_apsmail_provisioning.values())
		):
			raise NotImplementedError(
				"Disable APS mail provisioning flag is not implemented "
				"for Windows hosting settings restoration"
			)

		return extra_env
