from parallels.core.connections.connections import Connections
from parallels.core.utils.common import cached
from parallels.core.utils.common_constants import FTP_SOURCE_DEFAULT_SESSION_DIR
from parallels.core.utils.config_utils import ConfigSection
from parallels.plesk.hosting_description.config import HostingDescriptionConfig
from parallels.plesk.hosting_description.validate.mode import ValidationMode
from parallels.plesk.source.web.config import WebSourceConfig
from parallels.plesk.source.web.source import WebSource


class WebConnections(Connections):
    @cached
    def get_source_config_by_id(self, source_id):
        """Return configuration of source with given identifier

        :type source_id: str
        :rtype: parallels.plesk.source.confixx.settings.ConfixxConfig
        """
        config_section = ConfigSection(self._global_context.config, source_id)
        ftp_config_section = config_section.prefixed('ftp-')
        return WebSourceConfig(
            source_id,
            config_section,
            ftp_config_section.get('host'),
            ftp_config_section.get('username'),
            ftp_config_section.get('password'),
            config_section.get('session-dir', FTP_SOURCE_DEFAULT_SESSION_DIR),
            config_section.get('document-root'),
            config_section.get('base-url')
        )

    @cached
    def get_source_by_id(self, source_id):
        """Return web source with given identifier

        :type source_id: str
        :rtype: parallels.plesk.source.web.connections.WebSource
        """
        source_config = self.get_source_config_by_id(source_id)
        return WebSource(source_id, source_config, self._global_context.migrator_server)

    def iter_hosting_description_configs(self):
        """
        :rtype: collections.Iterable[parallels.plesk.hosting_description.config.HostingDescriptionConfig]
        """
        for section_name in self._get_source_config_section_names():
            section = ConfigSection(self._config, section_name)
            if 'hosting-description' in section:
                yield HostingDescriptionConfig(
                    source_id=section_name,
                    path=section.get('hosting-description'),
                    file_format=section.get('description-format', 'yaml').lower(),
                    validation_mode=section.get('validation-mode', ValidationMode.STOP).lower()
                )

    @staticmethod
    def iter_database_servers():
        return []

    def get_stats_server(self):
        """Get source panel IP address that will be saved in statistics report

        If we don't know it (for example in case of custom migration with absolutely no access
        to the source server) - return None

        :rtype: str | unicode | None
        """
        return None
