import logging

from parallels.core.utils.ftp import Ftp
from parallels.core.utils.paths.converters.unix.target import UnixTargetWebPathConverter
from parallels.core.utils.paths.converters.windows.target import WindowsTargetWebPathConverter
from parallels.core.utils.paths.copy_web_content import BaseWebPathConverter
from parallels.plesk.source.web import messages
from parallels.core.actions.base.subscription_action import SubscriptionAction
from parallels.core.actions.utils.multithreading_properties import MultithreadingProperties
from parallels.plesk.source.web.web_files import FtpPath

logger = logging.getLogger(__name__)


class CopyWebContentViaFtp(SubscriptionAction):
    """Copy web files for Unix servers"""

    def get_description(self):
        """
        :rtype: basestring
        """
        return messages.ACTION_COPY_WEB_CONTENT_VIA_FTP

    def get_failure_message(self, global_context, subscription):
        """
        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :type subscription: parallels.core.migrated_subscription.MigratedSubscription
        """
        return messages.ACTION_COPY_WEB_CONTENT_VIA_FTP_FAILED.format(source=subscription.model.source)

    def is_critical(self):
        """If action is critical or not

        If action is critical and it failed for a subscription, migration tool
        won't run the next operations for the subscription.

        :rtype: bool
        """
        return False

    def get_multithreading_properties(self):
        """Get how multithreading should be applied for that action

        :rtype: parallels.core.actions.utils.multithreading_properties.MultithreadingProperties
        """
        return MultithreadingProperties(can_use_threads=True, use_threads_by_default=True)

    def filter_subscription(self, global_context, subscription):
        """
        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :type subscription: parallels.core.migrated_subscription.MigratedSubscription
        """
        return global_context.migrator.web_files.need_to_copy_files(global_context, subscription)

    def run(self, global_context, subscription):
        """
        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :type subscription: parallels.core.migrated_subscription.MigratedSubscription
        """
        files_to_copy = global_context.migrator.web_files.list_files_to_copy(global_context, subscription)

        source = subscription.get_source_info().source
        ftp = Ftp(source.config.ftp_host, source.config.ftp_username, source.config.ftp_password)

        target_server = subscription.web_target_server
        for item in files_to_copy:
            source_path = FtpWebPathConverter().expand(item.source_path, None)
            if target_server.is_windows():
                target_path = WindowsTargetWebPathConverter().expand(item.target_path, target_server)
            else:
                target_path = UnixTargetWebPathConverter().expand(item.target_path, target_server)
            ftp.download_directory(source_path, target_server, target_path)


class FtpWebPathConverter(BaseWebPathConverter):
    """Class to convert abstract path descriptor to concrete absolute path on source ftp server"""

    def expand(self, path, server):
        """Convert abstract path descriptor to concrete absolute path for source ftp server

        :type path: parallels.core.utils.web_paths.WebHostingPath
        :rtype: str | unicode
        """
        if isinstance(path, FtpPath):
            return path.ftp_path
        else:
            assert False, messages.ACTION_COPY_WEB_CONTENT_VIA_FTP_UNSUPPORTED_SOURCE_WEB_PATH_TYPE
