import logging

from parallels.core.utils.database_utils import restore_db_from_dump_linux, restore_db_from_dump_windows
from parallels.plesk.source.web import messages
from parallels.core.actions.base.subscription_action import SubscriptionAction

logger = logging.getLogger(__name__)


class CopyDatabaseContent(SubscriptionAction):
    def get_description(self):
        return messages.ACTION_COPY_DATABASE_CONTENT

    def get_failure_message(self, global_context, subscription):
        """
        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :type subscription: parallels.core.migrated_subscription.MigratedSubscription
        """
        return messages.ACTION_COPY_DATABASE_CONTENT_FAILED

    def is_critical(self):
        """If action is critical or not

        If action is critical and it failed for a subscription, migration tool
        won't run the next operations for the subscription.

        :rtype: bool
        """
        return False

    def run(self, global_context, subscription):
        """
        :type global_context: parallels.plesk.source.web.global_context.WebGlobalMigrationContext
        :type subscription: parallels.core.migrated_subscription.MigratedSubscription
        """
        source = subscription.get_source_info().source

        for database in subscription.raw_dump.iter_all_databases():
            with source.runner() as source_runner:
                source_db_dump_path = source.get_session_file_path('%s_%s.sql' % (subscription.name, database.name))

                # create SQL dump of database on source
                exit_code, _, _ = source_runner.sh_unchecked(
                    u'{mysqldump_path} -u {user} -p"{password}" {database_name} '
                    u'--result-file={dump_filename}',
                    dict(
                        mysqldump_path=source.get_path_to_mysqldump(),
                        host=database.host,
                        port=database.port,
                        user=database.login,
                        password=database.password,
                        database_name=database.name,
                        dump_filename=source_db_dump_path
                    )
                )
                if exit_code != 0:
                    global_context.safe.fail_subscription(
                        subscription.name,
                        messages.ACTION_COPY_DATABASE_CONTENT_FAILED_DUMP.format(database_name=database.name),
                        is_critical=False
                    )
                    # skip this database and go to next
                    continue

                # download just created dump via FTP into subscription root directory on target
                session_db_dump_path = global_context.session_files.get_path_to_database_dump(database.name)
                source_runner.get_file(source_db_dump_path, session_db_dump_path)

                # retrieve target server where database dump will be restored
                target_db_server = subscription.db_target_servers.get('mysql')
                if target_db_server is None:
                    global_context.safe.fail_subscription(
                        subscription.name,
                        messages.ACTION_COPY_DATABASE_CONTENT_FAILED_NO_DB_SERVER.format(
                            database_name=database.name, database_type='mysql'
                        ),
                        is_critical=False
                    )
                    # skip this database and go to next
                    continue

                # restore database dump on target server
                with target_db_server.runner() as target_runner:
                    # copy database dump on target database server
                    target_db_dump_dir = target_db_server.panel_server.get_session_file_path('db-dumps')
                    target_db_dump_path = target_db_server.join_file_path(
                        target_db_dump_dir, '%s_%s.sql' % (subscription.name, database.name)
                    )
                    target_runner.mkdir(target_db_dump_dir)
                    target_runner.upload_file(session_db_dump_path, target_db_dump_path)

                    # restore dump
                    if target_db_server.utilities_server.is_windows():
                        restore_db_from_dump_windows(target_db_server, database.name, target_db_dump_path)
                    else:
                        restore_db_from_dump_linux(target_db_server, database.name, target_db_dump_path)
                    target_runner.remove_file(target_db_dump_path)
