import logging
import os
import shutil

from parallels.core import messages
from parallels.core.reports.model.issue import Issue
from parallels.core.utils.common import safe_format
from parallels.core.utils.pmm.agent import DumpSelected
from parallels.core.utils.pmm.pmmcli import PMMCLICommandWindows, PMMCLICommandUnix
from parallels.plesk.hosting_description.converter_to_dump import HostingDescriptionToPleskDumpConverter
from parallels.plesk.source.plesk.hosting_description.converter import PleskHostingDescriptionConverter
from parallels.plesk.source.plesk.hosting_description.provider.factory import create_plesk_hosting_description_provider

logger = logging.getLogger(__name__)


class HostingDescriptionAgent(object):
    def __init__(self, dump_agent):
        """Implements logic of creation of source server dump via creation of hosting description.

        :type dump_agent: parallels.core.utils.pmm.agent.PmmMigrationAgentBase
        """
        self._dump_agent = dump_agent

    def create_dump(self, local_dump_path, selection):
        """Create dump of source server

        :type local_dump_path: str | unicode
        :type selection: parallels.core.utils.pmm.agent.DumpAll | parallels.core.utils.pmm.agent.DumpSelected
        """
        hosting_description = self._create_hosting_description(selection)

        local_dump_filename, local_dump_extension = os.path.splitext(local_dump_path)
        hosting_description_dump_path = '%s.hosting-description%s' % (local_dump_filename, local_dump_extension)
        self._convert_hosting_description_to_dump(hosting_description, hosting_description_dump_path)

        remote_dump_path = '%s.dump%s' % (local_dump_filename, local_dump_extension)
        if not self._create_remote_dump_with_plesk_backup(selection, remote_dump_path):
            shutil.copy(hosting_description_dump_path, local_dump_path)
            return

        import_result = self._import_remote_dump(remote_dump_path)
        if import_result is None:
            shutil.copy(hosting_description_dump_path, local_dump_path)
            return

        merged_dump_path = '%s.merged%s' % (local_dump_filename, local_dump_extension)
        try:
            if not self._merge_dumps(merged_dump_path, hosting_description_dump_path, import_result.backup_prefix, import_result.backup_id):
                shutil.copy(hosting_description_dump_path, local_dump_path)
                return
        finally:
            self._remove_imported_dump(import_result.backup_xml_file_name)

        shutil.copy(merged_dump_path, local_dump_path)

    def _create_hosting_description(self, selection):
        """Create hosting description of source server

        :type selection: parallels.core.utils.pmm.agent.DumpAll | parallels.core.utils.pmm.agent.DumpSelected
        :rtype: dict
        """
        logger.info(messages.CREATE_HOSTING_DESCRIPTION)
        hosting_description_provider = create_plesk_hosting_description_provider(self._dump_agent, selection)
        return hosting_description_provider.get_description()

    def _convert_hosting_description_to_dump(self, hosting_description, hosting_description_dump_path):
        """Convert hosting description of source server to target Plesk dump format

        :type hosting_description: dict
        :type hosting_description_dump_path: str | unicode
        """
        logger.info(messages.CONVERT_HOSTING_DESCRIPTION.format(dump_path=hosting_description_dump_path))
        target_plesk_version = self._dump_agent.global_context.conn.target.plesk_server.get_plesk_version()
        dump_creator = HostingDescriptionToPleskDumpConverter(
            data=hosting_description, target_dump_file=hosting_description_dump_path,
            target_plesk_version=target_plesk_version, database_servers=[], is_windows=True,
            source_ip=self._dump_agent.source_server().ip()
        )
        dump_creator.write_dump()

    def _create_remote_dump_with_plesk_backup(self, selection, remote_dump_path):
        """Create the source server configuration dump using pleskbackup utility on the source server

        :type selection: parallels.core.utils.pmm.agent.DumpAll | parallels.core.utils.pmm.agent.DumpSelected
        :type remote_dump_path: str | unicode
        :rtype: bool
        """
        logger.info(messages.CREATE_REMOTE_CONFIGURATION_DUMP.format(dump_path=remote_dump_path))
        try:
            with self._dump_agent.source_server().runner() as runner:
                backup_path = self._make_backup(runner, selection)
                runner.get_file(backup_path, remote_dump_path)
            return True
        except Exception as e:
            logger.debug(messages.LOG_EXCEPTION, exc_info=True)
            self._dump_agent.global_context.pre_check_report.add_issue(
                'hosting_description_create_remote_dump',
                Issue.SEVERITY_WARNING, safe_format(messages.FAILED_TO_CREATE_REMOTE_CONFIGURATION_DUMP, error=unicode(e))
            )
            return False

    def _import_remote_dump(self, remote_dump_path):
        """Import the configuration dump created on the source server

        :type remote_dump_path: str | unicode
        :rtype: parallels.core.utils.pmm.pmmcli.ImportResult | None
        """
        logger.info(messages.IMPORT_REMOTE_CONFIGURATION_DUMP)
        try:
            if self._dump_agent.source_server().is_windows():
                pmmcli = PMMCLICommandWindows(self._dump_agent.global_context.conn.target.plesk_server)
            else:
                pmmcli = PMMCLICommandUnix(self._dump_agent.global_context.conn.target.plesk_server)
            result = pmmcli.import_dump(backup_path=remote_dump_path)
            if result.error_code not in {'0', '116', '152'}:
                self._dump_agent.global_context.pre_check_report.add_issue(
                    'hosting_description_import_remote_dump',
                    Issue.SEVERITY_WARNING,
                    safe_format(messages.FAILED_TO_IMPORT_REMOTE_CONFIGURATION_DUMP, error=result.error_message)
                )
                return None
            return result
        except Exception as e:
            logger.debug(messages.LOG_EXCEPTION, exc_info=True)
            self._dump_agent.global_context.pre_check_report.add_issue(
                'hosting_description_import_remote_dump',
                Issue.SEVERITY_WARNING, safe_format(messages.FAILED_TO_IMPORT_REMOTE_CONFIGURATION_DUMP, error=unicode(e))
            )
            return None

    def _remove_imported_dump(self, dump_name):
        """Remove the imported configuration dump

        :type dump_name: str | unicode
        """
        logger.info(safe_format(messages.REMOVE_IMPORTED_CONFIGURATION_DUMP, dump_name=dump_name))
        try:
            if self._dump_agent.source_server().is_windows():
                pmmcli = PMMCLICommandWindows(self._dump_agent.global_context.conn.target.plesk_server)
            else:
                pmmcli = PMMCLICommandUnix(self._dump_agent.global_context.conn.target.plesk_server)
            pmmcli.delete_dump(dump_name)
        except Exception as e:
            logger.debug(messages.LOG_EXCEPTION, exc_info=True)
            logger.warning(safe_format(messages.FAILED_TO_REMOVE_IMPORTED_CONFIGURATION_DUMP, dump_name=dump_name, error=unicode(e)))

    def _merge_dumps(self, merged_dump_path, hosting_description_dump_path, remote_dump_prefix, remote_dump_timestamp):
        """Merge configuration dumps

        :type merged_dump_path: str | unicode
        :type hosting_description_dump_path: str | unicode
        :type remote_dump_prefix: str | unicode
        :type remote_dump_timestamp: str | unicode
        :rtype: bool
        """
        logger.info(messages.MERGE_CONFIGURATION_DUMPS.format(dump_path=merged_dump_path))
        try:
            if os.path.exists(merged_dump_path):
                os.remove(merged_dump_path)
            converter = PleskHostingDescriptionConverter(
                result_dump_file=merged_dump_path, dump_file=hosting_description_dump_path,
                extra_dump_dir=self._dump_agent.global_context.conn.target.plesk_server.dump_dir,
                extra_dump_prefix=remote_dump_prefix, extra_dump_timestamp=remote_dump_timestamp,
            )
            schema_path = os.path.join(self._dump_agent.global_context.conn.target.plesk_server.plesk_dir, 'PMM', 'plesk.xsd')
            try:
                converter.load_schema(schema_path)
            except Exception as e:
                logger.debug(messages.LOG_EXCEPTION, exc_info=True)
                self._dump_agent.global_context.pre_check_report.add_issue(
                    'hosting_description_load_schema',
                    Issue.SEVERITY_WARNING, safe_format(messages.FAILED_TO_LOAD_PLESK_DUMP_SCHEMA, error=unicode(e))
                )
            converter.merge()
            return True
        except Exception as e:
            logger.debug(messages.LOG_EXCEPTION, exc_info=True)
            self._dump_agent.global_context.pre_check_report.add_issue(
                'hosting_description_merge_dumps',
                Issue.SEVERITY_WARNING, safe_format(messages.FAILED_TO_MERGE_CONFIGURATION_DUMPS, error=unicode(e))
            )
            return False

    def _make_backup(self, runner, selection):
        """
        :type runner: parallels.core.runners.base.BaseRunner
        :type selection: parallels.core.utils.pmm.agent.DumpAll|parallels.core.utils.pmm.agent.DumpSelected
        :rtype: str|unicode
        """
        command, options = self._get_create_backup_command(runner, selection)
        runner.remove_file(options['output_file_path'])
        runner.sh_unchecked(command, options)
        return options['output_file_path']

    def _get_create_backup_command(self, runner, selection):
        """
        :type runner: parallels.core.runners.base.BaseRunner
        :type selection: parallels.core.utils.pmm.agent.DumpAll|parallels.core.utils.pmm.agent.DumpSelected
        :rtype: tuple[str|unicode, dict[str|unicode, str|unicode]]
        """
        command = ur'{command} {backup_command} {configuration} {output_file} {output_file_path} {verbose} {polling_interval} {polling_interval_value}'
        if self._dump_agent.source_server().is_windows():
            options = {
                'command': r'%s\bin\pleskbackup.exe' % self._dump_agent.source_server().plesk_dir,
                'backup_command': '--domains-name',
                'configuration': '-configuration',
                'output_file': '-output-file',
                'output_file_path': self._dump_agent.source_server().get_session_file_path('backup.zip'),
                'verbose': '-verbose',
                'polling_interval': '-polling-interval',
                'polling_interval_value': '5',
            }
        else:
            options = {
                'command': r'%s/bin/pleskbackup' % self._dump_agent.source_server().plesk_dir,
                'backup_command': '--domains-name',
                'configuration': '--configuration',
                'output_file': '--output-file',
                'output_file_path': self._dump_agent.source_server().get_session_file_path('backup.tar'),
                'verbose': '-vvvvv',
                'polling_interval': '--polling-interval',
                'polling_interval_value': '5',
            }
        if isinstance(selection, DumpSelected):
            options['from_file'] = '-from-file' if self._dump_agent.source_server().is_windows() else '--from-file'
            options['from_file_path'] = self._create_filter_list_file(runner, selection)
            command += ur' {from_file} {from_file_path}'
        return command, options

    def _create_filter_list_file(self, runner, selection):
        """
        :type runner: parallels.core.runners.base.BaseRunner
        :type selection: parallels.core.utils.pmm.agent.DumpSelected
        :rtype: str|unicode
        """
        list_filename = self._dump_agent.source_server().get_session_file_path('filter.list')
        list_content = "\n".join(domain.encode('utf-8') for domain in selection.domains)
        runner.upload_file_content(list_filename, list_content)
        return list_filename
