<?xml version="1.0" encoding="utf-8" ?>
<!-- Copyright 1999-2012. Parallels IP Holdings GmbH. All Rights Reserved. -->
<xsl:stylesheet version="1.0" 
                xmlns:xsl="http://www.w3.org/1999/XSL/Transform">

  <xsl:output method="xml"/>

  <xsl:param name="exclude-server"/>

  <xsl:key name="db-server-key" match="//db_server" use="concat(concat(@host, ':'), @port)"/>
  <xsl:key name="system-ip-key" match="//ip_address" use="@ip_address"/>

  <xsl:template match="psadump">
    <migration-dump agent-name="pre9-backup-converter" dump-version="9.0">

      <xsl:call-template name="dump-info"/>

      <xsl:choose>
        
        <xsl:when test="$exclude-server">

          <xsl:for-each select="clients/client">
            <xsl:call-template name="client"/>
          </xsl:for-each>

          <xsl:for-each select="domains/domain">
            <xsl:call-template name="domain"/>
          </xsl:for-each>

        </xsl:when>
      
        <xsl:otherwise>

          <xsl:call-template name="admin"/>

          <xsl:for-each select="server">
            <xsl:call-template name="server"/>
          </xsl:for-each>

        </xsl:otherwise>
      
      </xsl:choose>

    </migration-dump>
  </xsl:template>

  <xsl:template name="content">
    <xsl:if test="count(*[(@arcname and (substring(@arcname,1,12)!='attachments ')) or @file])>0 or (@write_midify and @write_midify!='')">
      <content>
        <xsl:for-each select="*[(@arcname and (substring(@arcname,1,12)!='attachments ')) or @file]">
          <cid>

            <xsl:choose>
              <xsl:when test="name(..) = 'webapps'">
                <xsl:attribute name="app-name">
                  <xsl:value-of select="../@name"/>
                </xsl:attribute>
              </xsl:when>
              <xsl:when test="name(..)='database'">
                <xsl:attribute name="type">
                  <xsl:choose>
                    <xsl:when test="../@type='mysql' or ../@dumpType='newDump'">sqldump</xsl:when>
                    <xsl:otherwise>dbdump</xsl:otherwise>
                  </xsl:choose>
                </xsl:attribute>
              </xsl:when>
              <xsl:when test="name(..)='mail'">
                <xsl:attribute name="type">
                  <xsl:text>mailmsg</xsl:text>
                </xsl:attribute>
                <xsl:attribute name="referrer">
                  <xsl:text>true</xsl:text>
                </xsl:attribute>
                <xsl:attribute name="offset">
                  <xsl:value-of select="concat(../@mail_name,'@',../../@name)"/>
                </xsl:attribute>
                <xsl:attribute name="unpacksize">
                  <xsl:text>0</xsl:text>
                </xsl:attribute>
              </xsl:when>
            </xsl:choose>
            
            <content-file>
              <xsl:copy-of select="."/>
            </content-file>

          </cid>
        </xsl:for-each>

      </content>
    </xsl:if>
  </xsl:template>

  <xsl:template name="dump-info">
    <dump-info>
      
      <description>
        <xsl:value-of select="@description"/>
      </description>

      <os-description>
        <xsl:attribute name="type">windows</xsl:attribute>
        <xsl:attribute name="version">
          <xsl:value-of select="@OSVersion"/>
        </xsl:attribute>
      </os-description>

      <host-description>
        <xsl:attribute name="netbios-name">
          <xsl:value-of select="@MachineName"/>
        </xsl:attribute>
      </host-description>

      <cp-description>
        <xsl:attribute name="type">Plesk</xsl:attribute>
        <xsl:attribute name="version">
          <xsl:value-of select="@psaversion"/>
        </xsl:attribute>

        <xsl:for-each select="directories">
          <directories>
            <xsl:attribute name="vhosts">
              <xsl:value-of select="@vhosts_dir"/>
            </xsl:attribute>
            <xsl:attribute name="product_data">
              <xsl:value-of select="@product_data_dir"/>
            </xsl:attribute>
            <xsl:attribute name="product_root">
              <xsl:value-of select="@product_root_dir"/>
            </xsl:attribute>
          </directories>
        </xsl:for-each>
      </cp-description>
    </dump-info>
  </xsl:template>

  <xsl:template name="locale">
    <xsl:choose>
      <xsl:when test="string-length(.) = 0">
        <xsl:value-of select="'en-US'"/>
      </xsl:when>
      <xsl:when test="string-length(.) = 2">
        <xsl:choose>
          <xsl:when test=".='ar'">ar-SY</xsl:when>
          <xsl:when test=".='bg'">bg-BG</xsl:when>
          <xsl:when test=".='br'">pt-BR</xsl:when>
          <xsl:when test=".='ca'">ca-ES</xsl:when>
          <xsl:when test=".='cn'">zh-CN</xsl:when>
          <xsl:when test=".='de'">de-DE</xsl:when>
          <xsl:when test=".='en'">en-US</xsl:when>
          <xsl:when test=".='es'">es-ES</xsl:when>
          <xsl:when test=".='fi'">fi-FI</xsl:when>
          <xsl:when test=".='fr'">fr-FR</xsl:when>
          <xsl:when test=".='hk'">zh-HK</xsl:when>
          <xsl:when test=".='it'">it-IT</xsl:when>
          <xsl:when test=".='jp'">ja-JP</xsl:when>
          <xsl:when test=".='ko'">ko-KR</xsl:when>
          <xsl:when test=".='nl'">nl-NL</xsl:when>
          <xsl:when test=".='pl'">pl-PL</xsl:when>
          <xsl:when test=".='pt'">pt-PT</xsl:when>
          <xsl:when test=".='ru'">ru-RU</xsl:when>
          <xsl:when test=".='tr'">tr-TR</xsl:when>
          <xsl:when test=".='tw'">zh-TW</xsl:when>
          <xsl:otherwise>
            <xsl:message terminate="no">
              Unknown locale <xsl:value-of select="."/> in source dump.
            </xsl:message>
            <xsl:value-of select="'en-US'"/>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="."/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>
  
  <xsl:template name="admin">
    <admin>

      <xsl:if test="count(clients/client)>0">
        <clients>
          <xsl:for-each select="clients/client">
            <xsl:call-template name="client"/>
          </xsl:for-each>
        </clients>
      </xsl:if>

      <xsl:if test="count(domains/domain)>0">
        <domains>
          <xsl:for-each select="domains/domain">
            <xsl:call-template name="domain"/>
          </xsl:for-each>
        </domains>
      </xsl:if>

      <xsl:element name="preferences">
        <xsl:for-each select="server">
          <xsl:for-each select="params/misc[starts-with(@param, 'admin_')]">
            <xsl:variable name="name" select="substring-after(@param, 'admin_')"/>
            <xsl:if test="$name!='itmpl_id' and $name!='isPrefEdit'">
						  <xsl:choose>
							  <xsl:when test="$name='cname' or $name='pname' or $name='pcode' or $name='locale' or $name='email' or $name='phone' or $name='fax' or $name='address' or $name='city' or $name='state' or $name='country'">
								  <xsl:element name="pinfo">
									  <xsl:attribute name="name">
										  <xsl:choose>
											  <xsl:when test="$name='cname'">company</xsl:when>
											  <xsl:when test="$name='pname'">name</xsl:when>
											  <xsl:when test="$name='pcode'">zip</xsl:when>
											  <xsl:when test="$name='locale' or $name='email' or $name='phone' or $name='fax' or $name='address' or $name='city' or $name='state' or $name='country'">
												  <xsl:value-of select="$name"/>
											  </xsl:when>
										  </xsl:choose>
									  </xsl:attribute>

									  <xsl:choose>
										  <xsl:when test="$name='locale'">
											  <xsl:for-each select="@val">
												  <xsl:call-template name="locale"/>
											  </xsl:for-each>
										  </xsl:when>
										  <xsl:otherwise>
											  <xsl:value-of select="@val"/>
										  </xsl:otherwise>
									  </xsl:choose>
								  </xsl:element>
							  </xsl:when>
							  <xsl:otherwise>
                  <xsl:message>Unexpected name of pinfo: <xsl:value-of select="$name"/></xsl:message>
							  </xsl:otherwise>
						  </xsl:choose>
            </xsl:if>
          </xsl:for-each>
          
  	      <xsl:call-template name="shared-ssl-base">
      	    <xsl:with-param name="domainId" select="//domain[@shared_ssl_domain='true']/@id"/>
          </xsl:call-template>
          
        </xsl:for-each>
        
        <xsl:if test="count(scheduler) > 0">
          <xsl:element name="cron">
            <xsl:value-of select="scheduler/@arcname"/>
          </xsl:element>
        </xsl:if>
        
      </xsl:element>
    </admin>
  </xsl:template>

  <xsl:template name="server">
    <server>

      <properties>

        <xsl:for-each select="//ip_address[generate-id() = generate-id(key('system-ip-key', @ip_address)[1])]">
          <xsl:call-template name="system-ip"/>
        </xsl:for-each>

        <xsl:if test="params/misc[@param='FullHostName']">
          <hostname>
            <xsl:value-of select="params/misc[@param='FullHostName']/@val"/>
          </hostname>
        </xsl:if>

      </properties>

      <xsl:if test="count(//db_server)>0">
        <db-servers>
          <xsl:for-each select="//db_server[generate-id() = generate-id(key('db-server-key', concat(concat(@host, ':'), @port))[1])]">
            <xsl:call-template name="db-server"/>
          </xsl:for-each>
        </db-servers>
      </xsl:if>

      <xsl:for-each select="odbc/dsn[@configured = 'true']">
        <xsl:call-template name="dsn"/>
      </xsl:for-each>
      
      <xsl:call-template name="mail-settings"/>

      <xsl:call-template name="dns-settings"/>

      <xsl:if test="params/misc[@param = 'passive_ftp_port_range']/@val">
        <xsl:element name="ftp-settings">
          <xsl:attribute name="passive-mode-port-range">
            <xsl:value-of select="params/misc[@param = 'passive_ftp_port_range']/@val"/>
          </xsl:attribute>
        </xsl:element>
      </xsl:if>
      
      <xsl:if test="params/misc[@param = 'iis_app_pool_mode']/@val">
        <xsl:element name="iis-app-pool-global-settings">
          <xsl:attribute name="mode">
            <xsl:value-of select="params/misc[@param = 'iis_app_pool_mode']/@val"/>
          </xsl:attribute>
        </xsl:element>
      </xsl:if>

      <xsl:if test="count(params/misc[@param='cert_rep_id']/certificate)>0">
        <certificates>
          <xsl:for-each select="params/misc[@param='cert_rep_id']/certificate">
            <xsl:call-template name="certificate"/>
          </xsl:for-each>
        </certificates>
      </xsl:if>

      <interface-preferences>

        <xsl:for-each select="../custom_buttons/custom_button">
          <xsl:call-template name="custom-button"/>
        </xsl:for-each>

        <xsl:for-each select="skins/skin">
          <xsl:call-template name="skin"/>
        </xsl:for-each>

        <ui-mode>classic</ui-mode>
      </interface-preferences>

      <account-templates>

        <xsl:for-each select="templates/template[@type='domain_admin']">
          <xsl:call-template name="domain-template"/>
        </xsl:for-each>

        <xsl:for-each select="templates/template[@type='admin']">
          <xsl:call-template name="client-template"/>
        </xsl:for-each>

      </account-templates>

      <xsl:call-template name="applications"/>

      <xsl:call-template name="server-preferences"/>
      
      <xsl:for-each select="sa_confs/sa_conf[@mailname=''][1]">
        <xsl:call-template name="spamassassin"/>
      </xsl:for-each>

      <xsl:if test="count(params/misc[@param = 'server_wide_antivirus']) > 0">
        <xsl:variable name="serverWideAntivirus" select="params/misc[@param = 'server_wide_antivirus']/@val"/>
        <xsl:element name="virusfilter">
          <xsl:attribute name="state">
            <xsl:choose>
              <xsl:when test="$serverWideAntivirus = 'any'">
                <xsl:value-of select="'inout'"/>
              </xsl:when>
              <xsl:when test="$serverWideAntivirus = 'to'">
                <xsl:value-of select="'in'"/>
              </xsl:when>
              <xsl:when test="$serverWideAntivirus = 'from'">
                <xsl:value-of select="'out'"/>
              </xsl:when>
              <xsl:otherwise>
                <xsl:value-of select="'none'"/>
              </xsl:otherwise>
            </xsl:choose>
          </xsl:attribute>
          <xsl:if test="count(params/misc[@param = 'user_antivirus_incoming']) > 0 
                  or count(params/misc[@param = 'user_antivirus_outgoing']) > 0">
            <xsl:variable name="userIn" select="params/misc[@param = 'user_antivirus_incoming']/@val"/>
            <xsl:variable name="userOut" select="params/misc[@param = 'user_antivirus_outgoing']/@val"/>
            <xsl:attribute name="allow-personal">
              <xsl:choose>
                <xsl:when test="$userIn = 'true' and $userOut = 'true'">
                  <xsl:value-of select="'both'"/>
                </xsl:when>
                <xsl:when test="$userOut = 'true'">
                  <xsl:value-of select="'out'"/>
                </xsl:when>
                <xsl:when test="$userIn = 'true'">
                  <xsl:value-of select="'in'"/>
                </xsl:when>
                <xsl:otherwise>
                  <xsl:value-of select="'none'"/>
                </xsl:otherwise>
              </xsl:choose>
            </xsl:attribute>
          </xsl:if>
        </xsl:element>
      </xsl:if>

      <xsl:if test="count(params/misc[@param = 'actionlog_rot_type']) > 0">
        <xsl:element name="events">
          <xsl:variable name="rotationType" select="params/misc[@param = 'actionlog_rot_type']/@val"/>
          <xsl:element name="rotation">
            <xsl:choose>
              <xsl:when test="$rotationType = 'by_period'">
                <xsl:element name="by-time">
                  <xsl:attribute name="period">
                    <xsl:value-of select="params/misc[@param = 'actionlog_rot_num_periods']/@val"/>
                  </xsl:attribute>
                  <xsl:attribute name="period-type">
                    <xsl:value-of select="params/misc[@param = 'actionlog_rot_period']/@val"/>
                  </xsl:attribute>
                </xsl:element>
              </xsl:when>
              <xsl:when test="$rotationType = 'by_number'">
                <xsl:element name="by-entires">
                  <xsl:attribute name="count">
                    <xsl:value-of select="params/misc[@param = 'actionlog_rot_num_records']/@val"/>
                  </xsl:attribute>
                </xsl:element>
              </xsl:when>
              <xsl:otherwise>
                <xsl:element name="disabled"/>
              </xsl:otherwise>
            </xsl:choose>
          </xsl:element>
          <xsl:for-each select="actions/action">
            <xsl:element name="event">
              <xsl:attribute name="name">
                <xsl:value-of select="@name"/>
              </xsl:attribute>
              <xsl:attribute name="description">
                <xsl:value-of select="@descr"/>
              </xsl:attribute>
              <xsl:attribute name="enabled">
                <xsl:value-of select="@enabled"/>
              </xsl:attribute>
              <xsl:for-each select="event_handlers">
                <xsl:element name="handler">
                  <xsl:for-each select="@priority | @user |@command">
                    <xsl:copy/>
                  </xsl:for-each>
                </xsl:element>
              </xsl:for-each>
            </xsl:element>
          </xsl:for-each>
        </xsl:element>
      </xsl:if>
      
      <xsl:if test="count(params/misc[@param = 'exp_warn_time']) > 0">
        <xsl:element name="notifications">
          <xsl:attribute name="expiration-warning-days">
            <xsl:value-of select="params/misc[@param = 'exp_warn_time']/@val"/>
          </xsl:attribute>
          <xsl:for-each select="notes/note">
            <xsl:element name="notification">
              <xsl:attribute name="id">
                <xsl:value-of select="notifications/@id"/>
              </xsl:attribute>
              <xsl:attribute name="send2admin">
                <xsl:call-template name="intToBool">
                  <xsl:with-param name="value" select="notifications/@send2admin" />
                </xsl:call-template>
              </xsl:attribute>
              <xsl:attribute name="send2reseller">
                <xsl:value-of select="'false'"/>
              </xsl:attribute>
              <xsl:attribute name="send2client">
                <xsl:call-template name="intToBool">
                  <xsl:with-param name="value" select="notifications/@send2client" />
                </xsl:call-template>
              </xsl:attribute>
              <xsl:attribute name="send2email">
                <xsl:call-template name="intToBool">
                  <xsl:with-param name="value" select="notifications/@send2email" />
                </xsl:call-template>
              </xsl:attribute>
              <xsl:attribute name="email">
                  <xsl:value-of select="notifications/@email" />
              </xsl:attribute>
              <xsl:attribute name="subj">
                <xsl:value-of select="notifications/@subj" />
              </xsl:attribute>
              <xsl:element name="notice-text">
                <xsl:value-of select="@text"/>
              </xsl:element>
            </xsl:element>
          </xsl:for-each>
        </xsl:element>
      </xsl:if>
			
    </server>
  </xsl:template>

  <xsl:template name="intToBool">
    <xsl:param name="value"/>
    <xsl:choose>
      <xsl:when test="$value = '1'">
        <xsl:text>true</xsl:text>
      </xsl:when>
      <xsl:otherwise>
        <xsl:text>false</xsl:text>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <xsl:template name="strToBool">
    <xsl:param name="value"/>
    <xsl:choose>
      <xsl:when test="translate($value, 'TRUE', 'true') = 'true'">
        <xsl:text>true</xsl:text>
      </xsl:when>
      <xsl:otherwise>
        <xsl:text>false</xsl:text>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <xsl:template name="skin">
    <skin>
      <xsl:attribute name="name">
        <xsl:value-of select="@name"/>
      </xsl:attribute>
      <xsl:attribute name="place">
        <xsl:value-of select="@place"/>
      </xsl:attribute>

      <xsl:call-template name="content"/>
    </skin>
  </xsl:template>

  <xsl:template name="server-preferences">
    <server-preferences>

      <xsl:if test="params/misc[@param = 'forbid_create_dns_subzone']">
        <xsl:element name="forbid-create-dns-subzone">
          <xsl:call-template name="strToBool">
            <xsl:with-param name="value" select="params/misc[@param='forbid_create_dns_subzone']/@val"/>
          </xsl:call-template>
        </xsl:element>
      </xsl:if>

      <xsl:if test="params/misc[@param = 'force_db_user_prefix']">
        <xsl:element name="force-db-user-prefix">
          <xsl:call-template name="strToBool">
            <xsl:with-param name="value" select="params/misc[@param='force_db_user_prefix']/@val"/>
          </xsl:call-template>
        </xsl:element>
      </xsl:if>

      <xsl:if test="params/misc[@param='db_user_length']">
        <db-user-length>
          <xsl:value-of select="params/misc[@param='db_user_length']/@val"/>
        </db-user-length>
      </xsl:if>
      
      <xsl:if test="params/misc[@param='traffic_accounting']">
        <xsl:variable name="trafficDirection" select="params/misc[@param='traffic_accounting']/@val"/>
        <xsl:element name="traffic-direction">
          <xsl:attribute name="traffic">
            <xsl:choose>
              <xsl:when test="$trafficDirection = '1'">
                <xsl:value-of select="'in'"/>
              </xsl:when>
              <xsl:when test="$trafficDirection = '2'">
                <xsl:value-of select="'out'"/>
              </xsl:when>
              <xsl:otherwise>
                <xsl:value-of select="'both'"/>
              </xsl:otherwise>
            </xsl:choose>
          </xsl:attribute>
        </xsl:element>
      </xsl:if>
      
      <xsl:if test="params/misc[@param='stat_ttl']">
        <xsl:element name="stat-keep">
          <xsl:value-of select="params/misc[@param='stat_ttl']/@val"/>
        </xsl:element>
      </xsl:if>

      <xsl:element name="disk-usage">
        <xsl:if test="params/misc[@param = 'include_logs']/@val = 'true'">
          <xsl:element name="include-logs"/>
        </xsl:if>
        <xsl:if test="params/misc[@param = 'include_databases']/@val = 'true'">
          <xsl:element name="include-databases"/>
        </xsl:if>
        <xsl:if test="params/misc[@param = 'include_mailboxes']/@val = 'true'">
          <xsl:element name="include-mailboxes"/>
        </xsl:if>
        <xsl:if test="params/misc[@param = 'include_webapps']/@val = 'true'">
          <xsl:element name="include-webapps"/>
        </xsl:if>
        <xsl:if test="params/misc[@param = 'include_maillists']/@val = 'true'">
          <xsl:element name="include-maillists"/>
        </xsl:if>
        <xsl:if test="params/misc[@param = 'include_domaindumps']/@val = 'true'">
          <xsl:element name="include-domaindumps"/>
        </xsl:if>
        <xsl:if test="params/misc[@param = 'include_mysql_databases']/@val = 'true'">
          <xsl:element name="include-mysql-databases"/>
        </xsl:if>
        <xsl:if test="params/misc[@param = 'include_mssql_databases']/@val = 'true'">
          <xsl:element name="include-mssql-databases"/>
        </xsl:if>
      </xsl:element>
      
      <xsl:if test="params/misc[@param = 'force_db_prefix']">
        <xsl:element name="force-db-prefix">
          <xsl:call-template name="strToBool">
            <xsl:with-param name="value" select="params/misc[@param='force_db_prefix']/@val"/>
          </xsl:call-template>
        </xsl:element>
      </xsl:if>

      <!--
      <xsl:call-template name="shared-ssl-base">
        <xsl:with-param name="domainId" select="//domain[@shared_ssl_domain='true']/@id"/>
      </xsl:call-template>
      -->
      
      <xsl:if test="count(access/cp_access) > 0">
        <xsl:element name="admin-access-restrictions">
          <xsl:for-each select="access/cp_access">
            <xsl:element name="restriction-item">
              <xsl:attribute name="type">
                <xsl:value-of select="@type"/>
              </xsl:attribute>
              <xsl:attribute name="ip-address">
                <xsl:value-of select="@netaddr"/>
              </xsl:attribute>
              <xsl:attribute name="ip-subnet-mask">
                <xsl:value-of select="@netmask"/>
              </xsl:attribute>
            </xsl:element>
          </xsl:for-each>
        </xsl:element>
      </xsl:if>
      
    </server-preferences>
  </xsl:template>
  
  <xsl:template name="applications">
    <xsl:if test="count(siteapppackages/siteapppackage)>0">
      <application-vault>

        <sapp-packages-pool>
          <xsl:for-each select="siteapppackages/siteapppackage">
            <sapp-package>
              <sapp-name>
                <xsl:value-of select="@name"/>
              </sapp-name>
              <sapp-version>
                <xsl:value-of select="@version"/>
              </sapp-version>
              <sapp-release>
                <xsl:value-of select="@release"/>
              </sapp-release>

              <xsl:call-template name="content"/>
            </sapp-package>
          </xsl:for-each>
        </sapp-packages-pool>

        <sapp-items-pool>
          <xsl:for-each select="siteapppackages/siteapppackage/APSApplicationItems/APSApplicationItem">
            <xsl:call-template name="sapp-item"/>
          </xsl:for-each>
        </sapp-items-pool>

        <sapp-licenses-pool>
          <xsl:for-each select="siteapppackages/siteapppackage/APSLicenseType/APSLicense">
            <sapp-license>
              <key-number>
                <xsl:value-of select="@key_number"/>
              </key-number>
              <license-type>
                <xsl:value-of select="../@description"/>
              </license-type>
              <license-text>
              </license-text>
            </sapp-license>
          </xsl:for-each>
        </sapp-licenses-pool>

      </application-vault>
    </xsl:if>
  </xsl:template>

  <xsl:template name="sapp-item">
    <sapp-item>
      <xsl:attribute name="enabled">
        <xsl:choose>
          <xsl:when test="@disabled='true'">false</xsl:when>
          <xsl:otherwise>true</xsl:otherwise>
        </xsl:choose>
      </xsl:attribute>

      <sapp-spec>
        <sapp-name>
          <xsl:value-of select="../../@name"/>
        </sapp-name>
        <sapp-version>
          <xsl:value-of select="../../@version"/>
        </sapp-version>
        <sapp-release>
          <xsl:value-of select="../../@release"/>
        </sapp-release>
        <sapp-description>
          <xsl:value-of select="../../@description"/>
        </sapp-description>
        <xsl:if test="../../@access_level=1">
          <sapp-commercial/>
        </xsl:if>
        <xsl:if test="../../@integrated='true'">
          <sapp-integrated/>
        </xsl:if>
      </sapp-spec>

      <xsl:variable name="licenseTypeId" select="@license_type_id"/>
      <xsl:if test="$licenseTypeId and  $licenseTypeId!=0">
        <license-type>
          <xsl:value-of select="//APSLicenseType[@id=$licenseTypeId]/@description"/>
        </license-type>
      </xsl:if>

      <xsl:choose>
        <xsl:when test="@shared='true' or @shared='false'">
          <shared>
            <xsl:value-of select="@shared"/>
          </shared>
        </xsl:when>
      </xsl:choose>

      <description>
        <xsl:value-of select="@description"/>
      </description>

    </sapp-item>
  </xsl:template>

  <xsl:template name="dns-settings">
    <dns-settings>
      <xsl:attribute name="recursion">
        <xsl:choose>
          <xsl:when test="params/misc[@param='dns_recursion']">
            <xsl:choose>
              <xsl:when test="params/misc[@param='dns_recursion']/@val='on'">any</xsl:when>
              <xsl:when test="params/misc[@param='dns_recursion']/@val='local'">localnets</xsl:when>
              <xsl:when test="params/misc[@param='dns_recursion']/@val='off'">localhost</xsl:when>
              <xsl:otherwise>
                <xsl:message>Unexpected value on dns recursion: <xsl:value-of select="params/misc[@param='dns_recursion']/@val"/></xsl:message>
              </xsl:otherwise>
            </xsl:choose>
          </xsl:when>
          <xsl:otherwise>localnets</xsl:otherwise>
        </xsl:choose>
      </xsl:attribute>
      
      <dns-zone>
        <xsl:attribute name="email">
          <xsl:value-of select="params/misc[@param='admin_email']/@val"/>
        </xsl:attribute>
        <xsl:attribute name="type">
          <xsl:value-of select="'master'"/>
        </xsl:attribute>

        <status>
          <enabled/>
        </status>

        <xsl:for-each select="params/misc[starts-with(@param, 'soa_') and substring-before(@param, '_unit') = '' and @param!='soa_serial_format']">
					<xsl:variable name="name" select="./@param"/>
					<xsl:call-template name="dns-zone-param">
						<xsl:with-param name="name" select="substring-after(@param, 'soa_')"/>
						<xsl:with-param name="value" select="./@val"/>
						<xsl:with-param name="unitValue" select="number(../misc[@param = concat($name, '_unit')]/@val)"/>
					</xsl:call-template>
        </xsl:for-each>

        <xsl:for-each select="dns_templates/dns_template[@type!='AXFR']">
          <xsl:call-template name="dns-rec"/>
        </xsl:for-each>

      </dns-zone>

      <common-acl>
        <xsl:for-each select="dns_templates/dns_template[@type='AXFR']">
          <list-item>
            <xsl:value-of select="concat(concat(@val, '/'), @opt)"/>
          </list-item>
        </xsl:for-each>
      </common-acl>
    
    </dns-settings>
  </xsl:template>
  
  <xsl:template name="mail-settings">
    <mail-settings>
      
      <xsl:attribute name="use-vocabulary">
        <xsl:choose>
          <xsl:when test="params/misc[@param='use_vocabulary']">
            <xsl:value-of select="params/misc[@param='use_vocabulary']/@val"/>
          </xsl:when>
          <xsl:otherwise>false</xsl:otherwise>
        </xsl:choose>
      </xsl:attribute>
      
      <xsl:attribute name="relay">
        <xsl:choose>
          <xsl:when test="params/misc[@param='relay']">
            <xsl:value-of select="params/misc[@param='relay']/@val"/>
          </xsl:when>
          <xsl:otherwise>closed</xsl:otherwise>
        </xsl:choose>
      </xsl:attribute>
      
      <xsl:if test="params/misc[@param='disable_pop_auth']">
        <xsl:attribute name="pop-auth">
          <xsl:choose>
            <xsl:when test="params/misc[@param='disable_pop_auth']/@val = 1">false</xsl:when>
            <xsl:otherwise>true</xsl:otherwise>
          </xsl:choose>
        </xsl:attribute>
      </xsl:if>
      
      <xsl:if test="params/misc[@param='disable_smtp_auth']">
        <xsl:attribute name="smtp-auth">
          <xsl:choose>
            <xsl:when test="params/misc[@param='disable_smtp_auth']/@val = 1">false</xsl:when>
            <xsl:otherwise>true</xsl:otherwise>
          </xsl:choose>
        </xsl:attribute>
      </xsl:if>
      
      <xsl:if test="params/misc[@param='poplock_time']">
        <xsl:attribute name="poplock-time">
          <xsl:value-of select="params/misc[@param='poplock_time']/@val"/>
        </xsl:attribute>
      </xsl:if>
      
      <xsl:if test="mailsettings/@max_letter_size">
        <xsl:attribute name="max-letter-size">
          <xsl:value-of select="mailsettings/@max_letter_size"/>
        </xsl:attribute>
      </xsl:if>
      
      <xsl:if test="params/misc[@param='message_submission']">
        <xsl:attribute name="message-submission">
          <xsl:value-of select="params/misc[@param='message_submission']/@val"/>
        </xsl:attribute>
      </xsl:if>
      
      <xsl:if test="params/misc[@param='domain_keys_sign']">
        <xsl:attribute name="sign-outgoing-mail">
          <xsl:value-of select="params/misc[@param='domain_keys_sign']/@val"/>
        </xsl:attribute>
      </xsl:if>
      
      <xsl:if test="params/misc[@param='domain_keys_verify']">
        <xsl:attribute name="verify-incoming-mail">
          <xsl:value-of select="params/misc[@param='domain_keys_verify']/@val"/>
        </xsl:attribute>
      </xsl:if>

      <xsl:if test="params/misc[@param='rbl']">
        <rbl>
          <xsl:attribute name="status">
            <xsl:value-of select="params/misc[@param='rbl']/@val"/>
          </xsl:attribute>
          <rbl-server>
            <xsl:value-of select="params/misc[@param='rbl_server']/@val"/>
          </rbl-server>
        </rbl>
      </xsl:if>

      <black-list>
        <xsl:for-each select="badmail/badmailfrom">
          <list-item action="del">
            <xsl:value-of select="@domain"/>
          </list-item>
        </xsl:for-each>
        <xsl:for-each select="sa_confs/sa_conf/sa_list[@color='black']">
          <list-item action="mark">
            <xsl:value-of select="@pattern"/>
          </list-item>
        </xsl:for-each>
      </black-list>
      
      <white-list>
        <xsl:for-each select="sa_confs/sa_conf/sa_list[@color='white']">
          <list-item>
            <xsl:value-of select="@pattern"/>
          </list-item>
        </xsl:for-each>
        <xsl:for-each select="../smtp_poplocks">
          <xsl:element name="list-item">
            <xsl:value-of select="concat(@ip_address, '/', @ip_mask)"/>
          </xsl:element>
        </xsl:for-each>
      </white-list>

    </mail-settings>
  </xsl:template>
  
  <xsl:template name="db-server">
    <db-server>
      <xsl:attribute name="type">
        <xsl:value-of select="@type"/>
      </xsl:attribute>

      <host>
        <xsl:value-of select="@host"/>
      </host>

      <port>
        <xsl:value-of select="@port"/>
      </port>

      <db-admin>
        <xsl:attribute name="name">
          <xsl:value-of select="@admin_login"/>
        </xsl:attribute>

        <password type="plain">
          <xsl:value-of select="@admin_password"/>
        </password>

      </db-admin>

    </db-server>
  </xsl:template>
  
  <xsl:template name="system-ip">
    <system-ip>
      
      <xsl:attribute name="certificate">
        <xsl:value-of select="certificate/@name"/>
      </xsl:attribute>
      <xsl:if test="@ftps and @ftps!=''">
        <xsl:attribute name="ftps">
          <xsl:value-of select="@ftps"/>
        </xsl:attribute>
      </xsl:if>

      <xsl:call-template name="ip-address"/>

      <ip-netmask>
        <xsl:value-of select="@mask"/>
      </ip-netmask>
      
      <ip-interface>
        <xsl:value-of select="@iface"/>
      </ip-interface>
    
    </system-ip>
  </xsl:template>

  <xsl:template name="cr-date">
    <xsl:choose>
      <xsl:when test="contains(@cr_date, ':')">
        <xsl:choose>
          <xsl:when test="contains(@cr_date, '-') and string-length(@cr_date)>10">
            <xsl:value-of select="substring(@cr_date, 1, 10)"/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:message terminate="no">Unexpected date format: <xsl:value-of select="@cr_date"/></xsl:message>
            <xsl:text>0001-01-01</xsl:text>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:when>
      <xsl:otherwise>
        <xsl:choose>
          <xsl:when test="contains(@cr_date, '-') and string-length(@cr_date)=10">
            <xsl:value-of select="@cr_date"/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:message terminate="no">Unexpected date format: <xsl:value-of select="@cr_date"/></xsl:message>
            <xsl:text>0001-01-01</xsl:text>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>
  <xsl:template name="guid">
    <xsl:param name="guid"/>
    <xsl:param name="name"/>
    <xsl:param name="id"/>
    <xsl:choose>
      <xsl:when test="string-length($guid) = 36">
        <xsl:value-of select="concat($guid,$name,$id)"/>
      </xsl:when>
      <xsl:when test="(string-length($guid) = 38) and (substring($guid,1,1)='{') and (substring($guid,38,1)='}')">
        <xsl:value-of select="concat(substring($guid,2,36),$name,$id)"/>
      </xsl:when>
    </xsl:choose>
  </xsl:template>
  <xsl:template name="client">
    <client>
      <xsl:attribute name="id">
        <xsl:value-of select="@id"/>
      </xsl:attribute>
      <xsl:if test="@guid and @guid != '0000-0000-0000-0000'">
        <xsl:attribute name="guid">
          <xsl:call-template name="guid">
            <xsl:with-param name="guid" select="@guid"/>
          </xsl:call-template>
        </xsl:attribute>
      </xsl:if>
      <xsl:attribute name="name">
        <xsl:value-of select="@login"/>
      </xsl:attribute>
      <xsl:attribute name="contact">
        <xsl:value-of select="@pname"/>
      </xsl:attribute>
      <xsl:attribute name="cr-date">
        <xsl:call-template name="cr-date"/>
      </xsl:attribute>
      <xsl:if test="client_params/client_param[@param='max_button_length']/@val and client_params/client_param[@param='max_button_length']/@val>-1">
        <xsl:attribute name="max-button-length">
          <xsl:value-of select="client_params/client_param[@param='max_button_length']/@val"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:variable name="skinId" select="client_params/client_param[@param='skin_id']/@val"/>
      <xsl:if test="$skinId!='' and $skinId!=0">
        <xsl:attribute name="skin">
          <xsl:value-of select="//skin[@id=$skinId]/@name"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:attribute name="lock-screen">
        <xsl:choose>
          <xsl:when test="client_params/client_param[@param='disable_lock_screen']/@val='false'">false</xsl:when>
          <xsl:otherwise>true</xsl:otherwise>
        </xsl:choose>
      </xsl:attribute>

      <preferences>

        <pinfo name="country">
          <xsl:value-of select="@country"/>
        </pinfo>
        <pinfo name="phone">
          <xsl:value-of select="@phone"/>
        </pinfo>
        <pinfo name="fax">
          <xsl:value-of select="@fax"/>
        </pinfo>
        <pinfo name="state">
          <xsl:value-of select="@state"/>
        </pinfo>
        <pinfo name="email">
          <xsl:value-of select="@email"/>
        </pinfo>
        <pinfo name="city">
          <xsl:value-of select="@city"/>
        </pinfo>
        <pinfo name="zip">
          <xsl:value-of select="@pcode"/>
        </pinfo>
        <pinfo name="address">
          <xsl:value-of select="@address"/>
        </pinfo>
        <pinfo name="company">
          <xsl:value-of select="@cname"/>
        </pinfo>
        <pinfo name="locale">
          <xsl:for-each select="@locale">
            <xsl:call-template name="locale"/>
          </xsl:for-each>
        </pinfo>

        <xsl:for-each select="template">
          <xsl:call-template name="domain-template"/>
        </xsl:for-each>

        <xsl:for-each select="custom_buttons/custom_button">
          <xsl:call-template name="custom-button"/>
        </xsl:for-each>

        <xsl:for-each select="client_params/client_param[@param='shared_ssl_domain']">
          <xsl:call-template name="shared-ssl-base">
            <xsl:with-param name="domainId" select="@val"/>
          </xsl:call-template>
        </xsl:for-each>

			</preferences>

      <properties>
        <password type="plain">
          <xsl:value-of select="@passwd"/>
        </password>

        <xsl:call-template name="status"/>

        <xsl:call-template name="sb-reseller"/>

      </properties>

      <xsl:if test="count(limits/limit|permissions_list/permissions) > 0">
        <limits-and-permissions>
          <xsl:for-each select="limits/limit">
            <xsl:call-template name="client-limit"/>
          </xsl:for-each>
          <xsl:for-each select="permissions_list/permissions">
            <xsl:call-template name="permission"/>
          </xsl:for-each>
        </limits-and-permissions>
      </xsl:if>

      <xsl:if test="count(ip_pool/ip_address) > 0">
        <ip_pool>
          <xsl:for-each select="ip_pool/ip_address">
            <xsl:call-template name="ip-address"/>
          </xsl:for-each>
        </ip_pool>
      </xsl:if>

      <xsl:if test="count(APSClientApplicationItems/APSClientApplicationItem) > 0">
        <sapp-pool>
          <xsl:for-each select="APSClientApplicationItems/APSClientApplicationItem">
            <xsl:call-template name="sapp-client-item"/>
          </xsl:for-each>
        </sapp-pool>
      </xsl:if>

      <xsl:if test="count(domain) > 0">
        <domains>
          <xsl:for-each select="domain">
            <xsl:call-template name="domain"/>
          </xsl:for-each>
        </domains>
      </xsl:if>

    </client>
  </xsl:template>

  <xsl:template name="shared-ssl-base">
    <xsl:param name="domainId"/>
    <shared-ssl-base>
      <xsl:attribute name ="status">
        <xsl:choose>
          <xsl:when test="//domain[@id=$domainId]/@displayName">on</xsl:when>
          <xsl:otherwise>off</xsl:otherwise>
        </xsl:choose>
      </xsl:attribute>
      <xsl:if test="//domain[@id=$domainId]/@displayName">
        <xsl:attribute name ="domain">
          <xsl:value-of select="//domain[@id=$domainId]/@displayName"/>
        </xsl:attribute>
      </xsl:if>
    </shared-ssl-base>
  </xsl:template>

  <xsl:template name="sb-reseller">
    <xsl:if test="client_params/client_param[@param='sbnet_id']">
      <sb-reseller>
        <sb-client-login>
          <xsl:value-of select="@login"/>
        </sb-client-login>
        <sb-reseller-id>
          <xsl:value-of select="client_params/client_param[@param='sbnet_id']/@val"/>
        </sb-reseller-id>

        <xsl:for-each select="client_params/client_param[@param='sbnet_id']">
          <xsl:call-template name="content"/>
        </xsl:for-each>
      </sb-reseller>
    </xsl:if>
  </xsl:template>

  <xsl:template name="custom-button">
    <custom-button>
      <xsl:attribute name="url">
        <xsl:value-of select="@url"/>
      </xsl:attribute>
      <xsl:attribute name="text">
        <xsl:value-of select="@text"/>
      </xsl:attribute>
      <xsl:attribute name="sort-priority">
        <xsl:value-of select="@sort_key"/>
      </xsl:attribute>
      <xsl:attribute name="interface-place">
        <xsl:value-of select="@place"/>
      </xsl:attribute>
      <xsl:attribute name="conhelp">
        <xsl:value-of select="@conhelp"/>
      </xsl:attribute>
      <xsl:attribute name="open-in-same-frame">
        <xsl:choose>
          <xsl:when test="@options mod 256 - @options mod 128">true</xsl:when>
          <xsl:otherwise>false</xsl:otherwise>
        </xsl:choose>
      </xsl:attribute>
      <xsl:attribute name="visible-to-sublogins">
        <xsl:choose>
          <xsl:when test="@options mod 128 - @options mod 64">true</xsl:when>
          <xsl:otherwise>false</xsl:otherwise>
        </xsl:choose>
      </xsl:attribute>

      <xsl:call-template name="content"/>
    </custom-button>
  </xsl:template>

  <xsl:template name="domain-template">
    <domain-template>
      <xsl:attribute name="name">
        <xsl:value-of select="@name"/>
      </xsl:attribute>
      <xsl:for-each select="tmpldata">
        <xsl:call-template name="template-item"/>
      </xsl:for-each>
      <xsl:for-each select="tmpldata/log_rotation">
        <xsl:call-template name="log-rotation"/>
      </xsl:for-each>
    </domain-template>
  </xsl:template>

  <xsl:template name="client-template">
    <client-template>
      <xsl:attribute name="name">
        <xsl:value-of select="@name"/>
      </xsl:attribute>
      <xsl:for-each select="tmpldata">
        <xsl:call-template name="template-item"/>
      </xsl:for-each>
      <xsl:for-each select="tmpldata/log_rotation">
        <xsl:call-template name="log-rotation"/>
      </xsl:for-each>
    </client-template>
  </xsl:template>

  <xsl:template name="createTemplateItem">
    <xsl:param name="name"/>
    <xsl:param name="value"/>
    <xsl:element name="template-item">
      <xsl:attribute name="name">
        <xsl:value-of select="$name"/>
      </xsl:attribute>
      <xsl:value-of select="$value"/>
    </xsl:element>
  </xsl:template>
  
  <xsl:template name="template-item">
		<xsl:choose>
			<xsl:when test="@element = 'php_isapi' and @value = 'true'">
        <xsl:call-template name="createTemplateItem">
          <xsl:with-param name="name" select="'php_handler_type'"/>
          <xsl:with-param name="value" select="'isapi'"/>
        </xsl:call-template>
			</xsl:when>
			<xsl:when test="@element = 'make_dumps'">
        <xsl:call-template name="createTemplateItem">
          <xsl:with-param name="name" select="'allow_local_backups'"/>
          <xsl:with-param name="value" select="@value"/>
        </xsl:call-template>
        <xsl:call-template name="createTemplateItem">
          <xsl:with-param name="name" select="'allow_ftp_backups'"/>
          <xsl:with-param name="value" select="@value"/>
        </xsl:call-template>
			</xsl:when>
			<xsl:otherwise>
				<xsl:if test="@value and @value != '' and (
					@element = 'cp_access' or 
					@element = 'create_domains' or 
					@element = 'manage_phosting' or 
					@element = 'manage_php_safe_mode' or 
					@element = 'manage_quota' or 
					@element = 'change_limits' or 
					@element = 'manage_dns' or 
					@element = 'manage_log' or 
					@element = 'manage_crontab' or 
					@element = 'manage_anonftp' or 
					@element = 'manage_webapps' or 
					@element = 'manage_maillists' or 
					@element = 'manage_sh_access' or 
					@element = 'manage_not_chroot_shell' or 
					@element = 'manage_subdomains' or 
					@element = 'manage_virusfilter' or 
					@element = 'manage_drweb' or 
					@element = 'allow_local_backups' or 
					@element = 'allow_ftp_backups' or 
					@element = 'remote_access_interface' or 
					@element = 'manage_domain_aliases' or 
					@element = 'manage_spamfilter' or 
					@element = 'manage_dashboard' or 
					@element = 'dashboard' or 
					@element = 'stdgui' or 
					@element = 'manage_webstat' or 
					@element = 'manage_performance' or 
					@element = 'create_clients' or 
					@element = 'allow_oversell' or 
					@element = 'oversell' or 
					@element = 'overuse' or 
					@element = 'max_dom' or 
					@element = 'max_subdom' or 
					@element = 'max_dom_aliases' or 
					@element = 'disk_space' or 
					@element = 'disk_space_soft' or 
					@element = 'max_traffic' or 
					@element = 'max_traffic_soft' or 
					@element = 'max_wu' or 
					@element = 'max_db' or 
					@element = 'max_box' or 
					@element = 'mbox_quota' or 
					@element = 'max_redir' or 
					@element = 'max_mg' or 
					@element = 'max_resp' or 
					@element = 'max_maillists' or 
					@element = 'max_webapps' or 
					@element = 'expiration' or 
					@element = 'max_cl' or 
					@element = 'webmail' or 
					@element = 'keep_traf_stat' or 
					@element = 'maillist' or 
					@element = 'wuscripts' or 
					@element = 'no_usr' or 
					@element = 'log_rotate' or 
					@element = 'log_bysize' or 
					@element = 'log_bytime' or 
					@element = 'log_max_num' or 
					@element = 'log_compress' or 
					@element = 'log_email' or 
					@element = 'hosting' or 
					@element = 'quota' or 
					@element = 'ssl' or 
					@element = 'same_ssl' or 
					@element = 'fp' or 
					@element = 'fp_ssl' or 
					@element = 'fpauth' or 
					@element = 'asp' or 
					@element = 'ssi' or 
					@element = 'php' or 
					@element = 'cgi' or 
					@element = 'perl' or 
					@element = 'python' or 
					@element = 'fastcgi' or 
					@element = 'coldfusion' or 
					@element = 'webstat' or 
					@element = 'err_docs' or 
					@element = 'shell' or 
					@element = 'webstat_protdir' or 
					@element = 'dns_zone_type' or 
					@element = 'publish-sb-site' or 
					@element = 'bounce_mess' or 
					@element = 'dns_type' or 
					@element = 'errdocs' or 
					@element = 'fp_auth' or 
					@element = 'maillists' or 
					@element = 'nonexist_mail' or 
					@element = 'pdir_plesk_stat' or 
					@element = 'stat_ttl' or 
					@element = 'vh_type' or 
					@element = 'wu_script' or 
					@element = 'excl_ip_num' or 
					@element = 'catch_addr' or 
					@element = 'bandwidth' or 
					@element = 'catch_ip' or 
					@element = 'catch_mode' or 
					@element = 'cpu_usage' or 
					@element = 'iis_app_pool' or 
					@element = 'mssql_dbase_space' or 
					@element = 'mysql_dbase_space' or 
					@element = 'php_handler_type' or 
					@element = 'php_version' or 
					@element = 'sb_publish' or 
					@element = 'sb_subdomains' or 
					@element = 'use_sbnet' or 
					@element = 'total_mboxes_quota' or 
					@element = 'write_modify' or 
					@element = 'site_builder' or 
					@element = 'predefined_template' or 
					@element = 'max_cf_dsn' or 
					@element = 'max_connections' or 
					@element = 'max_fpse_users' or 
					@element = 'max_mssql_db' or 
					@element = 'max_odbc' or 
					@element = 'max_shared_ssl_links' or 
					@element = 'max_subftp_users' or 
					@element = 'max_iis_app_pools' or 
					@element = 'asp_dot_net' or 
					@element = 'managed_runtime_version' or 
					@element = 'miva' or 
					@element = 'ssi_html' or 
					@element = 'select_db_server' or 
					@element = 'manage_additional_permissions' or 
					@element = 'manage_iis_app_pool' or 
					@element = 'manage_shared_ssl' or 
					@element = 'manage_subftp')">
					<template-item>
						<xsl:attribute name="name">
							<xsl:choose>
								<xsl:when test="@element='use_sbnet'">publish-sb-site</xsl:when>
								<xsl:otherwise>
									<xsl:value-of select="@element"/>
								</xsl:otherwise>
							</xsl:choose>
						</xsl:attribute>
						<xsl:value-of select="@value"/>
					</template-item>
				</xsl:if>
			</xsl:otherwise>
		</xsl:choose>
  </xsl:template>

  <xsl:template name="log-rotation">
    <logrotation>
      <xsl:attribute name="max-number-of-logfiles">
        <xsl:choose>
          <xsl:when test="@max_number_of_logfiles=''">
            <xsl:text>0</xsl:text>
          </xsl:when>
          <xsl:otherwise>
            <xsl:value-of select="@max_number_of_logfiles"/>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:attribute>
      <xsl:attribute name="compress">
        <xsl:value-of select="@compress_enable"/>
      </xsl:attribute>
      <xsl:if test="@email">
        <xsl:attribute name="email">
          <xsl:value-of select="@email"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:attribute name="enabled">
        <xsl:value-of select="@turned_on"/>
      </xsl:attribute>

      <xsl:choose>
        <xsl:when test="@period_type='by_time'">
          <logrotation-period>
            <xsl:attribute name="period">
              <xsl:value-of select="@period"/>
            </xsl:attribute>
          </logrotation-period>
        </xsl:when>
        <xsl:when test="@period_type='by_size'">
          <logrotation-maxsize>
            <xsl:value-of select="@period"/>
          </logrotation-maxsize>
        </xsl:when>
        <xsl:otherwise>
          <xsl:message>Unexpected type of logrotation</xsl:message>
          <logrotation-period>
            <xsl:attribute name="period">
              <xsl:value-of select="@period"/>
            </xsl:attribute>
          </logrotation-period>
        </xsl:otherwise>
      </xsl:choose>
    </logrotation>
  </xsl:template>

  <xsl:template name="domain-limit">
    <xsl:choose>
      
      <xsl:when test="@limit_name='disk_space' or 
                  @limit_name='disk_space_soft' or 
                  @limit_name='max_traffic' or 
                  @limit_name='max_traffic_soft' or 
                  @limit_name='max_box' or 
                  @limit_name='mbox_quota' or 
                  @limit_name='total_mboxes_quota' or 
                  @limit_name='max_redir' or 
                  @limit_name='max_mg' or 
                  @limit_name='max_resp' or 
                  @limit_name='max_wu' or 
                  @limit_name='max_fpse_users' or 
                  @limit_name='max_subftp_users' or 
                  @limit_name='max_db' or 
                  @limit_name='max_mssql_db' or 
                  @limit_name='mysql_dbase_space' or 
                  @limit_name='mssql_dbase_space' or 
                  @limit_name='max_maillists' or 
                  @limit_name='max_webapps' or 
                  @limit_name='max_subdom' or 
                  @limit_name='max_dom_aliases' or 
                  @limit_name='max_shared_ssl_links' or 
                  @limit_name='expiration' or 
                  @limit_name='max_cf_dsn' or 
                  @limit_name='max_odbc'">

        <limit>

          <xsl:attribute name="name">
            <xsl:variable name="name" select="@limit_name"/>
            <xsl:choose>
              <xsl:when test="$name='total_mboxes_quota'">total_mbox_quota</xsl:when>
              <xsl:otherwise>
                <xsl:value-of select="$name"/>
              </xsl:otherwise>
            </xsl:choose>
          </xsl:attribute>

          <xsl:value-of select="@value"/>

        </limit>

      </xsl:when>

      <xsl:when test="@limit_name='max_dom' or
                      @limit_name='max_iis_app_pools'"/>

      <xsl:otherwise>
        <xsl:message terminate="no">Unexpected domain limit <xsl:value-of select="@limit_name"/>=<xsl:value-of select="@value"/></xsl:message>
      </xsl:otherwise>
    
    </xsl:choose>
  </xsl:template>

  <xsl:template name="client-limit">
    <xsl:choose>
      
      <xsl:when test="@limit_name='max_dom' or 
                  @limit_name='disk_space' or 
                  @limit_name='disk_space_soft' or 
                  @limit_name='max_traffic' or 
                  @limit_name='max_traffic_soft' or 
                  @limit_name='max_box' or 
                  @limit_name='mbox_quota' or 
                  @limit_name='total_mboxes_quota' or 
                  @limit_name='max_redir' or 
                  @limit_name='max_mg' or 
                  @limit_name='max_resp' or 
                  @limit_name='max_wu' or 
                  @limit_name='max_fpse_users' or 
                  @limit_name='max_subftp_users' or 
                  @limit_name='max_db' or 
                  @limit_name='max_mssql_db' or 
                  @limit_name='mysql_dbase_space' or 
                  @limit_name='mssql_dbase_space' or 
                  @limit_name='max_maillists' or 
                  @limit_name='max_webapps' or 
                  @limit_name='max_subdom' or 
                  @limit_name='max_dom_aliases' or 
                  @limit_name='max_iis_app_pools' or 
                  @limit_name='max_shared_ssl_links' or 
                  @limit_name='expiration' or 
                  @limit_name='max_cf_dsn' or 
                  @limit_name='max_odbc'">

        <limit>

          <xsl:attribute name="name">
            <xsl:variable name="name" select="@limit_name"/>
            <xsl:choose>
              <xsl:when test="$name='total_mboxes_quota'">total_mbox_quota</xsl:when>
              <xsl:otherwise>
                <xsl:value-of select="$name"/>
              </xsl:otherwise>
            </xsl:choose>
          </xsl:attribute>

          <xsl:value-of select="@value"/>

        </limit>

      </xsl:when>

      <xsl:otherwise>
        <xsl:message terminate="no">Unexpected client limit <xsl:value-of select="@limit_name"/>=<xsl:value-of select="@value"/></xsl:message>
      </xsl:otherwise>
      
    </xsl:choose>
  </xsl:template>

  <xsl:template name="createPermission">
    <xsl:param name="name"/>  
    <xsl:param name="value"/>
    <xsl:element name="permission">
      <xsl:attribute name="name">
        <xsl:value-of select="$name"/>
      </xsl:attribute>
      <xsl:attribute name="value">
        <xsl:value-of select="$value"/>
      </xsl:attribute>
    </xsl:element>
  </xsl:template>
  
  <xsl:template name="permission">
    <xsl:choose>
      <xsl:when test="@permission = 'manage_drweb'">
        <xsl:call-template name="createPermission">
          <xsl:with-param name="name" select="'manage_virusfilter'"/>
          <xsl:with-param name="value" select="@value"/>
        </xsl:call-template>
      </xsl:when>
      <xsl:when test="@permission = 'make_dumps'">
        <xsl:call-template name="createPermission">
          <xsl:with-param name="name" select="'allow_local_backups'"/>
          <xsl:with-param name="value" select="@value"/>
        </xsl:call-template>
        <xsl:call-template name="createPermission">
          <xsl:with-param name="name" select="'allow_ftp_backups'"/>
          <xsl:with-param name="value" select="@value"/>
        </xsl:call-template>
      </xsl:when>
      <xsl:otherwise>
        <xsl:call-template name="createPermission">
          <xsl:with-param name="name" select="@permission"/>
          <xsl:with-param name="value" select="@value"/>
        </xsl:call-template>
      </xsl:otherwise>
    </xsl:choose>
    <!--<xsl:element name="permission">
      <xsl:attribute name="name">
        <xsl:choose>
          <xsl:when test="@permission='manage_drweb'">manage_virusfilter</xsl:when>
          <xsl:otherwise>
            <xsl:value-of select="@permission"/>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:attribute>
      <xsl:attribute name="value">
        <xsl:value-of select="@value"/>
      </xsl:attribute>
    </xsl:element>-->
  </xsl:template>

  <xsl:template name="mailuser-permission">
    <xsl:element name="mailuser-permission">
      <xsl:attribute name="name">
        <xsl:choose>
          <xsl:when test="@permission='manage_spamfilter'">manage-spamfilter</xsl:when>
          <xsl:when test="@permission='manage_virusfilter' or @permission='manage_drweb'">manage-virusfilter</xsl:when>
          <xsl:when test="@permission='cp_access'">cp-access</xsl:when>
          <xsl:otherwise>
            <xsl:value-of select="@permission"/>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:attribute>
      <xsl:attribute name="value">
        <xsl:value-of select="@value"/>
      </xsl:attribute>
    </xsl:element>
  </xsl:template>

  <xsl:template name="ip-address">
    <xsl:call-template name="createIpElement">
      <xsl:with-param name="ipType" select="@type"/>
      <xsl:with-param name="ipAddress" select="@ip_address"/>
    </xsl:call-template>
  </xsl:template>

  <xsl:template name="createIpElement">
    <xsl:param name="ipType"/>
    <xsl:param name="ipAddress"/>
      <xsl:element name="ip">
        <xsl:element name="ip-type">
          <xsl:value-of select="$ipType"/>
        </xsl:element>
        <xsl:element name="ip-address">
          <xsl:value-of select="$ipAddress"/>
        </xsl:element>
      </xsl:element>
  </xsl:template>
  
  <xsl:template name="sapp-client-item">
    <sapp-item>

      <xsl:call-template name="sapp-spec"/>

      <xsl:variable name="itemId" select="@app_item_id"/>
      <xsl:variable name="licenseTypeId" select="//APSApplicationItem[@id=$itemId]/@license_type_id"/>

      <description>
        <xsl:value-of select="//APSApplicationItem[@id=$itemId]/@description"/>
      </description>

      <xsl:variable name="shared" select="//APSApplicationItem[@id=$itemId]/@shared"/>
      <xsl:if test="$shared='false'">
        <instances-limit>
          <xsl:value-of select="@instances_limit"/>
        </instances-limit>
      </xsl:if>

    </sapp-item>
  </xsl:template>

  <xsl:template name="sapp-spec">
    <xsl:variable name="package-name" select="@package-name"/>
    <xsl:variable name="package-version" select="@package-version"/>
    <xsl:variable name="package-release" select="@package-release"/>

    <sapp-spec>
      <sapp-name>
        <xsl:value-of select="$package-name"/>
      </sapp-name>
      <sapp-version>
        <xsl:value-of select="$package-version"/>
      </sapp-version>
      <sapp-release>
        <xsl:value-of select="$package-release"/>
      </sapp-release>
      <sapp-description>
        <xsl:value-of select="//siteapppackage[@name=$package-name and @version=$package-version and @release=$package-release]/@description"/>
      </sapp-description>
      <xsl:if test="//siteapppackage[@name=$package-name and @version=$package-version and @release=$package-release]/@access_level=1">
        <sapp-commercial/>
      </xsl:if>
      <xsl:if test="//siteapppackage[@name=$package-name and @version=$package-version and @release=$package-release]/@integrated='true'">
        <sapp-integrated/>
      </xsl:if>
    </sapp-spec>
  </xsl:template>

  <xsl:template name = "createCatchAllElement">
    <xsl:param name="value"/>
    <xsl:element name="catch-all">
      <xsl:value-of select="$value"/>
    </xsl:element>
  </xsl:template>

  <xsl:template name="domainServiceStatus">
    <status>
      <xsl:choose>
        <xsl:when test="@status=0 or (@status=2 and ../@status = 0)">
          <enabled/>
        </xsl:when>
        <xsl:when test="@status=2 and (../@status != 0)">
          <disabled-by name="parent"/>
        </xsl:when>
        <xsl:otherwise>
          <disabled-by name="admin"/>
        </xsl:otherwise>
      </xsl:choose>
    </status>
  </xsl:template>

  <xsl:template name="domain">
    <domain>
      <xsl:attribute name="id">
        <xsl:value-of select="@id"/>
      </xsl:attribute>
      <xsl:if test="@guid and @guid != '0000-0000-0000-0000'">
        <xsl:attribute name="guid">
          <xsl:call-template name="guid">
            <xsl:with-param name="guid" select="@guid"/>
          </xsl:call-template>
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="../@guid">
        <xsl:attribute name="owner-guid">
          <xsl:call-template name="guid">
            <xsl:with-param name="guid" select="../@guid"/>
          </xsl:call-template>
        </xsl:attribute>
      </xsl:if>
      <xsl:attribute name="name">
        <xsl:value-of select="@displayName"/>
      </xsl:attribute>
      <xsl:if test="dom_params/dom_param[@param='www_prefix']">
        <xsl:attribute name="www">
          <xsl:value-of select="dom_params/dom_param[@param='www_prefix']/@val"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:attribute name="cr-date">
        <xsl:call-template name="cr-date"/>
      </xsl:attribute>
        
        <preferences>
          <xsl:for-each select="domain_aliases/domain_alias">
            <xsl:call-template name="domain-alias"/>
          </xsl:for-each>
          <xsl:for-each select="custom_buttons/custom_button">
            <xsl:call-template name="custom-button"/>
          </xsl:for-each>
        </preferences>
        
        <properties>

          <xsl:for-each select="hosting/ip_address">
            <xsl:call-template name="ip-address"/>
          </xsl:for-each>
          <xsl:for-each select="forwarding/ip_address">
            <xsl:call-template name="ip-address"/>
          </xsl:for-each>
          
          <xsl:if test="count(hosting|forwarding) = 0">
            <xsl:variable name="domainName" select="@name"/>
            <xsl:variable name="ipFromDns" select="dns_zone/descendant::dns_rec[@type = 'A' and @host = concat($domainName, '.')]/@val"/>
            <xsl:choose>
              <xsl:when test="//ip_address[@ip-address = $ipFromDns]">
                <xsl:for-each select="//ip_address[@ip-address = $ipFromDns][1]">
                  <xsl:call-template name="ip-address"/>
                </xsl:for-each>
              </xsl:when>
              <xsl:when test="../ip_pool/ip_address[@type = 'shared']">
                <xsl:for-each select="../ip_pool/ip_address[@type = 'shared'][1]">
                  <xsl:call-template name="ip-address"/>
                </xsl:for-each>
              </xsl:when>
              <xsl:when test="../ip_pool/ip_address">
                <xsl:for-each select="../ip_pool/ip_address[1]">
                  <xsl:call-template name="ip-address"/>
                </xsl:for-each>
              </xsl:when>
              <xsl:otherwise>
                <xsl:call-template name="createIpElement">
                  <xsl:with-param name="ipType" select="'shared'"/>
                  <xsl:with-param name="ipAddress" select="$ipFromDns"/>
                </xsl:call-template>
              </xsl:otherwise>
            </xsl:choose>
          </xsl:if>

          <xsl:call-template name="status"/>

          <xsl:for-each select="dns_zone">
            <xsl:call-template name="dns-zone"/>
          </xsl:for-each>

        </properties>

        <xsl:if test="count(limits/limit) > 0">
          <limits-and-permissions>
            <xsl:for-each select="limits/limit">
              <xsl:call-template name="domain-limit"/>
            </xsl:for-each>
          </limits-and-permissions>
        </xsl:if>

        <mailsystem>
          <xsl:if test="count(.//dump[@type = 'directory' and (starts-with(@arcname, 'attachments ') or starts-with(@arcname, 'mail '))]) > 0 or count(.//mail[@contentType = 'independent'])">
            <xsl:element name="content">
              <xsl:if test="count(.//dump[@type = 'directory' and (starts-with(@arcname, 'mail ') or starts-with(@arcname, 'MailContent\'))]) > 0">
                <xsl:variable name="cid-type">
                  <xsl:choose>
                    <xsl:when test="mail[1]/@contentType='independent'">mailmsg</xsl:when>
                    <xsl:otherwise>mailnative</xsl:otherwise>
                  </xsl:choose>
                </xsl:variable>
                <xsl:element name="cid">
                  <xsl:attribute name="type">
                    <xsl:value-of select="$cid-type"/>
                  </xsl:attribute>
                  <xsl:element name="content-file">
                    <xsl:for-each select=".//dump[@type = 'directory' and (starts-with(@arcname, 'mail ') or starts-with(@arcname, 'MailContent\'))]">
                      <xsl:copy-of select="."/>
                    </xsl:for-each>
                  </xsl:element>
                </xsl:element>
              </xsl:if>
              <xsl:if test="count(.//dump[@type = 'directory' and starts-with(@arcname, 'attachments ')]) > 0">
                <xsl:element name="cid">
                  <xsl:attribute name="type">
                    <xsl:text>attaches</xsl:text>
                  </xsl:attribute>
                  <xsl:element name="content-file">
                    <xsl:for-each select=".//dump[@type = 'directory' and starts-with(@arcname, 'attachments ')]">
                      <dump type="directory" arcname="{concat('attachments ', ../@mail_name, '@',../../@name)}" />
                    </xsl:for-each>
                  </xsl:element>
                </xsl:element>
              </xsl:if>
            </xsl:element>
          </xsl:if>
          <properties>
            <xsl:choose>
              <xsl:when test="count(domainservices[@type='mail'])>0">
                <xsl:for-each select="domainservices[@type='mail']">
                  <xsl:call-template name="domainServiceStatus"/>
                </xsl:for-each>
              </xsl:when>
              <xsl:otherwise>
                <status>
                  <disabled-by name="admin"/>
                </status>
              </xsl:otherwise>
            </xsl:choose>
          </properties>

          <xsl:if test="count(mail) > 0">
            <mailusers>
              <xsl:for-each select="mail">
                <xsl:call-template name="mail"/>
              </xsl:for-each>
            </mailusers>
          </xsl:if>

          <xsl:if test="count(domainservices[@type='mail']) > 0">
            <xsl:variable name="parameters" select="domainservices[@type='mail']/parameters/parameter"/>            
            <xsl:element name="preferences">
              <xsl:if test="count($parameters[@parameter = 'nonexist_mail']) > 0">
                <xsl:variable name="nonexistMail" select="$parameters[@parameter = 'nonexist_mail']/@value"/>
                <xsl:choose>
                  <xsl:when test="$nonexistMail = 'catch'">
                    <xsl:variable name="catchMode" select="$parameters[@parameter = 'catch_mode']/@value"/>
                    <xsl:if test="$catchMode = 'catch_addr' or $catchMode = 'catch_ip'">
                      <xsl:call-template name="createCatchAllElement">
                        <xsl:with-param name="value" select="$parameters[@parameter = $catchMode]/@value"/>
                      </xsl:call-template>
                    </xsl:if>
                  </xsl:when>
                  <xsl:when test="$nonexistMail = 'reject' or $nonexistMail = 'discard'">
                    <xsl:call-template name="createCatchAllElement">
                      <xsl:with-param name="value" select="$nonexistMail"/>
                    </xsl:call-template>
                  </xsl:when>
                </xsl:choose>
              </xsl:if>
              <xsl:variable name="domainkeys" select="$parameters[@parameter = 'domain_keys_sign']/@value"/>
			  <xsl:variable name="bigparameters" select="domainservices[@type='mail']/bigparameters/bigparameter"/>
              <xsl:if test="$domainkeys = 'true'">
                <xsl:element name="domain-keys">
                  <xsl:attribute name="state">
                    <xsl:value-of select="$domainkeys"/>
                  </xsl:attribute>
                  <xsl:attribute name="private-key">
                    <xsl:value-of select="$bigparameters[@parameter = 'domain_key']/@value"/>
                  </xsl:attribute>
                  <xsl:attribute name="public-key">
                    <xsl:value-of select="$bigparameters[@parameter = 'domain_key_public']/@value"/>
                  </xsl:attribute>                  
                </xsl:element>
              </xsl:if>
              <xsl:variable name="webmail" select="$parameters[@parameter = 'webmail_type']/@value"/>
              <xsl:element name="web-mail">
                <xsl:choose>
                  <xsl:when test="$webmail != ''">
                    <xsl:value-of select="$webmail"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <xsl:text>none</xsl:text>
                  </xsl:otherwise>
                </xsl:choose>
              </xsl:element>
            </xsl:element>
          </xsl:if>
          
        </mailsystem>

        <xsl:if test="count(databases/database/@id[not(self::node()[.=//siteappresource[@type='database']/@res_id])]) > 0">
          <databases>
            <xsl:for-each select="databases/database">
              <xsl:call-template name="database"/>
            </xsl:for-each>
          </databases>
        </xsl:if>

        <xsl:for-each select="dsn[@configured='true']">
          <xsl:call-template name="dsn"/>
        </xsl:for-each>

        <xsl:if test="count(domainservices[@type='maillists']) > 0">
          <maillists>
            <xsl:for-each select="domainservices[@type='maillists']">
              <xsl:call-template name="domainServiceStatus"/>
            </xsl:for-each>
            <xsl:for-each select="maillists">
              <xsl:call-template name="maillist"/>
            </xsl:for-each>
          </maillists>
        </xsl:if>

        <xsl:if test="count(domainstraffic) > 0">
          <traffic>
            <xsl:for-each select="domainstraffic">
              <xsl:for-each select="@*[contains(name(), '_in') or contains(name(), '_out')]">
                <xsl:call-template name="traffic"/>
              </xsl:for-each>
            </xsl:for-each>
          </traffic>
        </xsl:if>

        <xsl:if test="count(certificate) > 0">
          <certificates>
            <xsl:for-each select="certificate">
              <xsl:call-template name="certificate"/>
            </xsl:for-each>
          </certificates>
        </xsl:if>

        <xsl:if test="count(domainservices[@type='tomcat']) > 0">
          <tomcat>
            <xsl:for-each select="domainservices[@type='tomcat']">
              <xsl:call-template name="domainServiceStatus"/>
              <xsl:for-each select="webapps">
                <webapp>
                  <xsl:attribute name="name">
                    <xsl:value-of select="@name"/>
                  </xsl:attribute>

                  <xsl:call-template name="status"/>

                  <xsl:call-template name="content"/>
                </webapp>
              </xsl:for-each>
            </xsl:for-each>
          </tomcat>
        </xsl:if>

        <xsl:if test="count(dom_level_usrs) > 0">
          <xsl:for-each select="dom_level_usrs">
            <xsl:call-template name="domainuser"/>
          </xsl:for-each>
        </xsl:if>

        <xsl:choose>
          <xsl:when test="@htype='vrt_hst'">
            <xsl:for-each select="hosting">
              <xsl:call-template name="phosting"/>
            </xsl:for-each>
          </xsl:when>
          <xsl:when test="@htype='std_fwd'">
            <xsl:for-each select="forwarding">
              <xsl:call-template name="shosting"/>
            </xsl:for-each>
          </xsl:when>
          <xsl:when test="@htype='frm_fwd'">
            <xsl:for-each select="forwarding">
              <xsl:call-template name="fhosting"/>
            </xsl:for-each>
          </xsl:when>
        </xsl:choose>

    </domain>
  </xsl:template>

  <xsl:template name="phosting">
    <phosting>
      <xsl:if test="../@guid and ../@guid != '0000-0000-0000-0000'">
        <xsl:attribute name="guid">
          <xsl:call-template name="guid">
            <xsl:with-param name="guid" select="../@guid"/>
            <xsl:with-param name="name" select="'_ph'"/>
          </xsl:call-template>
        </xsl:attribute>
        <xsl:attribute name="owner-guid">
          <xsl:call-template name="guid">
            <xsl:with-param name="guid" select="../@guid"/>
          </xsl:call-template>
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="@ssl and @ssl!=''">
        <xsl:attribute name="https">
          <xsl:value-of select="@ssl"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="@fp and @fp!=''">
        <xsl:attribute name="fp">
          <xsl:value-of select="@fp"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="@fp_ssl and @fp_ssl!=''">
        <xsl:attribute name="fpssl">
          <xsl:value-of select="@fp_ssl"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="@fp_enable and @fp_enable!=''">
        <xsl:attribute name="fpauth">
          <xsl:value-of select="@fp_enable"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:attribute name="webstat">
        <xsl:choose>
          <xsl:when test="@webstat">
            <xsl:value-of select="@webstat"/>
          </xsl:when>
          <xsl:otherwise>none</xsl:otherwise>
        </xsl:choose>
      </xsl:attribute>
      <xsl:attribute name="errdocs">
        <xsl:choose>
          <xsl:when test="errordocs">true</xsl:when>
          <xsl:otherwise>false</xsl:otherwise>
        </xsl:choose>
      </xsl:attribute>
      <xsl:attribute name="iis-app-pool">
        <xsl:choose>
          <xsl:when test="../dom_params/dom_param[@param='iis_app_pool']/@val='true'">true</xsl:when>
          <xsl:otherwise>false</xsl:otherwise>
        </xsl:choose>
      </xsl:attribute>
      <xsl:attribute name="wu_script">
        <xsl:choose>
          <xsl:when test="../dom_params/dom_param[@param='wu_script']/@val='true'">true</xsl:when>
          <xsl:otherwise>false</xsl:otherwise>
        </xsl:choose>
      </xsl:attribute>

      <xsl:call-template name="content"/>

      <preferences>
        <xsl:for-each select="sys_user">
          <xsl:call-template name="sysuser"/>
        </xsl:for-each>

        <xsl:for-each select="../dom_params/dom_param/log_rotation">
          <xsl:call-template name="log-rotation"/>
        </xsl:for-each>

        <xsl:for-each select="../anon_ftps/anon_ftp">
          <xsl:call-template name="anon-ftp"/>
        </xsl:for-each>

        <xsl:for-each select="../protected_dirs">
          <xsl:call-template name="protected-dir"/>
        </xsl:for-each>

				<xsl:for-each select="..">
					<xsl:call-template name="sb-domain">
						<xsl:with-param name="clientNode" select=".."/>
					</xsl:call-template>
				</xsl:for-each>
				
        <xsl:for-each select="qos">
          <xsl:call-template name="perfomance"/>
        </xsl:for-each>

        <xsl:for-each select="../hotlink_prot">
          <xsl:call-template name="hotlink-protection"/>
        </xsl:for-each>

        <xsl:for-each select="../shared_ssl_links/shared_ssl_link">
          <xsl:call-template name="shared-ssl-link"/>
        </xsl:for-each>

        <xsl:for-each select="../dom_params/dom_param[@param = 'iis_app_pool']">
          <xsl:element name="iis-application-pool">
            <xsl:attribute name="turned-on">
              <xsl:call-template name="strToBool">
                <xsl:with-param name="value" select="@val"/>
              </xsl:call-template>
            </xsl:attribute>
            <xsl:for-each select="iis_pool_cpu_monitoring">
              <xsl:element name="iis-app-pool-cpu-usage-monitoring">
                <xsl:attribute name="state">
                  <xsl:text>true</xsl:text>
                </xsl:attribute>
                <xsl:attribute name="value">
                  <xsl:value-of select="@cpu_usage"/>
                </xsl:attribute>
              </xsl:element>
            </xsl:for-each>
          </xsl:element>
        </xsl:for-each> 
        
      </preferences>

      <limits-and-permissions>
        <xsl:call-template name="scripting"/>
      </limits-and-permissions>

      <xsl:if test="count(../siteapps/siteapp)>0">
        <applications>
          <xsl:for-each select="../siteapps/siteapp">
            <xsl:call-template name="sapp-installed"/>
          </xsl:for-each>
        </applications>
      </xsl:if>

      <xsl:if test="count(../web_users) > 0">
        <webusers>
          <xsl:for-each select="../web_users">
            <xsl:call-template name="web-user"/>
          </xsl:for-each>
        </webusers>
      </xsl:if>

      <xsl:if test="../ftp_users/ftp_user">
        <ftpusers>
          <xsl:for-each select="../ftp_users/ftp_user">
            <xsl:call-template name="ftp-user"/>
          </xsl:for-each>
        </ftpusers>
      </xsl:if>

      <xsl:if test="../fpse_users/fpse_user">
        <frontpageusers>
          <xsl:for-each select="../fpse_users/fpse_user">
            <xsl:call-template name="frontpage-user"/>
          </xsl:for-each>
        </frontpageusers>
      </xsl:if>

      <xsl:if test="count(../subdomains) > 0">
        <subdomains>
          <xsl:for-each select="../subdomains">
            <xsl:call-template name="subdomain"/>
          </xsl:for-each>
        </subdomains>
      </xsl:if>
      
    </phosting>
  </xsl:template>
  
  <xsl:template name="subdomain">
    <subdomain>
      <xsl:if test="../@guid and ../@guid != '0000-0000-0000-0000'">
        <xsl:attribute name="owner-guid">
          <xsl:call-template name="guid">
            <xsl:with-param name="guid" select="../@guid"/>
          </xsl:call-template>
        </xsl:attribute>
      </xsl:if>
      <xsl:attribute name="name">
        <xsl:value-of select="@displayName"/>
      </xsl:attribute>
      <xsl:attribute name="webstat">
        <xsl:choose>
          <xsl:when test="@webstat">
            <xsl:value-of select="@webstat"/>
          </xsl:when>
          <xsl:otherwise>none</xsl:otherwise>
        </xsl:choose>
      </xsl:attribute>
      <xsl:attribute name="hosting-type">
        <xsl:choose>
          <xsl:when test="@htype">
            <xsl:value-of select="@htype"/>
          </xsl:when>
          <xsl:otherwise>vrt_hst</xsl:otherwise>
        </xsl:choose>
      </xsl:attribute>
      <xsl:if test="@htype='sub_hst'">
        <xsl:attribute name="www-root">
          <xsl:value-of select="@www_root"/>
        </xsl:attribute>
      </xsl:if>

      <xsl:if test="(@htype='vrt_hst' and count(../hosting/*[@arcname])>0) or (count(*[@arcname or @file])>0)">
        <content>
          <xsl:if test="@htype='vrt_hst'">
            <xsl:for-each select="../hosting">
              <xsl:if test="count(*[@arcname])>0">
                <xsl:for-each select="*[@arcname]">
                  <cid>
                    <content-file>
                      <xsl:copy-of select="."/>
                    </content-file>
                  </cid>
                </xsl:for-each>
              </xsl:if>
            </xsl:for-each>
          </xsl:if>
          <xsl:for-each select="*[@arcname or @file]">
            <cid>
              <content-file>
                <xsl:copy-of select="."/>
              </content-file>
            </cid>
          </xsl:for-each>
        </content>
      </xsl:if>

      <xsl:if test="@htype='vrt_hst' and ../hosting/sys_user/@login!=sys_user/@login">
        <xsl:for-each select="sys_user">
          <xsl:call-template name="sysuser"/>
        </xsl:for-each>
      </xsl:if>

      <xsl:call-template name="scripting"/>

      <xsl:if test="count(siteapps/siteapp)>0">
        <applications>
          <xsl:for-each select="siteapps/siteapp">
            <xsl:call-template name="sapp-installed"/>
          </xsl:for-each>
        </applications>
      </xsl:if>

      <preferences>

        <xsl:for-each select="hotlink_prot">
          <xsl:call-template name="hotlink-protection"/>
        </xsl:for-each>

        <xsl:for-each select="protected_dirs">
          <xsl:call-template name="protected-dir"/>
        </xsl:for-each>

        <xsl:for-each select="shared_ssl_links/shared_ssl_link">
          <xsl:call-template name="shared-ssl-link"/>
        </xsl:for-each>

        <xsl:for-each select="qos">
          <xsl:call-template name="perfomance"/>
        </xsl:for-each>

      </preferences>

			<xsl:call-template name="sb-domain">
				<xsl:with-param name="clientNode" select="../.."/>
			</xsl:call-template>

    </subdomain>
  </xsl:template>

  <xsl:template name="web-user">
    <webuser>
      <xsl:attribute name="name">
        <xsl:value-of select="sys_user/@login"/>
      </xsl:attribute>

      <xsl:call-template name="content"/>

      <xsl:for-each select="sys_user">
        <xsl:call-template name="sysuser"/>
      </xsl:for-each>

      <xsl:call-template name="scripting"/>
    </webuser>
  </xsl:template>
  
  <xsl:template name="shared-ssl-link">
    <shared-ssl>
      <xsl:attribute name="target">
        <xsl:value-of select="@target"/>
      </xsl:attribute>
      <xsl:attribute name="require-ssl">
        <xsl:value-of select="@require_ssl"/>
      </xsl:attribute>

      <virtual-directory-name>
        <xsl:value-of select="@name"/>
      </virtual-directory-name>

    </shared-ssl>
  </xsl:template>

  <xsl:template name="hotlink-extensions">
    <xsl:param name="list"/>

		<xsl:variable name="extension">
			<xsl:choose>
				<xsl:when test="substring-before($list, ',') != ''">
					<xsl:value-of select="substring-before($list, ',')"/>
				</xsl:when>
				<xsl:when test="$list != ''">
					<xsl:value-of select="$list"/>
				</xsl:when>
			</xsl:choose>
		</xsl:variable>
    <xsl:variable name="rest" select="substring-after($list, ',')"/>

    <xsl:if test="$extension and $extension != ''">

      <extension><xsl:value-of select="$extension"/></extension>

      <xsl:if test="$rest and $rest != ''">
        <xsl:call-template name="hotlink-extensions">
          <xsl:with-param name="list" select="$rest"/>
        </xsl:call-template>
      </xsl:if>

    </xsl:if>

  </xsl:template>

  <xsl:template name="hotlink-protection">
    <hotlink-protection>
      <xsl:attribute name="enabled">
        <xsl:value-of select="@enabled"/>
      </xsl:attribute>

      <xsl:call-template name="hotlink-extensions">
        <xsl:with-param name="list" select="@extensions"/>
      </xsl:call-template>

      <xsl:for-each select="../hotlink_friends/hotlink_friend">
        <friend>
          <xsl:value-of select="@friend_dom"/>
        </friend>
      </xsl:for-each>
    </hotlink-protection>
  </xsl:template>
  
  <xsl:template name="protected-dir">
    <pdir>

      <xsl:attribute name="name">
        <xsl:value-of select="@path"/>
      </xsl:attribute>
      <xsl:if test ="@realm and @realm!=''">
        <xsl:attribute name="title">
          <xsl:value-of select="@realm"/>
        </xsl:attribute>
      </xsl:if>

      <xsl:for-each select="pd_users">

        <pduser>

          <xsl:attribute name="name">
            <xsl:value-of select="@login"/>
          </xsl:attribute>

          <xsl:for-each select="account">
            <xsl:call-template name="password"/>
          </xsl:for-each>

        </pduser>

      </xsl:for-each>

    </pdir>
  </xsl:template>
  
  <xsl:template name="perfomance">
    <xsl:if test="(@bandwidth and @bandwidth>-1) or (@max-connections and @max-connections>-1)">
      <perfomance>
        <max-connections>
          <xsl:value-of select="@max-connections"/>
        </max-connections>
        <traffic-bandwidth>
          <xsl:value-of select="@bandwidth"/>
        </traffic-bandwidth>
      </perfomance>
    </xsl:if>
  </xsl:template>
  
  <xsl:template name="sb-domain">
		<xsl:param name="clientNode"/>
    <xsl:if test="wsb_params/wsb_param[@param='sbsite_guid']">
      <sb-domain>
        <sb-site-id>
          <xsl:value-of select="wsb_params/wsb_param[@param='sbsite_guid']/@val"/>
        </sb-site-id>
        <sb-siteowner-id>
          <xsl:value-of select="$clientNode/client_params/client_param[@param='sbnet_id']/@val"/>
        </sb-siteowner-id>
        <sb-siteowner-login>
          <xsl:value-of select="$clientNode/@login"/>
        </sb-siteowner-login>

        <xsl:for-each select="$clientNode/client_params/client_param[@param='sbnet_id']">
          <xsl:call-template name="content"/>
        </xsl:for-each>
      </sb-domain>
    </xsl:if>
  </xsl:template>

  <xsl:template name="scripting">
    <scripting>
      <xsl:if test="@fp and @fp!=''">
        <xsl:attribute name="fp">
          <xsl:value-of select="@fp"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="@fp_enable and @fp_enable!=''">
        <xsl:attribute name="fp_enable">
          <xsl:value-of select="@fp_enable"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="@ssi and @ssi!=''">
        <xsl:attribute name="ssi">
          <xsl:value-of select="@ssi"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="@php and @php!=''">
        <xsl:attribute name="php">
          <xsl:value-of select="@php"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="@cgi and @cgi!=''">
        <xsl:attribute name="cgi">
          <xsl:value-of select="@cgi"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="@perl and @perl!=''">
        <xsl:attribute name="perl">
          <xsl:value-of select="@perl"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="@asp and @asp!=''">
        <xsl:attribute name="asp">
          <xsl:value-of select="@asp"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="@asp_dot_net and @asp_dot_net!=''">
        <xsl:attribute name="asp_dot_net">
          <xsl:value-of select="@asp_dot_net"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="@python and @python!=''">
        <xsl:attribute name="python">
            <xsl:value-of select="@python"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="@miva and @miva!=''">
        <xsl:attribute name="miva">
          <xsl:value-of select="@miva"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:attribute name="write_modify">
        <xsl:if test="count(@write_modify) > 0">
            <xsl:value-of select="@write_modify"/>
        </xsl:if>
      </xsl:attribute>
			<xsl:if test="local-name(.) != 'web_users'">
				<xsl:if test="@ssi_html and @ssi_html!=''">
					<xsl:attribute name="ssi_html">
						<xsl:value-of select="@ssi_html"/>
					</xsl:attribute>
				</xsl:if>
				<xsl:if test="@coldfusion and @coldfusion!=''">
					<xsl:attribute name="coldfusion">
						<xsl:value-of select="@coldfusion"/>
					</xsl:attribute>
				</xsl:if>
				<xsl:if test="@managed_runtime_version and @managed_runtime_version != ''">
					<xsl:choose>
						<xsl:when test="starts-with(@managed_runtime_version, '1.1')">
							<xsl:attribute name="managed_runtime_version">
								<xsl:text>1.1</xsl:text>
							</xsl:attribute>
						</xsl:when>
						<xsl:when test="starts-with(@managed_runtime_version, '2.0')">
							<xsl:attribute name="managed_runtime_version">
								<xsl:text>2.0</xsl:text>
							</xsl:attribute>
						</xsl:when>
						<xsl:otherwise>
							<xsl:message terminate="no">
								Unexpected value of asp.net version: <xsl:value-of select="@managed_runtime_version"/>
							</xsl:message>
						</xsl:otherwise>
					</xsl:choose>
				</xsl:if>
				<xsl:if test="@php_version and @php_version!=''">
					<xsl:attribute name="php_version">
						<xsl:value-of select="@php_version"/>
					</xsl:attribute>
				</xsl:if>
				<xsl:if test="@php_handler_type and @php_handler_type!=''">
					<xsl:attribute name="php_handler_type">
						<xsl:value-of select="@php_handler_type"/>
					</xsl:attribute>
				</xsl:if>
			</xsl:if>
    </scripting>
  </xsl:template>
  
  <xsl:template name="anon-ftp">
    <anonftp>
      <xsl:attribute name="pub">
        <xsl:choose>
          <xsl:when test="@status='true'">true</xsl:when>
          <xsl:otherwise>false</xsl:otherwise>
        </xsl:choose>
      </xsl:attribute>
      <xsl:attribute name="incoming">
        <xsl:choose>
          <xsl:when test="@incoming='true'">true</xsl:when>
          <xsl:otherwise>false</xsl:otherwise>
        </xsl:choose>
      </xsl:attribute>
      <xsl:attribute name="display-login">
        <xsl:value-of select="@display_login"/>
      </xsl:attribute>

      <xsl:for-each select="../../hosting">
        <xsl:if test="count(*[@arcname])>0">
          <content>
            <xsl:for-each select="*[@arcname]">
              <cid>
                <content-file>
                  <xsl:copy-of select="."/>
                </content-file>
              </cid>
            </xsl:for-each>
          </content>
        </xsl:if>
      </xsl:for-each>

      <xsl:if test="@max_conn and @max_conn>0">
        <anonftp-limit name="max-connections">
          <xsl:value-of select="@max_conn"/>
        </anonftp-limit>
      </xsl:if>
      <xsl:if test="@bandwidth and @bandwidth>0">
        <anonftp-limit name="bandwidth">
          <xsl:value-of select="@bandwidth"/>
        </anonftp-limit>
      </xsl:if>
      <xsl:if test="@quota and @quota>0">
        <anonftp-limit name="incoming-disk-quota">
          <xsl:value-of select="@quota"/>
        </anonftp-limit>
      </xsl:if>

      <xsl:if test="@incoming_readable">
        <anonftp-permission name="incoming-download"/>
      </xsl:if>
      <xsl:if test="@incoming_subdirs">
        <anonftp-permission name="incoming-mkdir"/>
      </xsl:if>

      <login-message>
        <xsl:value-of select="@login_text"/>
      </login-message>
    </anonftp>
  </xsl:template>

  <xsl:template name="ftp-user">
    <ftpuser>

		<xsl:attribute name="name">
			<xsl:value-of select="sys_user/@login"/>
		</xsl:attribute>
      <xsl:for-each select="sys_user"> <!--loop is needed only for changing current context-->
				<xsl:call-template name="sysuser">
					<xsl:with-param name="include-relative-home" select="'true'"/>
				</xsl:call-template>
      </xsl:for-each>

      <xsl:if test="@permission mod 2 = 1">
        <permission>download</permission>
      </xsl:if>
      <xsl:if test="(@permission mod 4) - (@permission mod 2) = 2">
        <permission>upload</permission>
      </xsl:if>
    
    </ftpuser>
  </xsl:template>

  <xsl:template name="frontpage-user">
    <frontpageuser>
      <xsl:attribute name="name">
        <xsl:value-of select="sys_user/@login"/>
      </xsl:attribute>
      <xsl:for-each select="sys_user">
        <xsl:call-template name="sysuser"/>
      </xsl:for-each>
    </frontpageuser>
  </xsl:template>
  
  <xsl:template name="password">
    <password>
      <xsl:attribute name="type">
        <xsl:value-of select="@type"/>
      </xsl:attribute>
      <xsl:value-of select="@password"/>
    </password>
  </xsl:template>
  
  <xsl:template name="sapp-installed">
    <sapp-installed>

      <xsl:call-template name="content"/>
      
      <xsl:call-template name="sapp-spec"/>

      <xsl:for-each select="parameters/parameter">
        <sapp-param>
          <sapp-param-name encoding="plain">
            <xsl:value-of select="@parameter"/>
          </sapp-param-name>
          <sapp-param-value encoding="plain">
            <xsl:value-of select="@value"/>
          </sapp-param-value>
        </sapp-param>
      </xsl:for-each>

      <xsl:for-each select=".//database">
        <xsl:call-template name="database">
          <xsl:with-param name="siteapps" select="1"/>
        </xsl:call-template>
      </xsl:for-each>

      <xsl:for-each select="siteapp_resources/siteapp_resource/custom_button">
        <xsl:call-template name="custom-button"/>
      </xsl:for-each>

      <sapp-installdir>
        <sapp-prefix>
          <xsl:value-of select="@install_prefix"/>
        </sapp-prefix>
      </sapp-installdir>

    </sapp-installed>
  </xsl:template>

  <xsl:template name="sysuser">
		<xsl:param name="include-relative-home"/>
    <sysuser>
      <xsl:attribute name="name">
        <xsl:value-of select="@login"/>
      </xsl:attribute>
      <xsl:attribute name="quota">
        <xsl:choose>
          <xsl:when test="@quota='0'">
            <xsl:value-of select="'-1'"/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:value-of select="@quota"/>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:attribute>
      <xsl:if test="@shell='Login Enabled' or @shell='Login Disabled'">
        <xsl:attribute name="shell">
          <xsl:value-of select="@shell"/>
        </xsl:attribute>
      </xsl:if>
			<xsl:if test="$include-relative-home = 'true'">
				<xsl:attribute name="home">
					<xsl:variable name="vhosts-path" select="translate(/psadump/directories/@vhosts_dir, '\', '/')"/>
					<xsl:variable name="domain-root-path">
						<xsl:choose>
							<xsl:when test="substring($vhosts-path, string-length($vhosts-path)) = '/'">
								<xsl:value-of select="concat($vhosts-path, ancestor::domain/@name)"/>
							</xsl:when>
							<xsl:otherwise>
								<xsl:value-of select="concat($vhosts-path, '/', ancestor::domain/@name)"/>
							</xsl:otherwise>
						</xsl:choose>
					</xsl:variable>
					<xsl:value-of select="translate(substring(@home, string-length($domain-root-path) + 1 ), '\', '/')"/>
				</xsl:attribute>
			</xsl:if>

      <xsl:for-each select="account">
        <xsl:call-template name="password"/>
      </xsl:for-each>

      <xsl:if test="count(dump) > 0">
        <xsl:element name="cron">
          <xsl:value-of select="dump/@arcname"/>
        </xsl:element>
      </xsl:if>
      
    </sysuser>
  </xsl:template>

  <xsl:template name="shosting">
    <shosting>
      <xsl:value-of select="@redirect"/>
    </shosting>
  </xsl:template>

  <xsl:template name="fhosting">
    <fhosting>
      <xsl:value-of select="@redirect"/>
    </fhosting>
  </xsl:template>

  <xsl:template name="domainuser">
    <domainuser>
      <xsl:if test="cards/@personalName">
        <xsl:attribute name="contact">
          <xsl:value-of select="cards/@personalName"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:attribute name="status">
        <xsl:value-of select="@state"/>
      </xsl:attribute>
      <xsl:if test="../dom_params/dom_param[@param='max_button_length']/@val">
        <xsl:attribute name="max-button-length">
          <xsl:value-of select="../dom_params/dom_param[@param='max_button_length']/@val"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:variable name="skinId" select="../dom_params/dom_param[@param='skin_id']/@val"/>
      <xsl:if test="$skinId!='' and $skinId!=0">
        <xsl:attribute name="skin">
          <xsl:value-of select="//skin[@id=$skinId]/@name"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:attribute name="lock-screen">
        <xsl:choose>
          <xsl:when test="../dom_params/dom_param[@param='disable_lock_screen']/@val='false'">false</xsl:when>
          <xsl:otherwise>true</xsl:otherwise>
        </xsl:choose>
      </xsl:attribute>

      <password type="plain">
        <xsl:value-of select="@passwd"/>
      </password>

      <pinfo name="country">
        <xsl:value-of select="cards/@country"/>
      </pinfo>
      <pinfo name="phone">
        <xsl:value-of select="cards/@phone"/>
      </pinfo>
      <pinfo name="fax">
        <xsl:value-of select="cards/@fax"/>
      </pinfo>
      <pinfo name="state">
        <xsl:value-of select="cards/@state"/>
      </pinfo>
      <pinfo name="email">
        <xsl:value-of select="cards/@email"/>
      </pinfo>
      <pinfo name="city">
        <xsl:value-of select="cards/@city"/>
      </pinfo>
      <pinfo name="zip">
        <xsl:value-of select="cards/@pcode"/>
      </pinfo>
      <pinfo name="address">
        <xsl:value-of select="cards/@address"/>
      </pinfo>
      <pinfo name="company">
        <xsl:value-of select="cards/@companyName"/>
      </pinfo>
      <xsl:if test="@locale">
        <pinfo name="locale">
          <xsl:for-each select="@locale">
            <xsl:call-template name="locale"/>
          </xsl:for-each>
        </pinfo>
      </xsl:if>

      <xsl:for-each select="../dom_params/dom_param/permissions">
        <xsl:call-template name="permission"/>
      </xsl:for-each>
      
    </domainuser>
  </xsl:template>
  
  <xsl:template name="certificate">
    <certificate>
      <xsl:attribute name="name">
        <xsl:value-of select="@name"/>
      </xsl:attribute>

      <xsl:if test="@cert and @cert!=''">
        <certificate-data>
          <xsl:value-of select="@cert"/>
        </certificate-data>
      </xsl:if>
      <xsl:if test="@csr and @csr!=''">
        <signing-request>
          <xsl:value-of select="@csr"/>
        </signing-request>
      </xsl:if>
      <xsl:if test="@ca_cert and @ca_cert!=''">
        <ca-certificate>
          <xsl:value-of select="@ca_cert"/>
        </ca-certificate>
      </xsl:if>
      <xsl:if test="@pvt_key and @pvt_key!=''">
        <private-key>
          <xsl:value-of select="@pvt_key"/>
        </private-key>
      </xsl:if>
      <xsl:call-template name="content"/>
    </certificate>
  </xsl:template>
  
  <xsl:template name="traffic">
    <xsl:if test=".!=0">
      <xsl:variable name="realNameIn" select="substring-before(name(), '_in')"/>
      <xsl:variable name="realNameOut" select="substring-before(name(), '_out')"/>
      <xsl:variable name="realName" select="concat($realNameIn, $realNameOut)"/>
      <xsl:variable name="direction" select="substring-after(name(), concat($realName, '_'))"/>
      <xsl:value-of select="substring(../@date, 0, 11)"/>
      <xsl:text> </xsl:text>
      <xsl:value-of select="translate($realName, '_', '-')"/>
      <xsl:text> </xsl:text>
      <xsl:value-of select="$direction"/>
      <xsl:text> </xsl:text>
      <xsl:value-of select="."/>
      <xsl:text>
</xsl:text>
    </xsl:if>
  </xsl:template>
  
  <xsl:template name="domain-alias">
    <domain-alias>

      <xsl:attribute name="name">
        <xsl:choose>
          <xsl:when test="count(@displayName) > 0">
            <xsl:value-of select="@displayName"/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:value-of select="@name"/>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:attribute>

      <xsl:attribute name="mail">
        <xsl:value-of select="@mail"/>
      </xsl:attribute>
      <xsl:attribute name="web">
        <xsl:value-of select="@web"/>
      </xsl:attribute>
      <xsl:if test="@dns">
        <xsl:attribute name="dns">
          <xsl:value-of select="@dns"/>
        </xsl:attribute>
      </xsl:if>

      <xsl:call-template name="status"/>
      
      <xsl:for-each select="dns_zone">
        <xsl:call-template name="dns-zone"/>
      </xsl:for-each>
    
    </domain-alias>
  </xsl:template>

  <xsl:template name="status">
    <status>
      <xsl:choose>
        <xsl:when test="@status=0">
          <enabled/>
        </xsl:when>
        <xsl:otherwise>
          <disabled-by name="admin"/>
        </xsl:otherwise>
      </xsl:choose>
    </status>
  </xsl:template>
  
  <xsl:template name="dns-zone">
    <dns-zone>
      <xsl:attribute name="email">
        <xsl:value-of select="@email"/>
      </xsl:attribute>
      <xsl:attribute name="type">
        <xsl:value-of select="@type"/>
      </xsl:attribute>

      <xsl:call-template name="status"/>

      <xsl:for-each select="@ttl|@refresh|@retry|@expire|@minimum">
				<xsl:variable name="name" select="name()"/>
				<xsl:call-template name="dns-zone-param">
					<xsl:with-param name="name" select="$name"/>
					<xsl:with-param name="value" select="."/>
					<xsl:with-param name="unitValue" select="number(../@*[name() = concat($name, '_unit')])"/>
				</xsl:call-template>
      </xsl:for-each>

      <xsl:for-each select="dns_records/dns_rec">
        <xsl:call-template name="dns-rec"/>
      </xsl:for-each>

    </dns-zone>
  </xsl:template>

  <xsl:template name="dns-zone-param">
		<xsl:param name="name"/>
		<xsl:param name="value"/>
		<xsl:param name="unitValue"/>
    <dns-zone-param>
      <xsl:attribute name="name">
        <xsl:value-of select="$name"/>
      </xsl:attribute>
      <xsl:attribute name="value">
        <xsl:value-of select="number($value)"/>
      </xsl:attribute>
      <xsl:attribute name="unit">
        <xsl:choose>
          <xsl:when test="$unitValue=1">second</xsl:when>
          <xsl:when test="$unitValue=60">minutes</xsl:when>
          <xsl:when test="$unitValue=3600">hours</xsl:when>
          <xsl:when test="$unitValue=86400">days</xsl:when>
          <xsl:when test="$unitValue=604800">weeks</xsl:when>
          <xsl:otherwise>
            <xsl:choose>
              <xsl:when test="($name = 'minimum') or ($name = 'refresh') or ($name = 'retry')">
                <xsl:text>hours</xsl:text>
              </xsl:when>
              <xsl:when test="($name = 'ttl') or ($name = 'expire')">
                <xsl:text>days</xsl:text>
              </xsl:when>
            </xsl:choose>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:attribute>
    </dns-zone-param>
  </xsl:template>

  <xsl:template name="dns-rec">
    <dnsrec>
      <xsl:attribute name="type">
        <xsl:value-of select="@type"/>
      </xsl:attribute>
      <xsl:attribute name="src">
        <xsl:value-of select="@displayHost"/>
      </xsl:attribute>
      <xsl:if test="@val">
        <xsl:attribute name="dst">
          <xsl:value-of select="@val"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="@opt">
        <xsl:attribute name="opt">
          <xsl:value-of select="@opt"/>
        </xsl:attribute>
      </xsl:if>
    </dnsrec>
  </xsl:template>

  <xsl:template name="mail">
    <mailuser>
      <xsl:attribute name="id">
        <xsl:value-of select="@id"/>
      </xsl:attribute>
      <xsl:if test="../@guid and ../@guid != '0000-0000-0000-0000'">
        <xsl:attribute name="guid">
          <xsl:call-template name="guid">
            <xsl:with-param name="guid" select="../@guid"/>
            <xsl:with-param name="name" select="'_mn_'"/>
            <xsl:with-param name="id" select="@id"/>
          </xsl:call-template>
        </xsl:attribute>
        <xsl:attribute name="owner-guid">
          <xsl:call-template name="guid">
            <xsl:with-param name="guid" select="../@guid"/>
          </xsl:call-template>
        </xsl:attribute>
      </xsl:if>
      <xsl:attribute name="name">
        <xsl:value-of select="@mail_name"/>
      </xsl:attribute>
      <xsl:attribute name="mailgroup-enabled">
        <xsl:value-of select="@mail_group"/>
      </xsl:attribute>
      <xsl:if test="@mbox_quota and @mbox_quota>-1">
        <xsl:attribute name="mailbox-quota">
          <xsl:value-of select="@mbox_quota"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="mn_param[@param='max_button_length']/@val and mn_param[@param='max_button_length']/@val>-1">
        <xsl:attribute name="max-button-length">
          <xsl:value-of select="mn_param[@param='max_button_length']/@val"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:variable name="skinId" select="mn_param[@param='skin_id']/@val"/>
      <xsl:if test="$skinId!='' and $skinId!=0">
        <xsl:attribute name="skin">
          <xsl:value-of select="//skin[@id=$skinId]/@name"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:attribute name="lock-screen">
        <xsl:choose>
          <xsl:when test="mn_param[@param='disable_lock_screen']/@val='false'">false</xsl:when>
          <xsl:otherwise>true</xsl:otherwise>
        </xsl:choose>
      </xsl:attribute>

      <properties>
        <xsl:for-each select="account">
          <xsl:call-template name="password"/>
        </xsl:for-each>
      </properties>

      <limits-and-permissions>
        <xsl:for-each select="permissions_list/permissions">
          <xsl:call-template name="mailuser-permission"/>
        </xsl:for-each>
      </limits-and-permissions>

      <preferences>
        
        <mailbox enabled="{@postbox}">
          <xsl:if test="@contentType='independent'">
            <xsl:call-template name="content"/>
          </xsl:if>
        </mailbox>

        <xsl:for-each select="mail_aliases">
          <alias>
            <xsl:value-of select="@alias"/>
          </alias>
        </xsl:for-each>

        <xsl:if test="@redir_addr">
          <xsl:element name="redirect">
            <xsl:choose>
              <xsl:when test="@redirect = 'true'">
                <xsl:attribute name="enabled">
                  <xsl:text>true</xsl:text>
                </xsl:attribute>
                <xsl:value-of select="@redir_addr"/>
              </xsl:when>
              <xsl:otherwise>
                <xsl:attribute name="enabled">
                  <xsl:text>false</xsl:text>
                </xsl:attribute>
              </xsl:otherwise>
            </xsl:choose>
          </xsl:element>
        </xsl:if>
        
        <xsl:for-each select="mail_redir">
          <mailgroup-member>
            <xsl:value-of select="@address"/>
          </mailgroup-member>
        </xsl:for-each>

        <xsl:call-template name="autoresponders"/>

        <xsl:for-each select="sa_conf">
          <xsl:call-template name="spamassassin"/>
        </xsl:for-each>

        <xsl:if test="mn_param[@param='antivir_incoming']/@val='true' or mn_param[@param='antivir_outgoing']/@val='true'">
          <virusfilter>
            <xsl:attribute name="state">
              <xsl:choose>
                <xsl:when test="mn_param[@param='antivir_incoming']/@val='true' and mn_param[@param='antivir_outgoing']/@val='true'">inout</xsl:when>
                <xsl:when test="mn_param[@param='antivir_incoming']/@val='true'">in</xsl:when>
                <xsl:when test="mn_param[@param='antivir_outgoing']/@val='true'">out</xsl:when>
              </xsl:choose>
            </xsl:attribute>
          </virusfilter>
        </xsl:if>
        
        <xsl:for-each select="custom_buttons/custom_button">
          <xsl:call-template name="custom-button"/>
        </xsl:for-each>

      </preferences>
    </mailuser>
  </xsl:template>

  <xsl:template name="autoresponders">
    <autoresponders enabled="{@autoresponder}">
      <xsl:for-each select="mail_resp">
        <autoresponder>
          <xsl:attribute name="name">
            <xsl:value-of select="@resp_name"/>
          </xsl:attribute>
          <xsl:attribute name="status">
            <xsl:choose>
              <xsl:when test="@resp_on='true'">on</xsl:when>
              <xsl:otherwise>off</xsl:otherwise>
            </xsl:choose>
          </xsl:attribute>
          <xsl:if test="@key_where!='no'">
            <xsl:attribute name="require">
              <xsl:value-of select="concat(@key_where, ':', @keystr)"/>
            </xsl:attribute>
          </xsl:if>
          <xsl:attribute name="subject">
            <xsl:value-of select="@subject"/>
          </xsl:attribute>
          <xsl:attribute name="replyto">
            <xsl:value-of select="@reply_to"/>
          </xsl:attribute>
          <xsl:if test="resp_forward/@address">
            <xsl:attribute name="redirect">
              <xsl:value-of select="resp_forward/@address"/>
            </xsl:attribute>
          </xsl:if>
          <xsl:attribute name="content-type">
            <xsl:choose>
              <xsl:when test="@ishtml='true'">plain/html</xsl:when>
              <xsl:otherwise>plain/text</xsl:otherwise>
            </xsl:choose>
          </xsl:attribute>

          <text>
            <xsl:value-of select="@text"/>
          </text>

          <autoresponder-limit name="ans-freq">
            <xsl:value-of select="@ans_freq"/>
          </autoresponder-limit>
          <autoresponder-limit name="ans-count">
            <xsl:value-of select="@mem_limit"/>
          </autoresponder-limit>

          <xsl:for-each select="resp_attach">
            <attach file="{@filename}"/>
          </xsl:for-each>

        </autoresponder>
      </xsl:for-each>
      <xsl:if test="./dump[@type='directory' and @arcname=concat('attachments ',../@mail_name,'@',../../@name)]">
      <content>
        <xsl:element name="cid">
          <xsl:attribute name="type">
            <xsl:text>attaches</xsl:text>
          </xsl:attribute>
          <xsl:attribute name="referrer">
            <xsl:text>true</xsl:text>
          </xsl:attribute>
          <xsl:attribute name="offset">
            <xsl:value-of select="concat(@mail_name,'@',../@name)"/>
          </xsl:attribute>
          <xsl:attribute name="unpacksize">
            <xsl:text>0</xsl:text>
          </xsl:attribute>
          <content-file>
            <dump type="directory" arcname="{concat('attachments ',@mail_name,'@',../@name)}" />
          </content-file>
        </xsl:element>
      </content>
    </xsl:if>
    </autoresponders>
  </xsl:template>

  <xsl:template name="spamassassin">
		<xsl:variable name="isServer" select="local-name(../..) = 'server'"/>
    <spamassassin>
      <xsl:choose>
        <xsl:when test="@flt_enabled='both'">
          <xsl:attribute name="status">on</xsl:attribute>
					<xsl:choose>
						<xsl:when test="$isServer = false">
		          <xsl:attribute name="server-conf">true</xsl:attribute>
						</xsl:when>
						<xsl:otherwise>
							<xsl:attribute name="personal-conf">true</xsl:attribute>
						</xsl:otherwise>
					</xsl:choose>
        </xsl:when>
        <xsl:when test="@flt_enabled='user'">
					<xsl:choose>
						<xsl:when test="$isServer = false">
							<xsl:attribute name="status">on</xsl:attribute>
							<xsl:attribute name="server-conf">false</xsl:attribute>
						</xsl:when>
						<xsl:otherwise>
							<xsl:attribute name="status">off</xsl:attribute>
							<xsl:attribute name="personal-conf">true</xsl:attribute>
						</xsl:otherwise>
					</xsl:choose>
        </xsl:when>
        <xsl:when test="@flt_enabled='serv'">
          <xsl:attribute name="status">on</xsl:attribute>
          <xsl:attribute name="personal-conf">false</xsl:attribute>
        </xsl:when>
        <xsl:when test="@flt_enabled=''">
          <xsl:attribute name="status">off</xsl:attribute>
					<xsl:choose>
						<xsl:when test="$isServer = false">
		          <xsl:attribute name="server-conf">false</xsl:attribute>
						</xsl:when>
						<xsl:otherwise>
							<xsl:attribute name="personal-conf">false</xsl:attribute>
						</xsl:otherwise>
					</xsl:choose>
        </xsl:when>
        <xsl:otherwise>
          <xsl:attribute name="status">off</xsl:attribute>
					<xsl:choose>
						<xsl:when test="$isServer = false">
		          <xsl:attribute name="server-conf">false</xsl:attribute>
						</xsl:when>
						<xsl:otherwise>
							<xsl:attribute name="personal-conf">false</xsl:attribute>
						</xsl:otherwise>
					</xsl:choose>
          <xsl:message>Unexpected value of @flt_enabled</xsl:message>
        </xsl:otherwise>
      </xsl:choose>
      <xsl:attribute name="hits">
        <xsl:value-of select="@hits_required"/>
      </xsl:attribute>
      <xsl:attribute name="action">
        <xsl:choose>
          <xsl:when test="@spam_action='delete'">delete</xsl:when>
          <xsl:when test="@spam_action='save'">mark</xsl:when>
          <xsl:otherwise>
            <xsl:text>mark</xsl:text>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:attribute>
      <xsl:attribute name="modify-subj">
        <xsl:choose>
          <xsl:when test="@rw_subject_tag=''">false</xsl:when>
          <xsl:otherwise>true</xsl:otherwise>
        </xsl:choose>
      </xsl:attribute>
      <xsl:attribute name="subj-text">
        <xsl:value-of select="@rw_subject_tag"/>
      </xsl:attribute>
      <xsl:if test="sa_param[@name = 'max_mail']">
        <xsl:attribute name="max-mail">
          <xsl:value-of select="sa_param[@name = 'max_mail']/@value"/>
        </xsl:attribute>
      </xsl:if>

      <xsl:for-each select="sa_list[@color='black']">
        <blacklist-member>
          <xsl:value-of select="@pattern"/>
        </blacklist-member>
      </xsl:for-each>
      <xsl:for-each select="sa_list[@color='white']">
        <whitelist-member>
          <xsl:value-of select="@pattern"/>
        </whitelist-member>
      </xsl:for-each>

      <xsl:for-each select="sa_param[@name='ok_locale']">
        <trusted-locale>
          <xsl:value-of select="@value"/>
        </trusted-locale>
      </xsl:for-each>
      <xsl:for-each select="sa_param[@name='ok_language']">
        <trusted-language>
          <xsl:value-of select="@value"/>
        </trusted-language>
      </xsl:for-each>
      <xsl:for-each select="sa_param[@name='trusted_network']">
        <trusted-network>
          <xsl:value-of select="@value"/>
        </trusted-network>
      </xsl:for-each>

    </spamassassin>
  </xsl:template>

  <xsl:template name="dsn">
    <odbc-dsn>
      <xsl:attribute name="name">
        <xsl:value-of select="@name"/>
      </xsl:attribute>
      <description>
        <xsl:value-of select="@description"/>
      </description>
      <driver>
        <xsl:value-of select="@driver"/>
      </driver>
      <connection-string>
        <xsl:value-of select="@cstring"/>
      </connection-string>
    </odbc-dsn>
  </xsl:template>

  <xsl:template name="database">
    <xsl:param name="siteapps" select="boolean(false)"/>

    <xsl:variable name="dbId" select="@id"/>
    <xsl:variable name="isResource" select="count(//siteappresource[@type='database' and @res_id=$dbId]) > 0"/>

    <xsl:if test="$siteapps or not($isResource)">
      <database>
        <xsl:attribute name="name">
          <xsl:value-of select="@name"/>
        </xsl:attribute>
        <xsl:attribute name="type">
          <xsl:value-of select="@type"/>
        </xsl:attribute>

        <xsl:if test="../../@guid and ../../@guid != '0000-0000-0000-0000'">
          <xsl:attribute name="guid">
            <xsl:call-template name="guid">
              <xsl:with-param name="guid" select="../../@guid"/>
              <xsl:with-param name="name" select="'_db_'"/>
              <xsl:with-param name="id" select="@id"/>
            </xsl:call-template>
          </xsl:attribute>
          <xsl:attribute name="owner-guid">
            <xsl:call-template name="guid">
              <xsl:with-param name="guid" select="../../@guid"/>
            </xsl:call-template>
          </xsl:attribute>
        </xsl:if>
        
        <xsl:call-template name="content"/>

        <xsl:if test="db_server">
          <db-server>
            <xsl:attribute name="type">
              <xsl:value-of select="db_server/@type"/>
            </xsl:attribute>
            <host>
              <xsl:value-of select="db_server/@host"/>
            </host>
            <port>
              <xsl:value-of select="db_server/@port"/>
            </port>
          </db-server>
				</xsl:if>

				<xsl:choose>
					<xsl:when test="$isResource">
						<xsl:for-each select="//siteappresource[@type='database' and @res_id=$dbId]/../siteappresource[@type='dbuser']/database_user">
							<xsl:call-template name="db-user"/>
						</xsl:for-each>
					</xsl:when>
					<xsl:otherwise>
						<xsl:for-each select="users/database_user">
							<xsl:call-template name="db-user"/>
						</xsl:for-each>
					</xsl:otherwise>
				</xsl:choose>
        
      </database>
    </xsl:if>
  </xsl:template>

	<xsl:template name="db-user">
    <dbuser name="{@login}">
      <password type="plain">
        <xsl:value-of select="@passwd"/>
      </password>
      <xsl:for-each select="mysqluser">
        <accesshost><xsl:value-of select="@Host"/></accesshost>
      </xsl:for-each>
    </dbuser>
	</xsl:template>
	
  <xsl:template name="maillist">
    <maillist name="{@name}">
      <xsl:call-template name="status"/>

      <owner>
        <xsl:choose>
          <xsl:when test="@admin_email and @admin_email != ''">
            <xsl:value-of select="@admin_email"/>
          </xsl:when>
          <xsl:when test="@adminEmail and @adminEmail != ''">
            <xsl:value-of select="@adminEmail"/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:message terminate="no">
              Owner email for mail list <xsl:value-of select="@name"/>@<xsl:value-of select="../@displayName"/> not found in dump
            </xsl:message>
            <xsl:value-of select="concat('listowner@', ../@displayName)"/>
          </xsl:otherwise>
        </xsl:choose>
      </owner>

      <xsl:for-each select="listmember">
        <recipient>
          <xsl:value-of select="@email"/>
        </recipient>
      </xsl:for-each>
    </maillist>
  </xsl:template>

</xsl:stylesheet>
