from parallels.core.registry import Registry
from parallels.core.utils.restore_hosting_utils import restore_hosting_settings_lock
from parallels.plesk.source.plesk import messages
from parallels.core.actions.base.subscription_action import SubscriptionAction
from parallels.core.utils import subscription_filter
from parallels.core.utils import plesk_utils
from parallels.core.utils.common import exists


class FixPermissionsForIIS6Domains(SubscriptionAction):
    """Fix permissions for domains from IIS 6 with ASP.NET disabled

    Workaround for IIS 6 domains with ASP.NET disabled, but dedicated
    application pool enabled: IWPD* user had no access to httpdocs
    directory on source node in such situation, but in IIS 7 IWPD* user
    must have access to read web.config, otherwise the whole site won't
    work. The most easy way to fix that - disable and then enable
    dedicated application pool which will fix permissions both on
    filesystem and in '.Security' file
    """

    def filter_subscription(self, global_context, subscription):
        """
        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :type subscription: parallels.core.migrated_subscription.MigratedSubscription
        :rtype: bool
        """
        if not subscription_filter.windows_with_iis_dedicated_app_pool_enabled(subscription):
            return False
        if not exists(
            global_context.target_existing_objects.subscriptions,
            lambda s: s.name_canonical == subscription.name_canonical
        ):
            return True
        # Check that the action is already executed. This action is executed a long time
        # for subscriptions with huge count of folders. So we skip it repeat runs.
        state_vars = Registry.get_instance().get_migration_state().get_vars(
            action='copy-security/fix-permissions-for-iis6-domains',
            subscription=subscription.name
        )
        return not state_vars.get('executed', False)

    def get_description(self):
        return messages.ACTION_FIX_PERMISSIONS_FOR_IIS6_DOMAIN_DESCRIPTION

    def get_failure_message(self, global_context, subscription):
        """
        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :type subscription: parallels.core.migrated_subscription.MigratedSubscription
        """
        return messages.ACTION_FIX_PERMISSIONS_FOR_IIS6_DOMAIN_FAILURE % subscription.name

    def is_critical(self):
        """If action is critical or not

        If action is critical and it failed for a subscription, migration tool
        won't run the next operations for the subscription.

        :rtype: bool
        """
        return False

    def run(self, global_context, subscription):
        """
        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :type subscription: parallels.core.migrated_subscription.MigratedSubscription
        """
        for app_pool_state in (False, True):
            with restore_hosting_settings_lock:
                plesk_utils.change_dedicated_iis_app_pool_state(
                    global_context.conn.target.plesk_server,
                    subscription.name,
                    app_pool_state,
                )
