"""Module for working with pre-migration checks performed by PMM agent"""

from xml.etree import ElementTree as et

from parallels.core.reports.model.issue import Issue
from parallels.core.reports.model.report import Report
from parallels.core.utils.common import open_no_inherit
from parallels.plesk.source.legacy import messages


def add_report_xml_issues(backup, xml_report_filename, report):
    """Add issues from pre-migration XML to pre-migration report
    
    Arguments:
    - backup - instance of backup object from
      parallels.core.dump.dump
      It gives information about objects (clients, resellers) that are 
      going to be migrated, considering their hierarchy and filtered
      by migration list to exclude object that we are not going to migrate
    - xml_report_filename - name of a file with pre-migration XML which
      is generated by migration agent's pre-migration script
    - report - parallels.core.checking.Report object - report to
      put issues to
    """

    with open_no_inherit(xml_report_filename) as fp:
        root_node = et.fromstring(fp.read())

    def convert_report_xml(root_node):
        report = Report(messages.REPORT_PRE_MIGRATION_ISSUES, '')
        convert_report_node(report, root_node)
        return report

    def convert_report_node(report, report_node):
        for issue_node in report_node.findall('issues/issue'):
            report.add_issue_obj(convert_issue_node(issue_node))

        for subreport_node in report_node.findall('children/report'):
            subreport = report.subtarget(
                subreport_node.attrib['type'], 
                subreport_node.attrib['name']
            )
            convert_report_node(subreport, subreport_node)

    def convert_issue_node(issue_node):
        return Issue(
            issue_node.findtext('type'),
            issue_node.findtext('severity'),
            issue_node.findtext('description'),
            issue_node.findtext('solution')
        )

    pre_migration_report = convert_report_xml(root_node)

    # overall issues
    for issue in pre_migration_report.issues:
        report.add_issue_obj(issue)

    # reseller issues
    for reseller in backup.iter_resellers():
        reseller_report = report.subtarget('Reseller', reseller.login)
        pre_migration_reseller_report = pre_migration_report.subtarget(
            'Reseller', reseller.login
        )
        for issue in pre_migration_reseller_report.issues:
            reseller_report.add_issue_obj(issue)

        # client issues
        for client in reseller.clients:
            client_report = reseller_report.subtarget(
                'Client', client.login
            )
            pre_migration_client_report = \
                pre_migration_reseller_report.subtarget(
                    'Client', client.login
                )

            for issue in pre_migration_client_report.issues:
                client_report.add_issue_obj(issue)

            # client child reports and their issues 
            for pre_migration_child_report in  pre_migration_client_report.children:
                child_report = client_report.subtarget(
                    pre_migration_child_report.type,
                    pre_migration_child_report.name
                )
                for issue in pre_migration_child_report.issues:
                    child_report.add_issue_obj(issue)
