from parallels.plesk.hosting_repository.utils.cli.base import BaseCli


class DomainCreateCli(BaseCli):
    def __init__(self, runner, domain_dump, subscription_name):
        """Construct command to create add-on domain in Plesk via CLI

        :type runner: parallels.core.utils.plesk_cli_runner.PleskCLIRunnerBase
        :type domain_dump: parallels.core.dump.data_model.AddonDomain
        :type subscription_name: str
        """
        super(DomainCreateCli, self).__init__(runner)
        self._domain_dump = domain_dump
        self._subscription_name = subscription_name

    @property
    def name(self):
        return 'domain'

    def _get_args(self):
        args = [
            '--create', self._domain_dump.name,
            '-webspace-name', self._subscription_name,
            '-do-not-apply-skeleton',
            '-skip-plan',
            '-notify', 'false'
        ]

        guid = self._domain_dump.guid
        if guid:
            args.extend(['-guid', guid])

        creation_date = self._domain_dump.creation_date
        if creation_date is not None:
            args.extend(['-creation-date', creation_date])

        return args


class SubdomainCreateCli(BaseCli):
    def __init__(self, runner, subdomain_dump, domain_name, subscription_name):
        """Construct command to create subdomain in Plesk via CLI

        :type runner: parallels.core.utils.plesk_cli_runner.PleskCLIRunnerBase
        :type subdomain_dump: parallels.core.dump.data_model.Subdomain
        :type domain_name: str
        :type subscription_name: str
        """
        super(SubdomainCreateCli, self).__init__(runner)
        self._subdomain_dump = subdomain_dump
        self._domain_name = domain_name
        self._subscription_name = subscription_name

    @property
    def name(self):
        return 'domain'

    def _get_args(self):
        args = [
            '--create', self._subdomain_dump.name,
            '-webspace-name', self._subscription_name,
            '-parent-domain-name', self._domain_name,
            '-do-not-apply-skeleton',
            '-skip-plan',
            '-notify', 'false'
        ]

        guid = self._subdomain_dump.guid
        if guid:
            args.extend(['-guid', guid])

        creation_date = self._subdomain_dump.creation_date
        if creation_date is not None:
            args.extend(['-creation-date', creation_date])

        return args


class DomainUpdateUsernameCli(BaseCli):
    def __init__(self, runner, subscription_name, username):
        super(DomainUpdateUsernameCli, self).__init__(runner)
        self._subscription_name = subscription_name
        self._username = username

    @property
    def name(self):
        return 'domain'

    def _get_args(self):
        return [
            '--update', self._subscription_name,
            '-login', self._username
        ]


class DomainSetForwarding(BaseCli):
    def __init__(self, runner, domain_name, url, is_frame_forwarding):
        super(DomainSetForwarding, self).__init__(runner)
        self._domain_name = domain_name
        self._url = url
        self._is_frame_forwarding = is_frame_forwarding

    @property
    def name(self):
        return 'domain'

    def _get_args(self):
        return [
            '--update', self._domain_name,
            '-hosting', 'true',
            '-hst_type', 'frm' if self._is_frame_forwarding else 'std',
            '-target_url', self._url,
            '-do-not-apply-skeleton'
        ]


class DomainSetPhysicalHosting(BaseCli):
    def __init__(self, runner, domain_name, document_root, is_enable_ssl):
        super(DomainSetPhysicalHosting, self).__init__(runner)
        self._domain_name = domain_name
        self._document_root = document_root
        self._is_enable_ssl = is_enable_ssl

    @property
    def name(self):
        return 'domain'

    def _get_args(self):
        args = []
        args.extend([
            '--update', self._domain_name,
            '-hosting', 'true',
            '-hst_type', 'phys',
            '-ssl', 'true' if self._is_enable_ssl else 'false',
            '-do-not-apply-skeleton'
        ])
        if self._document_root is not None:
            args.extend(['-www-root', self._document_root])
        return args
