from parallels.core.hosting_repository.subdomain import SubdomainModel, SubdomainEntity
from parallels.core.registry import Registry
from parallels.core.utils.mysql import escape_args_list
from parallels.plesk import messages
from parallels.plesk.hosting_repository.base import PleskBaseModel
from parallels.plesk.hosting_repository.utils.cli.domain import SubdomainCreateCli, DomainSetPhysicalHosting, \
    DomainSetForwarding
from parallels.plesk.hosting_repository.utils.cli.repair import ReapirSubdomainsSecurityCli
from parallels.plesk.hosting_repository.utils.db import db_query


class PleskSubdomainModel(SubdomainModel, PleskBaseModel):
    def get_list(self, filter_name=None):
        """Retrieve list of subdomains in target panel

        :type filter_name: list[str] | None
        :rtype: list[parallels.core.hosting_repository.subdomain.SubdomainEntity]
        """
        subdomains = []

        query = """
            SELECT
                domains.id AS subdomain_id,
                domains.name AS subdomain_name,
                parent_domain.name AS subscription_name
            FROM
                domains LEFT JOIN domains AS parent_domain ON parent_domain.id = domains.webspace_id
            WHERE
                domains.parentDomainId != 0
        """
        query_args = {}
        if filter_name is not None:
            filter_name_placeholders, filter_name_values = escape_args_list(filter_name, 'name')
            query += ' AND domains.name in ({filter_name_placeholders_str})'.format(
                filter_name_placeholders_str=', '.join(filter_name_placeholders)
            )
            # idn domains stored in database in idna encoding, so perform encoding of given filter
            query_args.update({key: value.encode('idna') for key, value in filter_name_values.iteritems()})

        for row in db_query(self.plesk_server, query, query_args):
            subdomains.append(SubdomainEntity(
                subdomain_id=row['subdomain_id'],
                name=row['subdomain_name'],
                subscription_name=row['subscription_name']
            ))

        return subdomains

    def create_from_dump(self, subdomain_dump, domain_name, subscription_name):
        """Create subdomain in target Plesk

        :type subdomain_dump: parallels.core.dump.data_model.Subdomain
        :type domain_name: str
        :type subscription_name: str
        """
        command = SubdomainCreateCli(self.plesk_cli_runner, subdomain_dump, domain_name, subscription_name)
        command.run()

    def set_forwarding(self, subdomain_name, forwarding_url, is_frame_forwarding):
        """Enable forwarding on subdomain with given name in target Plesk

        :type subdomain_name: str
        :type forwarding_url: str
        :type is_frame_forwarding: bool
        """
        command = DomainSetForwarding(self.plesk_cli_runner, subdomain_name, forwarding_url, is_frame_forwarding)
        command.run()

    def set_physical_hosting(self, subdomain_name, document_root, is_enable_ssl):
        """Enable physical hosting on subdomain with given name in target Plesk

        :type subdomain_name: str
        :type document_root: str
        :type is_enable_ssl: bool
        """
        command = DomainSetPhysicalHosting(self.plesk_cli_runner, subdomain_name, document_root, is_enable_ssl)
        command.run()

    def is_exists(self, subdomain_name, guid=None):
        """Check if subdomain with given name or guid exists in target Plesk

        :type subdomain_name: str
        :type guid: str | None
        :rtype: bool
        """
        if not guid:
            result = db_query(self.plesk_server, 'SELECT id FROM domains WHERE name = %(subdomain_name)s', dict(
                subdomain_name=subdomain_name.encode('idna')
            ))
        else:
            result = db_query(
                self.plesk_server, 'SELECT id FROM domains WHERE name = %(subdomain_name)s or guid = %(guid)s', dict(
                    subdomain_name=subdomain_name.encode('idna'),
                    guid=guid
                )
            )
        return len(result) > 0

    def update_security(self, subscription_name):
        """Actualize security metadata for subdomains of given subscription in target Plesk

        :type subscription_name: str
        """
        if not Registry.get_instance().get_context().conn.target.is_windows:
            raise NotImplementedError(messages.UNIX_SUBDOMAIN_UPDATE_SECURITY_METADATA_LOCATION_NOT_APPLICABLE)

        command = ReapirSubdomainsSecurityCli(self.plesk_cli_runner, subscription_name)
        command.run()
