from parallels.core.utils.mysql import escape_args_list
from parallels.plesk.hosting_repository.base import PleskBaseModel
from parallels.core.hosting_repository.domain import DomainModel, DomainEntity
from parallels.plesk.hosting_repository.utils.cli.domain import DomainCreateCli, DomainSetForwarding, \
    DomainSetPhysicalHosting
from parallels.plesk.hosting_repository.utils.db import db_query


class PleskDomainModel(DomainModel, PleskBaseModel):
    def get_list(self, filter_name=None):
        """Retrieve list of domains in target panel

        :type filter_name: list[str] | None
        :rtype: list[parallels.core.hosting_repository.domain.DomainEntity]
        """
        domains = []

        query = """
            SELECT
                domains.id AS domain_id,
                domains.name AS domain_name,
                parent_domain.name AS subscription_name
            FROM domains
            LEFT JOIN domains parent_domain
                ON parent_domain.id = domains.webspace_id
            WHERE
                domains.parentDomainId = 0 AND
                domains.webspace_id != 0
        """
        query_args = {}
        if filter_name is not None:
            filter_name_placeholders, filter_name_values = escape_args_list(filter_name, 'name')
            query += ' AND domains.name in ({filter_name_placeholders_str})'.format(
                filter_name_placeholders_str=', '.join(filter_name_placeholders)
            )
            # idn domains stored in database in idna encoding, so perform encoding of given filter
            query_args.update({key: value.encode('idna') for key, value in filter_name_values.iteritems()})

        for row in db_query(self.plesk_server, query, query_args):
            domains.append(DomainEntity(
                domain_id=row['domain_id'], name=row['domain_name'], subscription_name=row['subscription_name']
            ))

        return domains

    def create_from_dump(self, domain_dump, subscription_name):
        """Create add-on domain in target Plesk

        :type domain_dump: parallels.core.dump.data_model.AddonDomain
        :type subscription_name: str
        """
        command = DomainCreateCli(self.plesk_cli_runner, domain_dump, subscription_name)
        command.run()

    def set_forwarding(self, domain_name, forwarding_url, is_frame_forwarding):
        """Enable forwarding on domain with given name in target Plesk

        :type domain_name: str
        :type forwarding_url: str
        :type is_frame_forwarding: bool
        """
        command = DomainSetForwarding(self.plesk_cli_runner, domain_name, forwarding_url, is_frame_forwarding)
        command.run()

    def set_physical_hosting(self, domain_name, document_root, is_enable_ssl):
        """Enable physical hosting on domain with given name in target Plesk

        :type domain_name: str
        :type document_root: str
        :type is_enable_ssl: bool
        """
        command = DomainSetPhysicalHosting(self.plesk_cli_runner, domain_name, document_root, is_enable_ssl)
        command.run()

    def is_exists(self, domain_name, guid=None):
        """Check if domain with given name or guid exists in target Plesk

        :type domain_name: str
        :type guid: str | None
        :rtype: bool
        """
        if not guid:
            result = db_query(self.plesk_server, 'SELECT id FROM domains WHERE name = %(domain_name)s', dict(
                domain_name=domain_name.encode('idna')
            ))
        else:
            result = db_query(
                self.plesk_server, 'SELECT id FROM domains WHERE name = %(domain_name)s or guid = %(guid)s', dict(
                    domain_name=domain_name.encode('idna'),
                    guid=guid
                )
            )
        return len(result) > 0
