from parallels.plesk import messages
import logging

from parallels.core.actions.base.common_action import CommonAction
from parallels.core.utils import plesk_utils
from parallels.core.utils.yaml_utils import write_yaml, read_yaml

logger = logging.getLogger(__name__)


class SetApacheRestartInterval(CommonAction):
    def get_description(self):
        return messages.ACTION_SET_APACHE_RESTART_INTERVAL

    def get_failure_message(self, global_context):
        """
        :type global_context: parallels.core.global_context.GlobalMigrationContext
        """
        return messages.FAILED_SET_APACHE_RESTART_INTERVAL_VALUE % (
            global_context.conn.target.settings.apache_restart_interval
        )

    def is_critical(self):
        """If action is critical or not

        If action is critical and it failed, migration tool won't run the next operations.

        :rtype: bool
        """
        return False

    def filter_action(self, global_context):
        """Check whether we should run this action or not.

        Arguments:
        - global_context - registry with different objects that reused among different actions

        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :rtype: bool
        """
        return len(self._get_apache_servers(global_context)) > 0

    def run(self, global_context):
        """
        :type global_context: parallels.core.global_context.GlobalMigrationContext
        """
        if global_context.conn.target.settings.apache_restart_interval == 'none':
            return

        for target_server in self._get_apache_servers(global_context):
            self._set_apache_restart_interval(global_context, target_server)

    @staticmethod
    def _get_apache_servers(global_context):
        """
        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :rtype: set[parallels.core.connections.target_servers.TargetServer]
        """
        return {
            subscription.panel_target_server for subscription in global_context.iter_all_subscriptions()
            if not subscription.panel_target_server.is_windows()
        }

    @staticmethod
    def _set_apache_restart_interval(global_context, target_server):
        """
        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :type target_server: parallels.core.connections.target_servers.TargetServer
        """
        saved_restart_interval = read_yaml(
            global_context.session_files.get_path_to_apache_restart_interval(target_server.ip()),
            True,
            None
        )
        if saved_restart_interval is not None:
            logger.warning(
                messages.MIGRATION_TOOL_DETECTED_THAT_OLD_APACHE % saved_restart_interval
            )
        old_interval_value = saved_restart_interval
        logger.info(
            messages.SET_APACHE_RESTART_INTERVAL_S_SECONDS, global_context.conn.target.settings.apache_restart_interval
        )
        if old_interval_value is None:
            old_interval_value = plesk_utils.get_apache_restart_interval(target_server)
            write_yaml(
                global_context.session_files.get_path_to_apache_restart_interval(target_server.ip()),
                old_interval_value
            )
        plesk_utils.set_apache_restart_interval_value(
            target_server, global_context.conn.target.settings.apache_restart_interval
        )
