from parallels.hosting_check.signatures import TextSignature, RegexSignature
from parallels.hosting_check.utils.issue_types import IssueIdsSet

issue_ids = IssueIdsSet()


class WebIssueSignatureType(object):
    # ===== Signatures that are found on source server =======
    # that usually means that website did not work even before transfer

    # Critical error when executing PHP script which stopped script execution
    # Example: script was out of memory
    SOURCE_PHP_FATAL_ERROR = \
        issue_ids.unique('source_php_fatal_error')
    # Non-critical warning when executing PHP script. In that case script 
    # continue working
    SOURCE_PHP_WARNING = \
        issue_ids.unique('source_php_warning')
    # Non-critical notice when executing PHP script, usually a minor one.
    # In that case script continue working
    SOURCE_PHP_NOTICE = \
        issue_ids.unique('source_php_notice')
    # Error when parsing PHP script source file. Script was not even executed.
    SOURCE_PHP_PARSE_ERROR = \
        issue_ids.unique('source_php_parse_error')
    # Warning that proposes to change code for better compatibility with
    # the next versions of PHP
    SOURCE_PHP_STRICT_WARNING = \
        issue_ids.unique('source_strict_warning')
    # Remote MySQL server is not available; typical reasons: server is 
    # stopped, server hostname/IP is not correct, firewall blocks connections
    SOURCE_MYSQL_CANNOT_CONNECT_REMOTE = \
        issue_ids.unique('source_mysql_cannot_connect_remote')
    # Local MySQL server is not available; typical reason: server is stopped,
    # or server hostname/IP is not correct
    SOURCE_MYSQL_CANNOT_CONNECT_LOCAL = \
        issue_ids.unique('source_mysql_cannot_connect_local')
    # There is no such database on database server; possible reason - 
    # database was not created/migrated, or has been removed
    SOURCE_MYSQL_UNKNOWN_DATABASE = \
        issue_ids.unique('source_mysql_unknown_database')
    # Failed to log in to database server; usually that means that
    # credentials (login/password) are not correct, also can mean
    # that database does not allow remote access for that user
    SOURCE_MYSQL_USER_ACCESS_DENIED = \
        issue_ids.unique('source_mysql_user_access_denied')

    # ==== Signatures that are found only on target server =====
    # Signatures that were found only on target server, but not on source
    # server. That usually means that site get broken before migration

    # Critical error when executing PHP script which stopped script execution
    # Example: script was out of memory
    ONLY_TARGET_PHP_FATAL_ERROR = \
        issue_ids.unique('only_target_php_fatal_error')
    # Non-critical warning when executing PHP script. In that case script 
    # continue working
    ONLY_TARGET_PHP_WARNING = \
        issue_ids.unique('only_target_php_warning')
    # Non-critical notice when executing PHP script, usually a minor one.
    # In that case script continue working
    ONLY_TARGET_PHP_NOTICE = \
        issue_ids.unique('only_target_php_notice')
    # Error when parsing PHP script source file. Script was not even executed.
    ONLY_TARGET_PHP_PARSE_ERROR = \
        issue_ids.unique('only_target_php_parse_error')
    # Warning that proposes to change code for better compatibility with
    # the next versions of PHP
    ONLY_TARGET_PHP_STRICT_WARNING = \
        issue_ids.unique('only_target_strict_warning')
    # Remote MySQL server is not available; typical reasons: server is 
    # stopped, server hostname/IP is not correct, firewall blocks connections
    ONLY_TARGET_MYSQL_CANNOT_CONNECT_REMOTE = \
        issue_ids.unique('only_target_mysql_cannot_connect_remote')
    # Local MySQL server is not available; typical reason: server is stopped,
    # or server hostname/IP is not correct
    ONLY_TARGET_MYSQL_CANNOT_CONNECT_LOCAL = \
        issue_ids.unique('only_target_mysql_cannot_connect_local')
    # There is no such database on database server; possible reason - 
    # database was not created/migrated, or has been removed
    ONLY_TARGET_MYSQL_UNKNOWN_DATABASE = \
        issue_ids.unique('only_target_mysql_unknown_database')
    # Failed to log in to database server; usually that means that
    # credentials (login/password) are not correct, also can mean
    # that database does not allow remote access for that user
    ONLY_TARGET_MYSQL_USER_ACCESS_DENIED = \
        issue_ids.unique('only_target_mysql_user_access_denied')

WEB_SIGNATURES = [
    TextSignature(
        'PHP Fatal error:', 
        WebIssueSignatureType.SOURCE_PHP_FATAL_ERROR,
        WebIssueSignatureType.ONLY_TARGET_PHP_FATAL_ERROR
    ),
    TextSignature(
        'PHP Warning:', 
        WebIssueSignatureType.SOURCE_PHP_WARNING,
        WebIssueSignatureType.ONLY_TARGET_PHP_WARNING
    ),
    TextSignature(
        'PHP Notice:', 
        WebIssueSignatureType.SOURCE_PHP_NOTICE,
        WebIssueSignatureType.ONLY_TARGET_PHP_NOTICE
    ),
    TextSignature(
        'Parse error:', 
        WebIssueSignatureType.SOURCE_PHP_PARSE_ERROR,
        WebIssueSignatureType.ONLY_TARGET_PHP_PARSE_ERROR
    ),
    TextSignature(
        'strict warning:', 
        WebIssueSignatureType.SOURCE_PHP_STRICT_WARNING,
        WebIssueSignatureType.ONLY_TARGET_PHP_STRICT_WARNING
    ),
    RegexSignature(
        "Can't connect to MySQL server on '\S+?'", 
        WebIssueSignatureType.SOURCE_MYSQL_CANNOT_CONNECT_REMOTE,
        WebIssueSignatureType.ONLY_TARGET_MYSQL_CANNOT_CONNECT_REMOTE,
    ),
    RegexSignature(
        "Can't connect to local MySQL server through socket '.+?'", 
        WebIssueSignatureType.SOURCE_MYSQL_CANNOT_CONNECT_LOCAL,
        WebIssueSignatureType.ONLY_TARGET_MYSQL_CANNOT_CONNECT_LOCAL
    ),
    RegexSignature(
        "Unknown database '.+?'", 
        WebIssueSignatureType.SOURCE_MYSQL_UNKNOWN_DATABASE,
        WebIssueSignatureType.ONLY_TARGET_MYSQL_UNKNOWN_DATABASE
    ),
    RegexSignature(
        "Access denied for user '.+?'", 
        WebIssueSignatureType.SOURCE_MYSQL_USER_ACCESS_DENIED,
        WebIssueSignatureType.ONLY_TARGET_MYSQL_USER_ACCESS_DENIED
    )
]

