from parallels.core import messages
import posixpath
from pipes import quote
import re


def is_debian(runner):
    result = runner.run("/bin/sh", ["-c", '/usr/bin/test -f /etc/debian_version && echo "yes" || echo "no"']).strip()
    if result == "yes":
        return True
    elif result == "no":
        return False
    else:
        raise Exception(messages.FAILED_DETECT_OS_INVALID_OUTPUT % (result,))


def is_centos(runner):
    result = runner.run("/bin/sh", ["-c", '/usr/bin/test -f /etc/redhat-release && echo "yes" || echo "no"']).strip()
    if result == "yes":
        return True
    elif result == "no":
        return False
    else:
        raise Exception(messages.FAILED_DETECT_OS_INVALID_OUTPUT % (result,))


def is_suse(runner):
    result = runner.run("/bin/sh", ["-c", '/usr/bin/test -f /etc/SuSE-release && echo "yes" || echo "no"']).strip()
    if result == "yes":
        return True
    elif result == "no":
        return False
    else:
        raise Exception(messages.FAILED_DETECT_OS_INVALID_OUTPUT % (result,))


def map_copy_owner_user(runner_source, runner_target, source_basedir, target_basedir, owner_from, owner_to):
    """For each file on source node located in 'source_basedir' and owned by user 'owner_from' 
    change user to 'owner_to' on target node in 'target_basedir'"""
    _map_copy_owner(
        runner_source, runner_target, source_basedir, target_basedir, owner_from, owner_to, "user", "/bin/chown"
    )


def map_copy_owner_group(runner_source, runner_target, source_basedir, target_basedir, owner_from, owner_to):
    """For each file on source node located in 'source_basedir' and owned by group 'owner_from' 
    change group to 'owner_to' on target node in 'target_basedir'"""
    _map_copy_owner(
        runner_source, runner_target, source_basedir, target_basedir, owner_from, owner_to, "group", "/bin/chgrp"
    )


def _map_copy_owner(
    runner_source, runner_target, source_basedir, target_basedir, owner_from, owner_to, find_criteria, change_command
):
    filenames = [
        filename for filename in runner_source.run(
            '/usr/bin/find', [source_basedir, '-%s' % (find_criteria,), owner_from, '-printf', r'%P\n']
        ).splitlines() 
        if filename != ''
    ]
    for filename in filenames:
        target_filename = posixpath.join(target_basedir, filename)
        if file_exists(runner_target, target_filename):
            runner_target.run(
                change_command, [owner_to, target_filename]
            )


def format_command(template, *args, **kw):
    qargs = map(quote, args)
    qkw = dict((k, quote(unicode(v))) for k, v in kw.iteritems())
    return unicode(template).format(*qargs, **qkw)


def format_command_list(cmd, args):
    return " ".join([
        quote(arg if isinstance(arg, basestring) else str(arg))
        for arg in [cmd] + ([] if args is None else args)
    ])


def file_exists(runner, filepath):
    exit_code, _, _ = runner.run_unchecked('/usr/bin/test', ['-e', filepath])
    return exit_code == 0


def is_directory(runner, filepath):
    """Check whether provided path is a directory

    :type filepath: str | unicode
    :rtype: bool
    """
    exit_code, _, _ = runner.run_unchecked('/usr/bin/test', ['-d', filepath])
    return exit_code == 0


def get_files_list(runner, path):
    files = runner.sh('find {path} -maxdepth 1 -type f -printf "%f\\0"', dict(path=path)).split("\0")
    return [item for item in files if item]


def get_apache_httpd_conf(runner):
    try:
        # Detect apache httpd.conf. In all systems it can be in different places. Therefore, we use the setting
        # of running apache.
        apache_process = runner.sh(u'ps -ef | grep "apache2\|httpd" | grep -v grep').splitlines()[0]
        httpd = re.search(r'(([^\s]+\s*){8}([^\s]+).*)', apache_process).group(2)
        _server_config_file = runner.sh(u'{httpd_bin} -V | grep SERVER_CONFIG_FILE', dict(httpd_bin=httpd)).strip()
        server_config_file = re.search(r'(.*)SERVER_CONFIG_FILE=(.*)', _server_config_file).group(2).strip('"')
        _httpd_root = runner.sh(u'{httpd_bin} -V | grep HTTPD_ROOT', dict(httpd_bin=httpd)).strip()
        httpd_root = re.search(r'(.*)HTTPD_ROOT=(.*)', _httpd_root).group(2).strip('"')
        apache_httpd_conf = posixpath.join(httpd_root, server_config_file)
        if not runner.file_exists(apache_httpd_conf):
            return None
        return apache_httpd_conf
    except:
        return None


def get_apache_user(runner, apache_httpd_conf):
    try:
        if apache_httpd_conf is None:
                return None

        result = runner.sh(u"grep -i '^user\s' {httpd_conf}", dict(httpd_conf=apache_httpd_conf))
        user = re.search(r'[^\s]+\s*([^\s]+).*', result).group(1)
        exit_code, _, _ = runner.sh_unchecked(u"grep '^{user}:' /etc/passwd", dict(user=user))
        if exit_code != 0:
            return None
        return user
    except:
        return None


def get_apache_group(runner, apache_httpd_conf):
    try:
        if apache_httpd_conf is None:
                return None

        result = runner.sh(u"grep -i '^group\s' {httpd_conf}", dict(httpd_conf=apache_httpd_conf))
        group = re.search(r'[^\s]+\s*([^\s]+).*', result).group(1)
        exit_code, _, _ = runner.sh_unchecked(u"grep '^{group}:' /etc/group", dict(group=group))
        if exit_code != 0:
                return None
        return group
    except:
        return None
