import posixpath
from ftplib import FTP, error_perm

from parallels.core import messages

import logging

from parallels.core.registry import Registry
from parallels.core.utils.common import open_no_inherit

logger = logging.getLogger(__name__)


class Ftp(object):
    def __init__(self, host, username, password):
        logger.debug(messages.DEBUG_FTP_CONNECTION.format(host=host, username=username))
        self._ftp = FTP(host)
        self._ftp.login(username, password)

    def upload_file(self, source_path, target_path):
        """Upload given file from server where Plesk Migrator is running to remote server via FTP

        :type source_path: str
        :type target_path: str
        """
        logger.debug(messages.DEBUG_FTP_UPLOAD_FILE.format(source_path=source_path, target_path=target_path))
        with open_no_inherit(source_path, 'rb') as fp:
            self._ftp.storbinary('STOR %s' % target_path, fp)

    def download_file(self, source_path, target_server, target_path):
        logger.debug(messages.DEBUG_FTP_DOWNLOAD_FILE.format(source_path=source_path, target_path=target_path))
        try:
            def _write_target_file_content(content):
                with target_server.runner() as runner:
                    runner.append_file_content(target_path, content)

            # create empty file on target
            with target_server.runner() as runner:
                runner.create_file(target_path)
            # retrieve content via FTP and write in into just created file
            self._ftp.retrbinary('RETR %s' % source_path, _write_target_file_content)
        except error_perm as e:
            logger.debug(messages.DEBUG_FTP_DOWNLOAD_FILE_FAILED.format(
                source_path=source_path, target_path=target_path, message=e.message
            ))

    def download_file_local(self, source_path, target_path):
        target_server = Registry.get_instance().get_context().migrator_server
        self.download_file(source_path, target_server, target_path)

    def download_directory(self, source_path, target_server, target_path):
        """Retrieve given directory and all of its contents from remote server via FTP
        and store it in given location on target server

        :type source_path: str
        :type target_server: parallels.core.connections.target_servers.TargetServer
        :type target_path: str
        """
        logger.debug(messages.DEBUG_FTP_DOWNLOAD_DIR.format(
            source_host=self._ftp.host,
            target_host=target_server.description(),
            source_path=source_path,
            target_path=target_path
        ))
        self._process_download_path(source_path, target_server, target_path)

    def _process_download_path(self, source_path, target_server, target_path):
        try:
            self._ftp.cwd(source_path)
            # change directory command was executed successfully so given path is a directory
            # create corresponding directory on target
            with target_server.runner() as runner:
                runner.mkdir(target_path)
            # retrieve list of nested items (both files and directories) and process it
            for item in self._ftp.nlst():
                self._process_download_path(
                    posixpath.join(source_path, item),
                    target_server,
                    target_server.join_file_path(target_path, item)
                )
        except error_perm:
            # change directory command was failed with 5xx error code so consider that given path is a file
            self.download_file(source_path, target_server, target_path)
