import os
import posixpath
import ntpath

from parallels.core.utils.common import open_no_inherit


def iter_full_file_paths(directory):
    """Iterate over full paths of all files in a directory

    :type directory: str | unicode
    :rtype: list[str | unicode]
    """
    if isinstance(directory, unicode):
        # we work here with byte strings, as file names may contain not-unicode symbols
        directory = directory.encode('utf-8')

    for root, dirs, files in os.walk(directory):
        for name in files:
            full_path = os.path.join(root, name)
            if os.path.isfile(full_path):
                yield os.path.join(root, name)


def read_file_contents(filename):
    with open_no_inherit(filename, 'rb') as item_fp:
        return item_fp.read()


def files_have_equal_contents(filename1, filename2):
    """Check whether 2 files have equal content

    :type filename1: str | unicode
    :type filename2: str | unicode
    :rtype: bool
    """
    return read_file_contents(filename1) == read_file_contents(filename2)


def is_include_path(sub_path, root_path, server):
    """Check if root_path include sub_path; return True if yes

    :type sub_path: str
    :type root_path: str
    :type server: parallels.core.connections.server.Server
    :rtype: bool
    """
    sub_path_parts = _get_path_parts(sub_path, server)
    root_path_parts = _get_path_parts(root_path, server)
    if len(root_path_parts) > len(sub_path_parts):
        return False
    for i, item in enumerate(root_path_parts):
        if sub_path_parts[i] != item:
            return False
    return True


def get_relative_path(sub_path, root_path, server):
    """Retrieve path to sub_path related to root_path; return None if root_path not include sub_path

    :type sub_path: str
    :type root_path: str
    :type server: parallels.core.connections.server.Server
    :rtype: str | None
    """
    sub_path_parts = _get_path_parts(sub_path, server)
    root_path_parts = _get_path_parts(root_path, server)
    if len(root_path_parts) > len(sub_path_parts):
        return None
    related_path_parts = []
    for i, item in enumerate(sub_path_parts):
        if i < len(root_path_parts):
            if root_path_parts[i] != item:
                return None
            continue
        related_path_parts.append(item)
    if len(related_path_parts) == 0:
        return ''
    return ntpath.join(*related_path_parts) if server.is_windows() else posixpath.join(*related_path_parts)


def _get_path_parts(path, server):
    norm_path = ntpath.normpath(path) if server.is_windows() else posixpath.normpath(path)
    canonical_path = ntpath.normcase(norm_path) if server.is_windows() else posixpath.normcase(norm_path)
    parts = _split_path(canonical_path, server)
    return [x for x in parts if x != '.']


def _split_path(path, server):
    head, tail = ntpath.split(path) if server.is_windows() else posixpath.split(path)
    if tail:
        return _split_path(head, server) + [tail]
    return []
