from parallels.core.runners.base import BaseRunner
from parallels.core.runners.web.utils import WebAgentPool
from parallels.core.utils import unix_utils, windows_utils
from parallels.core.utils.common import cached


class WebAgentRunner(BaseRunner):

    def __init__(self, host, server_config):
        super(WebAgentRunner, self).__init__(None, host)
        self.agent = WebAgentPool.get_instance().get(server_config, allow_autodeploy=True)

    @property
    @cached
    def is_windows(self):
        return self.agent.is_windows()

    def file_exists(self, file_path):
        return self.agent.is_file_exists(file_path)

    def get_files_list(self, dir_path):
        return self.agent.get_files_list(dir_path)

    def mkdir(self, dir_path):
        self.agent.mkdir(dir_path)

    def get_file(self, remote_file_path, local_file_path):
        self.agent.download_file(remote_file_path, local_file_path)

    def get_file_contents(self, remote_filename):
        return self.agent.get_file_contents(remote_filename)

    def upload_file(self, local_file_path, remote_file_path):
        self.agent.upload_file(local_file_path, remote_file_path)

    def upload_file_content(self, file_path, content):
        self.agent.upload_file_content(file_path, content)

    def _sh_unchecked_no_logging(
            self, cmd_str, args=None, stdin_content=None, output_codepage=None,
            error_policy='strict', env=None, log_output=True, working_dir=None,
            redirect_output_file=None
    ):
        """Substitute format fields in 'cmd_str', then execute the command string. No logging is performed.

        :type cmd_str: str | unicode
        :type args: dict[str | unicode, str | unicode] | None
        :type stdin_content: str | unicode | None
        :type output_codepage: str | unicode | None
        :type error_policy: str | unicode
        :type env: dict[str | unicode]
        :type log_output: bool
        :type working_dir: str | unicode
        :type redirect_output_file: str | unicode | None
        :rtype: tuple
        """
        cmd_str = self._format_sh_command(cmd_str, args)
        return self.agent.sh_unchecked(cmd_str, stdin_content, env, log_output, working_dir, redirect_output_file)

    def _format_sh_command(self, cmd, args=None):
        """Format command arguments passed for 'sh' and 'sh_unchecked' methods into command line string

        This command line will be added to log and error messages, so it is expected that you could
        simply copy-paste it from logs or migrator output, and execute.

        :type cmd: str | unicode
        :type args: dict[str | unicode, str | unicode] | None
        :rtype: str | unicode
        """
        if args is not None and len(args) > 0:
            if self.is_windows:
                return windows_utils.format_command(cmd, **args)
            else:
                return unix_utils.format_command(cmd, **args)
        else:
            return cmd
