"""Various utility methods over runner objects"""
import os
import subprocess
import threading
from StringIO import StringIO

from parallels.core import messages
from parallels.core import popen_no_inherit
from parallels.core.runners.windows.agent import WindowsAgentRunner
from parallels.core.runners.windows.local import LocalWindowsRunner
from parallels.core.utils import windows_utils
from parallels.core.utils.common.logging import create_safe_logger
from parallels.core.utils.entity import Entity

logger = create_safe_logger(__name__)


def transfer_file(source_runner, source_filename, target_runner, target_filename):
    """Transfer file between 2 runner objects

    Now supports transfer only for WindowsAgentRunner (RPC agent) and LocalWindowsRunner

    :type source_runner: parallels.core.runners.base.BaseRunner
    :type target_runner: parallels.core.runners.base.BaseRunner
    :type source_filename: str | unicode
    :type target_filename: str | unicode
    """
    if isinstance(source_runner, WindowsAgentRunner) and isinstance(target_runner, WindowsAgentRunner):
        chunk_size = 1024 * 1024  # 1 MB
        target_runner.remote.create_file(target_filename)
        offset = 0
        while True:
            data = source_runner.remote.get_file_chunk(source_filename, offset, chunk_size)
            target_runner.remote.append_file(target_filename, data)
            offset += len(data)
            if len(data) < chunk_size:
                break
    elif isinstance(source_runner, WindowsAgentRunner) and isinstance(target_runner, LocalWindowsRunner):
        source_runner.get_file(source_filename, target_filename)
    elif isinstance(source_runner, LocalWindowsRunner) and isinstance(target_runner, WindowsAgentRunner):
        target_runner.upload_file(source_filename, target_filename)
    else:
        raise NotImplementedError()


def pipe_commands(source_runner, source_command, source_args, target_runner, target_command, target_args):
    """Run 2 commands on 2 servers, connecting stdout of one of them to stdin of the other one

    Source means source of stdout, target means destination stdin.
    '*_command' arguments are a format strings, '*_args' arguments are dictionaries with data to be
    substituted into format string, in the same way that is done when using "sh" and "sh_unchecked" methods of runner.

    Implemented only for piping from RPC agent runner to Windows local runner.

    :type source_runner: parallels.core.runners.base.BaseRunner
    :type source_command: str | unicode
    :type source_args: dict | None
    :type target_runner: parallels.core.runners.base.BaseRunner
    :type target_command: str | unicode
    :type target_args: dict | None
    :rtype: parallels.core.runners.utils.PipeCommandResult
    """
    if isinstance(source_runner, WindowsAgentRunner) and isinstance(target_runner, LocalWindowsRunner):
        target_command_str = windows_utils.format_command_dict(target_command, target_args)
        source_command_str = windows_utils.format_command_dict(source_command, source_args)

        logger.fdebug(
            messages.PIPE_COMMAND_REMOTE_TO_LOCAL_START,
            remote_server=source_runner.settings.ip,
            command_remote=source_command_str, command_local=target_command_str
        )

        target_command_env = os.environ.copy()
        # always execute commands with en_US locale, so we could rely on their output
        target_command_env['LANG'] = 'en_US.utf-8'

        popen_handle = popen_no_inherit(
            target_command_str,
            stdin=subprocess.PIPE,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            env=target_command_env,
        )

        source_stderr_data = StringIO()
        target_stderr_data = StringIO()

        def consume_source_stdout(data):
            popen_handle.stdin.write(data)

        def consume_source_stderr(data):
            source_stderr_data.write(data)

        def consume_target_stdout():
            while popen_handle.returncode is None:
                popen_handle.poll()
                popen_handle.stdout.read(1024 * 1024)

        def consume_target_stderr():
            while popen_handle.returncode is None:
                popen_handle.poll()
                stderr_data = popen_handle.stderr.read(1024 * 1024)
                if len(stderr_data) > 0:
                    target_stderr_data.write(stderr_data)

        result = PipeCommandResult()

        def finalize_source(return_code):
            popen_handle.stdin.close()
            result.source_exit_code = return_code

        thread_stdout = threading.Thread(target=consume_target_stdout)
        thread_stdout.start()
        thread_stderr = threading.Thread(target=consume_target_stderr)
        thread_stderr.start()

        source_runner.piped_sh_unchecked(
            source_command, source_args, consume_source_stdout, consume_source_stderr, finalize_source
        )
        thread_stdout.join()
        thread_stderr.join()

        stderr_last_data = popen_handle.stderr.read()
        if len(stderr_last_data) > 0:
            target_stderr_data.write(stderr_last_data)

        result.target_exit_code = popen_handle.returncode

        result.source_stderr = source_stderr_data.getvalue()
        source_stderr_data.close()
        result.target_stderr = target_stderr_data.getvalue()
        target_stderr_data.close()

        logger.fdebug(
            messages.PIPE_COMMAND_REMOTE_TO_LOCAL_FINISH,
            exit_code_local=result.target_exit_code,
            exit_code_remote=result.source_exit_code,
            stderr_local=result.target_stderr,
            stderr_remote=result.source_stderr,
        )

        return result
    else:
        raise NotImplementedError()


class PipeCommandResult(Entity):
    def __init__(self, source_exit_code=None, source_stderr=None, target_exit_code=None, target_stderr=None):
        self._source_exit_code = source_exit_code
        self._source_stderr = source_stderr
        self._target_exit_code = target_exit_code
        self._target_stderr = target_stderr

    @property
    def source_exit_code(self):
        """Exit code of the command on the source server

        :rtype: int | None
        """
        return self._source_exit_code

    @source_exit_code.setter
    def source_exit_code(self, value):
        """Set exit code on the source server

        :type value: int
        :rtype: None
        """
        self._source_exit_code = value

    @property
    def source_stderr(self):
        """stderr as string on the source server

        :rtype: str | unicode | None
        """
        return self._source_stderr

    @source_stderr.setter
    def source_stderr(self, value):
        """Set stderr on the source server

        :type value: str | unicode
        :rtype: None
        """
        self._source_stderr = value

    @property
    def target_exit_code(self):
        """Exit code of the command on the target server

        :rtype: int | None
        """
        return self._target_exit_code

    @target_exit_code.setter
    def target_exit_code(self, value):
        """Set exit code on the target server

        :type value: int
        :rtype: None
        """
        self._target_exit_code = value

    @property
    def target_stderr(self):
        """stderr as string on the target server

        :rtype: str | unicode | None
        """
        return self._target_stderr

    @target_stderr.setter
    def target_stderr(self, value):
        """Set stderr on the target server

        :type value: str | unicode
        :rtype: None
        """
        self._target_stderr = value
