import os
import shutil
import posixpath

from parallels.core import local_command_exec
from parallels.core.runners.unix.base import UnixRunner
from parallels.core.utils.common import mkdir_p, open_no_inherit
from parallels.core.utils.file_utils import iter_full_file_paths
from parallels.core.utils.migrator_utils import secure_write_open


class LocalUnixRunner(UnixRunner):
    """Execute commands on local server"""

    def __init__(self):
        super(LocalUnixRunner, self).__init__(host_description='the local server', hostname=None)

    @property
    def codepage(self):
        return 'utf-8'

    def _run_unchecked_no_logging(
        self, cmd, args=None, stdin_content=None, output_codepage=None, error_policy='strict', env=None
    ):
        command = [cmd] + ([] if args is None else args)
        codepage = output_codepage if output_codepage else self.codepage
        return local_command_exec(command, stdin_content, codepage, error_policy, env)

    def _sh_unchecked_no_logging(
        self, cmd_str, args=None, stdin_content=None, output_codepage=None,
        error_policy='strict', env=None, log_output=True, working_dir=None,
        redirect_output_file=None
    ):
        if redirect_output_file is not None:
            raise NotImplementedError()

        command_str = self._format_sh_command(cmd_str, args)
        command = ['/bin/sh', '-c', command_str]
        codepage = output_codepage if output_codepage else self.codepage
        return local_command_exec(command, stdin_content, codepage, error_policy, env, working_dir)

    def mkdir(self, dirname):
        mkdir_p(dirname)

    def create_file(self, filename):
        secure_write_open(filename)

    def append_file_content(self, filename, content):
        with secure_write_open(filename, False) as fp:
            fp.write(content)

    def upload_file_content(self, filename, content):
        with secure_write_open(filename) as fp:
            fp.write(content)

    def upload_file(self, local_filename, remote_filename):
        if local_filename != remote_filename:
            shutil.copy2(local_filename, remote_filename)

    def get_file(self, remote_filename, local_filename):
        if local_filename != remote_filename:
            shutil.copy2(remote_filename, local_filename)

    def get_file_contents(self, remote_filename):
        with open_no_inherit(remote_filename, "rb") as fp:
            return fp.read().decode('utf-8')

    def get_file_size(self, remote_filename):
        return os.path.getsize(remote_filename)

    def move(self, src_path, dst_path):
        shutil.move(src_path, dst_path)

    def get_files_list(self, path):
        return os.listdir(path)

    def iter_files_list_nested(self, path):
        """Iterate over all (including files in nested directories) file names inside a path"""
        return iter_full_file_paths(path)

    def remove_file(self, filename):
        if self.file_exists(filename):
            os.remove(filename)

    def remove_directory(self, directory, is_remove_root=True):
        for root, dirs, files in os.walk(directory):
            for name in files:
                self.remove_file(posixpath.join(directory, name))
            for name in dirs:
                self.remove_directory(posixpath.join(directory, name))
            if is_remove_root:
                os.rmdir(root)

    def check(self, server_description):
        # No need to check local execution, it should never fail
        pass

    def file_exists(self, filename):
        return os.path.exists(filename)
