import json
from collections import defaultdict
import itertools

from parallels.core.migration_list.entities.list_data import MigrationListData
from parallels.core.migration_list.entities.plans import MigrationListPlans
from parallels.core.migration_list.entities.subscription_info import SubscriptionMappingInfo
from parallels.core.migration_list.reader.base_reader import MigrationListReaderBase


class MigrationListReaderJSON(MigrationListReaderBase):
    def read(self, fileobj, subscriptions_only=False):
        """Read migration list from file in JSON format

        Returns tuple, where:
        1) The first element is object which describes migration list data:
        parallels.core.migration_list.entities.list_data.MigrationListData
        2) The second element is list of error messages occurred when reading migration list.

        :param fileobj: file-like object to read migration list from
        :param bool subscriptions_only: whether to include subscriptions only, or also read plans, resellers, etc
        :rtype: tuple(parallels.core.migration_list.entities.list_data.MigrationListData, list[str | unicode))
        """
        data = json.load(fileobj)

        subscriptions = {}

        for subscription_name, subscription in data.get("subscriptions", {}).iteritems():
            subscriptions[subscription_name] = SubscriptionMappingInfo(
                plan=subscription.get('plan') if not subscriptions_only else None,
                addon_plans=subscription.get('addon_plans') if not subscriptions_only else [],
                owner=subscription.get('owner') if not subscriptions_only else None,
                ipv4=subscription.get('ipv4'),
                ipv6=subscription.get('ipv6')
            )

        resellers = {}
        if not subscriptions_only:
            for reseller_login, reseller_info in data.get("resellers", {}).iteritems():
                resellers[reseller_login] = reseller_info.get("plan")

        customers = {}
        if not subscriptions_only:
            for customer_login, customer_info in data.get("customers", {}).iteritems():
                customers[customer_login] = customer_info.get("owner")

        plans = defaultdict(set)
        if not subscriptions_only:
            for plan in itertools.chain(data.get("plans", []), data.get("addon_plans", [])):
                plans[plan.get("owner")].add(plan["name"])

        migration_list_data = MigrationListData(subscriptions, resellers, customers, plans)
        return migration_list_data, []

    def read_resellers(self, fileobj):
        """Read resellers from migration list file in JSON format

        Returns tuple, where:
        1) The first element is a dictionary, describing resellers:
        {reseller login: reseller plan}.
        If plan is not set and reseller subscription is custom, then reseller plan is None.
        2) The second element is list of error messages occurred when reading migration list.

        :param fileobj: file-like object to read migration list from
        :rtype: tuple(dict[basestring, basestring], list[str | unicode])
        """
        data = json.load(fileobj)
        resellers = {}
        for reseller_login, reseller_info in data.get("resellers", {}).iteritems():
            resellers[reseller_login] = reseller_info.get("plan")

        return resellers, []

    @classmethod
    def read_plans(cls, fileobj):
        """Read plans from migration list file in JSON format

        Returns tuple, where:
        1) The first element is object which describes migration list plans:
        parallels.core.migration_list.entities.plans.MigrationListPlans
        2) The second element is list of error messages occurred when reading migration list.

        :param fileobj: file-like object to read migration list from
        :rtype: tuple(parallels.core.migration_list.entities.plans.MigrationListPlans, list[str | unicode])
        """
        data = json.load(fileobj)

        plans = defaultdict(set)
        for plan in data.get("plans", []):
            plans[plan.get("owner")].add(plan["name"])

        addon_plans = defaultdict(set)
        for addon_plan in data.get("addon_plans", []):
            addon_plans[addon_plan.get("owner")].add(addon_plan["name"])

        plans = MigrationListPlans(plans, addon_plans)

        return plans, []
