from parallels.core import messages
from parallels.core.application_adjuster.application_configs.base import ApplicationConfig
from parallels.core.application_adjuster.application_configs.utils import split_by_start_end
from parallels.core.utils.common import safe_format
from parallels.core.utils.common.ip import resolve_all_safe
from parallels.core.utils.database_utils import join_mssql_hostname_and_instance, split_mssql_host_parts_str
from parallels.core.utils.line_processor import LineProcessor, ReplaceResults, ChangedLineInfo


class WindowsWebConfig(ApplicationConfig):
    """Class for detection and fixing web.config files of ASP.NET applications"""

    def get_config_file_description(self, filename):
        return safe_format(
            messages.WINDOWS_WEB_CONFIG_FILE_DESCRIPTION,
            filename=filename
        )

    def filename_match(self, path):
        """Check if given path looks like web.config file of ASP.NET application

        :type path: str | unicode
        :rtype: boolean
        """
        # Match all "*.config" files, as configuration may be located in different files, which are included
        # from main web.config file
        return path.lower().endswith('.config')

    def contents_match(self, contents):
        """Check if given file contents looks like web.config file of ASP.NET application

        :type contents: str | unicode
        :rtype: boolean
        """
        contents_lower = contents.lower()
        return 'connectionstring' in contents_lower or 'sqlserver' in contents_lower

    def fix_database_hostname(self, contents, db_host_mapping):
        """Fix database hostname in web.config file of ASP.NET application

        :type contents: str | unicode
        :type db_host_mapping: dict[str | unicode, str | unicode]
        :rtype: parallels.core.utils.line_processor.ReplaceResults
        """
        processor = LineProcessor(contents)
        changed_lines = []

        for line in processor.iter_lines():
            line_contents_lower = line.contents.lower()
            if 'connectionstring' in line_contents_lower or 'sqlserver' in line_contents_lower:
                split_result = self._split_hostname(line.contents)
                if split_result is None:
                    continue

                before, hostname, after = split_result

                if hostname in db_host_mapping and db_host_mapping[hostname] != hostname:
                    old_line_contents = line.contents
                    line.contents = before + db_host_mapping[hostname] + after
                    changed_lines.append(ChangedLineInfo(old_line_contents, line.contents, line.number))
                else:
                    host, instance, port = split_mssql_host_parts_str(hostname)
                    ips = resolve_all_safe(host)
                    for ip in ips:
                        full_name = join_mssql_hostname_and_instance(ip, instance, port)
                        if full_name in db_host_mapping and db_host_mapping[full_name] != hostname:
                            old_line_contents = line.contents
                            line.contents = before + db_host_mapping[full_name] + after
                            changed_lines.append(ChangedLineInfo(old_line_contents, line.contents, line.number))

        return ReplaceResults(processor.serialize(), changed_lines)

    @staticmethod
    def _split_hostname(line):
        """Split line into 3 parts: before database hostname, database hostname, after database hostname

        For example, if configuration file line is:
        connectionString="Server=192.168.1.1\MSSQL2005;Database=DotNetNuke_1;uid=DotNetNuke_e;pwd=123qwe;"
        then this function will return
        ('connectionString="Server=', '192.168.1.1\MSSQL2005', ';Database=DotNetNuke_1;uid=DotNetNuke_e;pwd=123qwe;')

        This function considers that database hostname could be specified by different parameters:
        "Data Source", "Server", and so on.

        :type line: str | unicode
        :rtype: tuple[str | unicode]
        """
        param_names = ["Data Source", "Server", "Address", "Addr", "Network Address"]
        for param_name in param_names:
            split_results = split_by_start_end(start_str='%s=' % param_name, end_str=';', data_str=line)
            if split_results is not None:
                return split_results

        return None