from parallels.core.utils.common.ip import resolve_all_safe
from parallels.core.utils.line_processor import LineProcessor, ChangedLineInfo, ReplaceResults


def path_endswith(path, suffix):
    """Check whether the last part of the path matches specified suffix

    Works both for Windows and for Linux. It is considered that suffix contains entire item
    (filename or directory) - for example:
    path_endswith('/var/www/vhosts/mysite.tld/wordpress/wp-config.php', 'wordpress/wp-config.php') will match
    path_endswith('/var/www/vhosts/mysite.tld/mywordpress/wp-config.php', 'wordpress/wp-config.php') won't match

    :type path: str | unicode
    :type suffix: str | unicode
    :rtype: boolean
    """
    return path.replace('\\', '/').lower().endswith('/%s' % suffix.lower())


def replace_line(contents, condition_str, start_str, end_str, db_host_mapping):
    """Replace database host in a line

    Algorithm is the following:
    for each line in contents, if line contains condition_str:
    1) if some source database host from database host mapping is in line - perform simple replace
    2) otherwise, consider everything between start_str and end_str as hostname, try to resolve it
    and then replace (if possible) according to database host mapping

    :type contents: str | unicode
    :type start_str: str | unicode
    :type end_str: str | unicode
    :type db_host_mapping: dict[str | unicode, str | unicode]
    :rtype: parallels.core.utils.line_processor.ReplaceResults
    """
    changed_lines = []
    processor = LineProcessor(contents)
    for line in processor.iter_lines():
        if condition_str in line.contents:
            old_line_contents = line.contents
            replaced = False
            for source_db, target_db in db_host_mapping.iteritems():
                if source_db in line.contents:
                    # exact match for source database host
                    line.contents = line.contents.replace(source_db, target_db)
                    replaced = True

            if not replaced:
                # no exact match, try to find hostname in a line, resolve it and check if it is
                # in database host mapping
                split_result = split_by_start_end(start_str, end_str, line.contents)
                if split_result is None:
                    continue

                before, hostname, after = split_result
                ips = resolve_all_safe(hostname)
                for ip in ips:
                    if ip in db_host_mapping and db_host_mapping[ip] != hostname:
                        line.contents = before + db_host_mapping[ip] + after

            if line.contents != old_line_contents:
                changed_lines.append(ChangedLineInfo(old_line_contents, line.contents, line.number))

    return ReplaceResults(processor.serialize(), changed_lines)


def split_by_start_end(start_str, end_str, data_str):
    """Split line into 3 parts: before start string, between start and end, after end string

    For example, if string is "aaaBEGINbbbENDccc", this function will return
    ('aaaBEGIN', 'bbb', 'ENDccc')
    If start string or end strings are not found, none is returned

    :type start_str: str | unicode
    :type end_str: str | unicode
    :type data_str: str | unicode
    :rtype: tuple[str | unicode] | None
    """
    if start_str in data_str:
        start_pos = data_str.find(start_str) + len(start_str)
        if end_str in data_str[start_pos + 1:]:
            end_pos = data_str[start_pos + 1:].find(end_str) + start_pos + 1
            return data_str[:start_pos], data_str[start_pos:end_pos], data_str[end_pos:]

    return None