import os
import shutil

from parallels.core import messages
from parallels.core.actions.base.common_action import CommonAction
from parallels.core.registry import Registry
from parallels.core.utils.common.logging import create_safe_logger
from parallels.core.utils.common_constants import LOCK_RUN_QUEUE
from parallels.core.utils.locks.session_file_lock import SessionFileLock
from parallels.core.utils.stop_mark import StopMark

logger = create_safe_logger(__name__)


class StopMigrationAction(CommonAction):
    """Stop all migrations running within current session

    This action:
    1) Stops currently running migration (commands like "transfer-accounts", "copy-content", etc).
    2) Stops queue runner ("run-queue" command).
    3) Removes all queued subscriptions from the queue.

    This action works with the help of stop file and locks.
    By stop file we notify commands that they should terminate.
    By acquiring locks we ensure that commands were terminated.
    """
    def get_description(self):
        """Get short description of action as string

        :rtype: str
        """
        return messages.ACTION_STOP_MIGRATION_DESCRIPTION

    def get_failure_message(self, global_context):
        """Get message for situation when action failed

        :type global_context: parallels.core.global_context.GlobalMigrationContext
        """
        return messages.ACTION_STOP_MIGRATION_FAILURE

    def run(self, global_context):
        """Run action

        :type global_context: parallels.core.global_context.GlobalMigrationContext
        """
        StopMark.set()
        run_queue_lock = SessionFileLock(LOCK_RUN_QUEUE)
        run_queue_lock.acquire_block()
        try:
            migration_lock = Registry.get_instance().get_migration_lock()
            migration_lock.acquire_block()
            try:
                StopMark.remove()
                tasks_dir = global_context.session_files.get_queue_tasks_dir()
                if os.path.isdir(tasks_dir):
                    shutil.rmtree(tasks_dir)
                global_context.subscriptions_status.set_all_not_queued()
            finally:
                migration_lock.release()
        finally:
            run_queue_lock.release()