import os

import shutil

from parallels.core import messages
from parallels.core.actions.base.common_action import CommonAction
from parallels.core.utils.common.logging import create_safe_logger
from parallels.core.utils.common_constants import LOCK_QUEUE
from parallels.core.utils.json_utils import read_json, write_json
from parallels.core.utils.locks.session_file_lock import SessionFileLock

logger = create_safe_logger(__name__)


class RemoveQueueSubscriptionsAction(CommonAction):
    """Remove specified subscriptions from queue, request stop of operation on this subscription if running"""
    def get_description(self):
        """Get short description of action as string

        :rtype: str
        """
        return messages.ACTION_REMOVE_QUEUE_SUBSCRIPTIONS_DESCRIPTION

    def get_failure_message(self, global_context):
        """Get message for situation when action failed

        :type global_context: parallels.core.global_context.GlobalMigrationContext
        """
        return messages.ACTION_REMOVE_QUEUE_SUBSCRIPTIONS_FAILURE

    def run(self, global_context):
        """Run action

        :type global_context: parallels.core.global_context.GlobalMigrationContext
        """
        subscriptions = set(global_context.migration_list_data.subscriptions_mapping.keys())

        queue_lock = SessionFileLock(LOCK_QUEUE)
        queue_lock.acquire_block()

        try:
            tasks_dir = global_context.session_files.get_queue_tasks_dir()
            if not os.path.exists(tasks_dir):
                return

            running_task_id = global_context.session_files.get_running_task_file().read()
            tasks = os.listdir(tasks_dir)
            for task_id in tasks:
                if task_id == running_task_id:
                    continue

                task_dir = os.path.join(tasks_dir, task_id)
                migration_list_file = os.path.join(task_dir, 'migration-list.json')
                if os.path.exists(migration_list_file):
                    remove_task = self._exclude_subscriptions_from_migration_list(
                        migration_list_file, subscriptions
                    )
                    if remove_task:
                        shutil.rmtree(task_dir)

            for subscription in subscriptions:
                global_context.subscriptions_status.remove_queued(subscription)
                global_context.subscriptions_status.set_stop(subscription)
        finally:
            queue_lock.release()

    @staticmethod
    def _exclude_subscriptions_from_migration_list(migration_list_file, subscriptions):
        """Exclude subscriptions from migration list file.

        Return True if migration list becomes unnecessary and task can be completely removed from queue,
        False otherwise.

        :type migration_list_file: str | unicode
        :type subscriptions: set[str | unicode]
        :rtype: bool
        """
        data = read_json(migration_list_file)
        if 'subscriptions' not in data:
            return True

        for subscription in subscriptions:
            if subscription in data['subscriptions']:
                del data['subscriptions'][subscription]

        write_json(migration_list_file, data)

        return len(data['subscriptions']) == 0
