import os
import shutil
import time

from parallels.core import messages
from parallels.core.actions.base.common_action import CommonAction
from parallels.core.migration_list.reader.json_reader import MigrationListReaderJSON
from parallels.core.utils.common import mkdir_p, unused, open_no_inherit
from parallels.core.utils.common.logging import create_safe_logger
from parallels.core.utils.common_constants import LOCK_QUEUE
from parallels.core.utils.json_utils import read_json
from parallels.core.utils.locks.session_file_lock import SessionFileLock
from parallels.core.utils.migration_progress import SubscriptionMigrationStatus
from parallels.core.utils.migrator_utils import get_optional_option_value
from parallels.core.utils.subscription_operations import command_name_to_operation

logger = create_safe_logger(__name__)


class AddQueueTaskAction(CommonAction):
    """Add task to a queue

    Task consists of 2 parts:
    1) Command to be executed, for example 'transfer-accounts' or 'copy-web-content'.
    Command is stored as JSON file, which contains command to be executed and list of additional arguments.
    2) List of subscriptions on which the command should be executed.
    Format of the file is JSON migration list.

    This command adds a new task to the queue. Required arguments are file with command description (JSON) and
    migration list file (JSON).
    """
    def get_description(self):
        """Get short description of action as string

        :rtype: str
        """
        return messages.ACTION_ADD_QUEUE_TASK_DESCRIPTION

    def get_failure_message(self, global_context):
        """Get message for situation when action failed

        :type global_context: parallels.core.global_context.GlobalMigrationContext
        """
        return messages.ACTION_ADD_QUEUE_TASK_FAILURE

    def run(self, global_context):
        """Run action

        :type global_context: parallels.core.global_context.GlobalMigrationContext
        """
        queue_lock = SessionFileLock(LOCK_QUEUE)
        queue_lock.acquire_block()

        try:
            tasks_dir = global_context.session_files.get_queue_tasks_dir()
            mkdir_p(tasks_dir)

            task_id = get_optional_option_value(global_context.options, 'task_id', time.strftime('%Y-%m-%d-%H-%M-%S'))
            task_dir = os.path.join(tasks_dir, task_id)
            mkdir_p(task_dir)

            task_command_file = os.path.join(task_dir, 'command.json')

            shutil.copyfile(global_context.options.command_file, task_command_file)
            if global_context.options.migration_list_file is not None:
                shutil.copyfile(
                    global_context.options.migration_list_file,
                    os.path.join(task_dir, 'migration-list.json')
                )

                command_info = read_json(global_context.options.command_file)
                command_name = command_info['command']

                # mark subscriptions from migration list as queued
                migration_list_reader = MigrationListReaderJSON(
                    global_context.migrator.get_migration_list_source_data()
                )
                with open_no_inherit(global_context.options.migration_list_file, 'r') as fp:
                    migration_list_data, errors = migration_list_reader.read(fp, subscriptions_only=True)
                    # migration list errors are simply ignored here,
                    # anyway you will see them once the task is executed
                    unused(errors)
                    for subscription_name in migration_list_data.subscriptions_mapping.iterkeys():
                        operation = command_name_to_operation(command_name)
                        if operation is not None:
                            global_context.subscriptions_status.set_operation_status(
                                subscription_name, operation, SubscriptionMigrationStatus.ON_HOLD
                            )
                        global_context.subscriptions_status.set_queued(subscription_name)
                        global_context.subscriptions_status.remove_stop(subscription_name)
        finally:
            queue_lock.release()
