import logging

from parallels.common import migrator_config
from parallels.common.actions.base.subscription_action import SubscriptionAction
from parallels.common.actions.utils.logging_properties import LoggingProperties
from parallels.common.checking import Report, Problem, PlainReport
from parallels.common.migrator_config import ActionRunnerType
from parallels.common.utils import migrator_utils

logger = logging.getLogger(__name__)


class PrintSubscriptionStatus(SubscriptionAction):
	def get_description(self):
		return "Print subscription migration status"

	def get_failure_message(self, global_context, subscription):
		"""
		:type global_context: parallels.common.global_context.GlobalMigrationContext
		:type subscription: parallels.common.migrated_subscription.MigratedSubscription
		"""
		return "Failed to print subscription migration status"

	def get_logging_properties(self):
		"""Get how action should be logged to migration tools end-user

		:rtype: parallels.common.actions.utils.logging_properties.LoggingProperties
		"""
		return LoggingProperties(info_log=False)

	def filter_subscription(self, global_context, subscription):
		"""
		:type global_context: parallels.common.global_context.GlobalMigrationContext
		:type subscription: parallels.common.migrated_subscription.MigratedSubscription
		"""
		action_runner_type = migrator_config.get_action_runner_type(global_context.config)
		return action_runner_type == ActionRunnerType.BY_SUBSCRIPTION

	def is_critical(self):
		"""If action is critical or not

		If action is critical and it failed for a subscription, migration tool
		won't run the next operations for the subscription.

		:rtype: bool
		"""
		return False

	def run(self, global_context, subscription):
		"""
		:type global_context: parallels.common.global_context.GlobalMigrationContext
		:type subscription: parallels.common.migrated_subscription.MigratedSubscription
		"""
		safe = global_context.safe
		subscription_report = Report('Subscription', subscription.name)

		# add issues from 'Safe'
		if subscription.name in safe.failed_objects.subscriptions:
			global_context.migrator._add_report_issues(
				subscription_report,
				safe.failed_objects.subscriptions[subscription.name]
			)
		for issue in safe.issues.subscriptions[subscription.name]:
			subscription_report.add_issue_obj(issue)

		# add issues from pre-migration checks
		plain_pre_check_report = PlainReport(
			global_context.pre_check_report, *global_context.migrator._extract_source_objects_info()
		)
		subscription_pre_check_report = plain_pre_check_report.get_subscription_report(
			subscription.model.source, subscription.name
		)
		for issue in subscription_pre_check_report.issues:
			subscription_report.add_issue_obj(issue)

		# if report has no issues - add successful status
		if not subscription_report.has_issues():
			subscription_report.add_issue(
				Problem(
					u'subscription_migr_success', Problem.INFO,
					u'Subscription migrated successfully'
				),
				u""
			)

		# print report
		report_str = migrator_utils.format_report(
			subscription_report, None, None, show_no_issue_branches=False
		)
		logger.info("Subscription migration finished, here is a status:\n" + report_str)