"""Global configuration of hosting checkers"""

from parallels.utils.ip import is_ipv4
from parallels.common.utils.config_utils import ConfigSection
from parallels.common import MigrationError

class HostingCheckersConfig(object):
	"""Hosting checkers global configurations

	Properties:

	'messages_delta_limit': The post-migration mail checks compare the number
	of messages in the source and the target mailboxes. If the number of
	messages differs more than it is specified in "messages_delta_limit", it is
	considered an error.

	'external_dns_servers': List of IP v4 addresses of external DNS servers
	that allows recursive DNS queries.  Hosting checks use the specified
	addresses to test whether general Internet users can obtain the correct DNS
	records for the transferred domains.

	'save_external_report_data_function': Function is used to store additional
	data that is too long for report.  It accepts two arguments - base name of
	a file, and file contents. It should store the contents to a file in some
	session directory and return path to the file (the returned path will be
	printed to report). 	

	'dns_server_name': Name of target DNS server in report and log messages

	'panel_file_id': Name of a target panel in external data. Used to create
	names for files used by save_external_report_data_function function
	"""
	def __init__(
		self, messages_delta_limit, external_dns_servers,
		save_external_report_data_function,
		dns_server_name, panel_file_id
	):
		self.messages_delta_limit = messages_delta_limit
		self.external_dns_servers = external_dns_servers
		self.save_external_report_data_function = \
			save_external_report_data_function
		self.dns_server_name = dns_server_name
		self.panel_file_id = panel_file_id

class MigratorHostingCheckersConfig(HostingCheckersConfig):
	"""Implementation of hosting checkers configratuon used by migrator"""

	def __init__(self, config, save_external_report_data_function):
		self.config = config
		super(MigratorHostingCheckersConfig, self).__init__(
			messages_delta_limit=self._get_messages_delta_limit(),
			external_dns_servers=self._get_external_dns_servers(),
			save_external_report_data_function=\
				save_external_report_data_function,
			dns_server_name='target DNS server',
			panel_file_id='target'
		)

	def _get_messages_delta_limit(self):
		global_section = ConfigSection(self.config, 'GLOBAL')
		return global_section.get('mail-messages-delta-limit', 10)

	def _get_external_dns_servers(self):
		external_dns_servers = []
		global_section = ConfigSection(self.config, 'GLOBAL')
		external_dns_server_ip = global_section.get(
			'external-dns-server', '8.8.8.8'
		)
		if external_dns_server_ip == 'none':
			# 'none' is a special value saying migrator to skip external DNS
			# tests
			pass 
		elif is_ipv4(external_dns_server_ip):
			external_dns_servers.append(external_dns_server_ip)
		else:
			raise MigrationError(
				"'external-dns-servers' option in [GLOBAL] section of config "
				"file of migration tool contains invalid value: '%s'. Valid IPv4 "
				"address should be specified in order for all DNS checks to be "
				"performed." % (
					external_dns_server_ip
				)
			)
		return external_dns_servers
