import logging
import threading
import Queue

from parallels.common.actions.base.action_pointer import ActionPointer
from parallels.common.actions.base.common_action import CommonAction
from parallels.common.actions.base.compound_action import CompoundAction
from parallels.common.actions.base.condition_action_pointer import ConditionActionPointer
from parallels.common.actions.base.subscription_action import SubscriptionAction
from parallels.common.logging_context import log_context

logger = logging.getLogger(__name__)


class ParallelActionRunner(object):
	def __init__(self, global_context):
		self._global_context = global_context

	def run_entry_point(self, entry_point):
		self.run(entry_point)

	def run(self, action, action_id=None):
		blocks = self._split_actions_by_blocks(self._flat_action(action))
		for block in blocks:
			if isinstance(block, CommonActionBlock):
				self._run_common_action_block(block)
			elif isinstance(block, SubscriptionActionBlock):
				self._run_subscription_action_block(block)
			else:
				assert False

	def _run_common_action_block(self, block):
		logger.info("ENTER COMMON ACTION BLOCK")
		for action in block.actions:
			action.run(self._global_context)
		logger.info("EXIT COMMON ACTION BLOCK")

	def _run_subscription_action_block(self, block):
		logger.info("ENTER SUBSCRIPTION ACTION BLOCK")

		actions = block.actions
		locks = {}

		for action in actions:
			if not action.get_multithreading_properties().can_use_threads:
				locks[action] = threading.Lock()

		subscription_queue = Queue.Queue()

		def worker():
			while True:
				try:
					subscription = subscription_queue.get(False)
				except Queue.Empty:
					return  # no more subscriptions in a queue

				logger.info("START Processing subscription '%s'", subscription.name)
				with log_context(subscription.name):
					for action in actions:
						logger.info("START: %s", action.get_description())
						if action in locks:
							locks[action].acquire()

						if action.filter_subscription(self._global_context, subscription):
							action.run(self._global_context, subscription)

						if action in locks:
							locks[action].release()
						logger.info("END: %s", action.get_description())
				logger.info("END Processing subscription '%s'", subscription.name)
				subscription_queue.task_done()

		for subscription in self._global_context.iter_all_subscriptions():
			subscription_queue.put(subscription)

		for i in range(0, 5):
			thread = threading.Thread(target=worker)
			thread.start()

		subscription_queue.join()

		logger.info("EXIT SUBSCRIPTION ACTION BLOCK")

	def _flat_action(self, action):
		if isinstance(action, ActionPointer):
			action = action.resolve()
		elif isinstance(action, ConditionActionPointer):
			action = action.resolve(self._global_context)

		if isinstance(action, CompoundAction):
			return sum([
				self._flat_action(child_action)
				for _, child_action in action.get_all_actions()
			], [])
		else:
			return [action]

	@staticmethod
	def _split_actions_by_blocks(actions):
		previous_action_type = None
		blocks = []
		current_block = []
		for action in actions:
			if isinstance(action, SubscriptionAction):
				action_type = 'subscription'
			elif isinstance(action, CommonAction):
				action_type = 'common'
			else:
				assert False

			if action_type == previous_action_type:
				current_block.actions.append(action)
			else:  # start new block
				if isinstance(action, SubscriptionAction):
					current_block = SubscriptionActionBlock()
				elif isinstance(action, CommonAction):
					current_block = CommonActionBlock()
				else:
					assert False
				current_block.actions.append(action)
				blocks.append(current_block)

			previous_action_type = action_type
		return blocks


class BaseActionBlock(object):
	def __init__(self, actions=None):
		if actions is None:
			actions = []
		self.actions = actions

	def __eq__(self, other):
		return other.actions == self.actions


class SubscriptionActionBlock(BaseActionBlock):
	pass


class CommonActionBlock(BaseActionBlock):
	pass