import logging
from xml.etree import ElementTree

logger = logging.getLogger(__name__)


class RestoreResult(object):
	"""Object that represents Plesk restore result XML"""

	def __init__(self, domain_name, plesk_restore_stdout):
		self._response_xml = self._extract_xml_from_stdout(
			plesk_restore_stdout
		)
		self._domain_name = domain_name

	@staticmethod
	def _extract_xml_from_stdout(plesk_restore_stdout):
		"""Get XML out of Plesk restore stdout.

		Returns:
			XML object, if response contains one. Otherwise return None.
		"""
		start_xml_declaration = '<?xml version="1.0" encoding="UTF-8"?>'
		start_xml_declaration_pos = plesk_restore_stdout.find(start_xml_declaration)
		end_tag = "</restore>"
		end_tag_pos = plesk_restore_stdout.find(end_tag)
		if start_xml_declaration_pos != -1 and end_tag_pos != -1:
			xml_str_content = plesk_restore_stdout[
				start_xml_declaration_pos:end_tag_pos+len(end_tag)
			].strip()
			try:
				return ElementTree.fromstring(xml_str_content)
			except Exception as e:
				logger.debug(u"Error while parsing Plesk restore XML", exc_info=e)
				return None
		else:
			return None

	def has_errors(self):
		"""Check if there were restoration error (some known errors are ignored)
		
		Args:
			response: stdout of 'pleskrestore' command
			domain_name: the name of the domain specified in arguments for
			'pleskrestore'
		Returns:
			'True', if errors are found, 'False' otherwise.
		"""
		ignored_codes = {
			# code for regenerated password for sys, main, db users with empty password 
			'PasswordGenerated'
		}
		ignored_descriptions = {
			# ignore error about missing PPA functionality
			u"Execution of /usr/local/psa/admin/plib/api-cli/domain_pref.php"
			u" --update {domain} -webmail smwebmail -ignore-nonexistent-options"
			u" failed with return code 1."
			u"""
	Stderr is
	An error occurred while updating domain settings: Cannot set webmail: Webmail service is switched off.

	""".format(domain=self._domain_name)
		}

		if self._response_xml is None:
			# no valid response XML -> no errors
			return False

		for message_object in self._response_xml.findall('.//message'):
			if message_object.attrib.get('code') in ignored_codes:
				continue
			elif message_object.find('description').text in ignored_descriptions:
				continue
			elif message_object.attrib.get('code') == 'resolver' \
					and message_object.attrib.get('id') == 'component_not_installed_spam_assassin':
				# spamassassin is not available now, but was available before -
				# we do not want to spam customer with such errors
				continue
			else:
				return True	
		return False

	def get_error_messages(self):
		"""Get restoration error messages, enhance some of them

		Enhance message wording, when it is too cryptic or too verbose.
		"""
		replacements = {
			"It will be disabled for the following objects: %s" % (self._domain_name,):
				"This feature will be disabled.",
			'Backup file has wrong signature for this server': '',
			'managed_runtime_version': '.NET runtime'
		}

		# find messages in XML
		messages_raw = []
		if self._response_xml is not None:
			for message_node in self._response_xml.findall('.//message'):
				messages_raw.append(
					message_node.find('description').text.strip()
				)

		# perform replacements
		messages = []
		for message in messages_raw:
			for replace_me, new_text in replacements.iteritems():
				message = message.replace(replace_me, new_text)
			messages.append(message)

		return "\n\n".join(messages)