# Copyright 1999-2015. Parallels IP Holdings GmbH. All Rights Reserved.
package SiteApp::SiteAppGlobalSettings;

use strict;
use XmlNode;
use Logging;
use AgentConfig;

sub new ( $ ) {
  my $class = shift;
  my $self = {};
  bless($self, $class);
  $self->{xmlNode} = shift;
  return $self;
}

sub getSappsGlobalSettings {
  
  my $utilityResult = _getSappsGlobalSettingsUtilityResult();
  return unless defined $utilityResult;

  my $globalSettingsArray = _getSappsArray($utilityResult);
  return unless defined $globalSettingsArray;

  return _getSappInstalledXmlNodes($globalSettingsArray);
}

sub _getSappsGlobalSettingsUtil {  
  return AgentConfig::apsInstancesUtil() . " --get-packages-global-settings";
}

sub _getSappsGlobalSettingsUtilityResult { 
  
  my $cmd = _getSappsGlobalSettingsUtil();
  
  my $result = `$cmd`;
  Logging::debug($cmd);
  Logging::trace($result);
  chomp($result);

  my $retCode = $? >> 8;
  if( $retCode != 0 ) {
    Logging::warning( "Unable to get Global settings for APS Applications (ErrorCode: $retCode, STDOUT:$result)." ,'UtilityError');
    return;
  }

  return $result;
}

sub _getSappsArray {
  my $input = shift;
  my $sappsArrayDef = $input;

  my @packages;
  eval $sappsArrayDef;
  if ( $@ ) {
    Logging::warning( "Unable to parse APS definition from utility output ($@):\n $input", 'UtilityError' );
    return;
  }

  return \@packages;
}

sub _getSappInstalledXmlNodes {
  my $sappsArray = shift;

  my @sapps = @{$sappsArray};
  my @sappInstalledXmlNodes;

  foreach my $sapp (@sapps) {
    my $xmlNode = _hash2SappXmlNode($sapp);
    if ( defined $xmlNode ) {
      push @sappInstalledXmlNodes, $xmlNode;
    }
  }

  return @sappInstalledXmlNodes;
}

sub _hash2SappXmlNode {
  my $hashPtr = shift;
  return unless ( ref($hashPtr) =~ /HASH/ );

  my %hash = %{$hashPtr};

  my $name = $hash{'_name'};
  return unless defined $name;

  my $xmlNode = XmlNode->new( $name );
  if ( defined ( $hash{'_value'} ) and ( $hash{'_value'} ne '' ) ) {
    $xmlNode->setText( $hash{'_value'} );
  }

  if ( defined ( $hash{'_attrs'} ) and ( ref ( $hash{'_attrs'} ) =~ /HASH/ ) ) {
    my %attrs = %{$hash{'_attrs'}};
    foreach my $attrName ( keys %attrs ) {
      $xmlNode->setAttribute( $attrName, $attrs{$attrName} ) if defined $attrs{$attrName};
    }
  }

  if ( defined ( $hash{'_children'} ) and ( ref ( $hash{'_children'} ) =~ /ARRAY/ ) ) {
    my @children = @{$hash{'_children'}};
    foreach my $child ( @children ) {
      next unless ( ref( $child ) =~ /HASH/ );
      my $childXmlNode = _hash2SappXmlNode($child);
      $xmlNode->addChild($childXmlNode) if defined $childXmlNode;
    }
  }

  $xmlNode;
}

sub getPackageArchive {
  my ($self) = @_;
  my $packageArchiveNode = $self->{xmlNode}->getChild('package-archive');
  if ( defined $packageArchiveNode ) {
    return $packageArchiveNode->getText();
  }
}

sub getRegistryUid {
  my ($self) = @_;
  my $registryUidNode = $self->{xmlNode}->getChild('registryUid');
  if ( defined $registryUidNode ) {
    return $registryUidNode->getText();
  }
}

sub getSappPackageId {
  my ($self) = @_;
  my $sappPackageIdNode = $self->{xmlNode}->getChild('sapp-package-id');
  if ( defined $sappPackageIdNode ) {
    return $sappPackageIdNode->getText();
  }
}

# Returns %hash
sub getSettings
{
  my $self = shift;

  my %settings;
  
  my $settingsNode = $self->{xmlNode}->getChild('settings');

  if ( defined $settingsNode ) {
    my @settingsNodes = $settingsNode->getChildren('setting');

    if ( @settingsNodes ) {
      foreach my $settingNode ( @settingsNodes ) {
        my $settingName = $settingNode->getChild('name')->getText();
        my @settingValueNodes = $settingNode->getChildren('value');
        my @settingValues;
        if (@settingValueNodes) {
          foreach my $settingValueNode ( @settingValueNodes ) {
            push @settingValues, $settingValueNode->getText();
          }
          $settings{ $settingName } = join(",", @settingValues);
        } else {
          $settings{ $settingName } = $settingNode->getChild('value')->getText();
        }
        
      }
    }  
  
    return \%settings;
  }
  
  return;
}

1;
