#!/usr/bin/perl
# Copyright 1999-2012. Parallels IP Holdings GmbH. All Rights Reserved.


use strict;
use warnings;

use lib '.';
use genericAgent;
use CompatArgParse;
use Storage::Storage;
use DumpStatus;
use Logging;
use XmlNode;
use AgentConfig;

sub printHelp {
  print <<HELP;

Usage:
$0 <command> [<options>]
Commands:
  -s |--get-status           get status of the host: is this agent applicable
  -dd|--dump-domains=<list>  comma-separated list of domains to dump
  -da|--dump-all             dump all objects on hosts
  -lc|--get-content-list     get list of content files
  -h |--help                 this help

Options:
  -nc|--no-content           make a shallow structure dump
  -nz|--no-compress          do not compress content files
  -ss|--split-size=<size>    Split the dump to the parts of <size>.
                             Additional parts have .001, .002... suffixes.
                             Dump is not split if such option is not given.
HELP
}

sub main {

  my $mysqldbpassword = "";

  my %arg_opts = ('--help|-h'=>'',
                  '--get-status|-s'=>'',
                  '--dump-domains|-dd'=>'s',
                  '--dump-all|-da'=>'',
                  '--get-content-list|-lc'=>'',
                  '--no-content|-nc'=>'',
                  '--no-compress|-nz'=>'',
                  '--output|-o'=>'s',
                  '--split-size|-ss'=>'s'
                );

  my $args = CompatArgParse::parse(\@ARGV,\%arg_opts);

  if (exists $args->{'help'}) {
    printHelp();
    return 0;
  }

  if (exists $args->{'get-status'}){
    my $root = XmlNode->new('agent-status');
    if (!defined AgentConfig::iconvBin()) {
      $root->addChild(XmlNode->new('wrong-platform', 'content' => "No 'iconv' binary found on the host"));
    }

    $root->serialize(\*STDOUT);

    return 0;
  }

  my $do_gzip = exists $args->{'no-compress'} ? undef : 1;

  my ($storage, $status);

  my $workDir = AgentConfig::cwd();

  $storage = Storage::Storage::createFileStorage($do_gzip, $workDir, $args->{'split-size'});
  $status = DumpStatus::createMigration($workDir . "/dumping-status.xml");

  if (exists $args->{'get-content-list'}) {
    print $storage->getContentList();
    return 0;
  }

  my $generic = genericAgent->new($storage, $status, $mysqldbpassword, { 'web' => 1, 'mail' => 1, 'db' => 1 } );
  $generic->noContent() if exists $args->{'no-content'};

  if (exists $args->{'dump-domains'}) {
    my $value = $args->{'dump-domains'};
    if ($value eq "-") {
      $value = <STDIN>;
      chomp $value;
    }
    my @domains = split(/\s*,\s*/, $value);
    Logging::debug("Selected domains: " . join (", ", @domains));
    $generic->selectDomains(@domains);
  }

  if (exists $args->{'dump-all'}) {
    Logging::debug("Selected all objects");
    $generic->selectAll();
  }

  my $res = $generic->dump();
  $status->finish();
  return $res;
}

exit(main());