<!-- Copyright 1999-2012. Parallels IP Holdings GmbH. All Rights Reserved. -->
<xsl:stylesheet version="1.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform">
  <xsl:output indent="yes" encoding="UTF-8"/>
  <xsl:variable name='agent_name' select='RaQ4dump/@agent-name'/>

  <xsl:template match="RaQ4dump">
    <migration-dump agent-name="{$agent_name}">
      <xsl:apply-templates select="client"/>
    </migration-dump>
  </xsl:template>

  <xsl:template match="@ip">
    <ip>
      <xsl:choose>
        <xsl:when test="../anonftp">
          <ip-type>exclusive</ip-type>
        </xsl:when>
        <xsl:otherwise>
          <ip-type>shared</ip-type>
        </xsl:otherwise>
      </xsl:choose>
      <ip-address>
        <xsl:value-of select='.'/>
      </ip-address>
    </ip>
  </xsl:template>
  
  <!--
  Key used for removal of duplicate IP addresses in client's pool:
  [IP address value][name of the client][indication of ip based hosting]
  -->
  <xsl:key name="ip" match="domain/@ip" use="concat(., ../../@name, ../anonftp)"/>

  <xsl:template match="client">

    <client name="{@name}">
         <xsl:copy-of select="@contact"/>
	  <xsl:copy-of select="password"/>
	  <xsl:choose>
		<xsl:when test="@state and (@state = 'off' or @state = 'false')">
		  <status><disabled-by name="admin"/></status>
		</xsl:when>
		<xsl:otherwise>
		  <status><enabled/></status>
		</xsl:otherwise>
	  </xsl:choose>

	  <xsl:if test="@email">
		<pinfo name="email"><xsl:value-of select="@email"/></pinfo>
	  </xsl:if>

      <ip_pool>
        <xsl:apply-templates select="domain/@ip[generate-id(.) = generate-id(key('ip',concat(., ../../@name, ../anonftp)))]"/>
      </ip_pool>
      
      <xsl:apply-templates select="domain"/>
    </client>
  </xsl:template>
  
  <xsl:template match="domain">
    
    <domain name="{@name}">
      <xsl:if test="@www = 'true'">
        <xsl:attribute name="www">true</xsl:attribute>
      </xsl:if>

      <!-- hosting attribute ignored -->

      <xsl:apply-templates select="@ip"/>

      <!-- STATUS -->
      <xsl:choose>
        <xsl:when test="@state and (@state='off' or @state='false')">
		  <status><disabled-by name="admin"/></status>
        </xsl:when>
        <xsl:otherwise>
		  <status><enabled/></status>
        </xsl:otherwise>
      </xsl:choose>

      <!-- LIMIT -->
      <xsl:for-each select="limit">
        <xsl:if test="@name='traffic'">
          <xsl:call-template name="limit">
            <xsl:with-param name="limit" select="'max_traffic'"/>
            <xsl:with-param name="value" select="text()"/>
          </xsl:call-template>
        </xsl:if>
        <xsl:if test="@name='space'">
          <xsl:call-template name="limit">
            <xsl:with-param name="limit" select="'disk_space'"/>
            <xsl:with-param name="value" select="text()"/>
          </xsl:call-template>
        </xsl:if>
        <xsl:if test="@name='account'">
          <xsl:call-template name="limit">
            <xsl:with-param name="limit" select="'max_wu'"/>
            <xsl:with-param name="value" select="text()"/>
          </xsl:call-template>
        </xsl:if>
      </xsl:for-each>

      <!-- MAILSYSTEM -->
      <xsl:call-template name="create-mailsystem"/>

      <!-- DOMAIN ALIAS-->
      <xsl:copy-of select="domain-alias"/>

      <xsl:copy-of select="database"/>

      <!-- TODO MAILLIST -->

      <!-- TODO TRAFFIC -->

      <!-- CERTIFICATE -->
      <xsl:copy-of select="certificate"/>

      <!-- PHOSTING -->
	  <xsl:if test="admin">
		<phosting fpauth="false">
		  <xsl:if test="@cid">
			<xsl:attribute name="cid_docroot"><xsl:value-of select="@cid"/></xsl:attribute>
		  </xsl:if>
		  <xsl:copy-of select="@cid_private"/>

		  <xsl:call-template name="convert-boolean-permission">
			<xsl:with-param name="raqname" select="'ssl'"/>
			<xsl:with-param name="pleskname" select="'https'"/>
		  </xsl:call-template>
		  <xsl:call-template name="convert-boolean-permission">
			<xsl:with-param name="raqname" select="'frontpage'"/>
			<xsl:with-param name="pleskname" select="'fp'"/>
		  </xsl:call-template>

		  <xsl:choose>
			<xsl:when test="permission[@name='frontpage' and @value='on'] and permission[@name='ssl' and @value='on']">
			  <xsl:attribute name="fpssl">true</xsl:attribute>
			</xsl:when>
			<xsl:otherwise>
			  <xsl:attribute name="fpssl">false</xsl:attribute>
			</xsl:otherwise>
		  </xsl:choose>

		  <!-- casp permission ignored -->

		  <!-- SYSUSER -->
		  <xsl:copy-of select="admin/sysuser"/>

		  <!-- SCRIPTING -->
		  <xsl:call-template name="create-scripting"/>

		  <xsl:if test="permission[@name='frontpage']">
			<fpuser>
			  <xsl:attribute name="name">
				<xsl:call-template name="mangle-sysuser-name">
				  <xsl:with-param name="name" select="admin/@name"/>
				</xsl:call-template>
			  </xsl:attribute>
			  <xsl:copy-of select="admin/sysuser/password"/>
			</fpuser>
		  </xsl:if>

		  <!-- ANONFTP -->
		  <xsl:if test="anonftp">
			<xsl:call-template name="create-anonftp"/>
		  </xsl:if>

		  <xsl:apply-templates select="user" mode="webuser"/>
		</phosting>
	  </xsl:if>
    </domain>
  </xsl:template>

  <xsl:template name="create-mailsystem">
    <mailsystem>
	  <status><enabled/></status>
      <xsl:for-each select="admin|user">
        <xsl:call-template name="create-mailuser"/>
      </xsl:for-each>
	  <xsl:if test="@catch-all">
		<catch-all><xsl:value-of select="@catch-all"/></catch-all>
	  </xsl:if>
    </mailsystem>
  </xsl:template>

  <xsl:template name="create-mailuser">
    <mailuser name="{substring-before(mail/@mailname, '@')}">
      <xsl:choose>
        <xsl:when test="mail/forward">
          <xsl:attribute name="mailgroup-enabled">true</xsl:attribute>
        </xsl:when>
        <xsl:otherwise>
          <xsl:attribute name="mailgroup-enabled">false</xsl:attribute>
        </xsl:otherwise>
      </xsl:choose>
      <xsl:copy-of select="sysuser/password"/>
      <xsl:copy-of select="mail/mailbox"/>
      <xsl:copy-of select="mail/alias"/>
      <xsl:for-each select="mail/forward">
        <mailgroup-member><xsl:value-of select="text()"/></mailgroup-member>
      </xsl:for-each>
      <xsl:for-each select="mail/autoresponder">
		<autoresponders enabled="true">
		  <autoresponder name="@mailname">
			<xsl:attribute name="status">
			  <xsl:choose>
				<xsl:when test="@state='off'">off</xsl:when>
				<xsl:otherwise>on</xsl:otherwise>
			  </xsl:choose>
			</xsl:attribute>
			<text><xsl:value-of select="text()"/></text>
		  </autoresponder>
		</autoresponders>
      </xsl:for-each>
    </mailuser>
  </xsl:template>

  <xsl:template name="create-anonftp">
    <anonftp>
      <xsl:attribute name="pub">
        <xsl:choose>
          <xsl:when test="anonftp/@cid">true</xsl:when>
          <xsl:otherwise>false</xsl:otherwise>
        </xsl:choose>
      </xsl:attribute>
      <xsl:attribute name="cid">
        <xsl:value-of select="anonftp/@cid"/>
      </xsl:attribute>
      <xsl:attribute name="incoming">
        <xsl:choose>
          <xsl:when test="anonftp/@cid_incoming">true</xsl:when>
          <xsl:otherwise>false</xsl:otherwise>
        </xsl:choose>
      </xsl:attribute>
      <xsl:attribute name="cid_incoming">
        <xsl:value-of select="anonftp/@cid_incoming"/>
      </xsl:attribute>

      <xsl:if test="anonftp/@cid_incoming and limit[@name='space']">
        <anonftp-limit name="incoming-disk-quota">
          <xsl:value-of select="limit[@name='space']/text()"/>
        </anonftp-limit>
      </xsl:if>
      <xsl:if test="anonftp/limit[@name='account']">
        <anonftp-limit name="max-connections">
          <xsl:value-of select="anonftp/limit[@name='account']/text()"/>
        </anonftp-limit>
      </xsl:if>
    </anonftp>
  </xsl:template>

  <xsl:template name="create-scripting">
    <scripting>
      <xsl:call-template name="convert-boolean-permission">
        <xsl:with-param name="raqname" select="'cgi'"/>
        <xsl:with-param name="pleskname" select="'cgi'"/>
      </xsl:call-template>

      <xsl:call-template name="convert-boolean-permission">
        <xsl:with-param name="raqname" select="'ssi'"/>
        <xsl:with-param name="pleskname" select="'ssi'"/>
      </xsl:call-template>

      <xsl:call-template name="convert-boolean-permission">
        <xsl:with-param name="raqname" select="'php'"/>
        <xsl:with-param name="pleskname" select="'php'"/>
      </xsl:call-template>

      <xsl:call-template name="convert-boolean-permission">
        <xsl:with-param name="raqname" select="'casp'"/>
        <xsl:with-param name="pleskname" select="'asp'"/>
      </xsl:call-template>

    </scripting>
  </xsl:template>

  <xsl:template name="mangle-sysuser-name">
    <xsl:param name="name"/>
    <xsl:choose>
      <xsl:when test="$name = 'admin' or $name = 'ftp' or $name = 'anonymous' or $name = 'root' or $name = 'mailman' or $name = 'tomcat4'"><xsl:value-of select="concat($name, '1')"/></xsl:when>
      <xsl:otherwise><xsl:value-of select="$name"/></xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <xsl:template name="mangle-sysuser">
    <xsl:param name="sysuser"/>
    <sysuser>
      <xsl:attribute name="name">
        <xsl:call-template name="mangle-sysuser-name">
          <xsl:with-param name="name" select="$sysuser/@name"/>
        </xsl:call-template>
      </xsl:attribute>
      <xsl:copy-of select="$sysuser/@quota"/>
      <xsl:copy-of select="$sysuser/@shell"/>
      <xsl:copy-of select="$sysuser/password"/>
    </sysuser>
  </xsl:template>

  <xsl:template name="create-sysuser">
    <xsl:param name="user"/>
    <sysuser>
      <xsl:attribute name="name">
        <xsl:call-template name="mangle-sysuser-name">
          <xsl:with-param name="name" select="$user/sysuser/@name"/>
        </xsl:call-template>
      </xsl:attribute>
      <xsl:if test="$user/permission[@name = 'shell']">
        <xsl:attribute name="shell">/bin/bash</xsl:attribute>
      </xsl:if>
      <xsl:copy-of select="$user/sysuser/password"/>
    </sysuser>
  </xsl:template>

  <xsl:template match="user" mode="webuser">
    <webuser>
      <!-- sysuser -->
      <xsl:call-template name="create-sysuser">
        <xsl:with-param name="user" select="."/>
      </xsl:call-template>
    </webuser>
  </xsl:template>

  <xsl:template name="convert-boolean-permission">
    <xsl:param name="raqname"/>
    <xsl:param name="pleskname"/>
    <xsl:choose>
      <xsl:when test="permission[@name = $raqname and @value = 'on']">
        <xsl:attribute name="{$pleskname}">true</xsl:attribute>
      </xsl:when>
      <xsl:otherwise>
        <xsl:attribute name="{$pleskname}">false</xsl:attribute>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <!-- - - - - - - - - - - - - - - - - - - - - - -
       Outputs Plesk's limit (used for domains
       and clients accounts)
       Parameters:
       limit - limit name
       value - limit value
       - - - - - - - - - - - - - - - - - - - - - - - -->
  <xsl:template name="limit">
    <xsl:param name="limit"/>
    <xsl:param name="value"/>
    <xsl:element name="limit">
      <xsl:attribute name="name">
        <xsl:value-of select="$limit"/>
      </xsl:attribute>
      <xsl:value-of select="format-number($value, '#')"/>
    </xsl:element>
  </xsl:template>

</xsl:stylesheet>
