package DomainDumper;

use strict;
use warnings;

use AgentConfig;
use DumperUtils;
use EnsimGuidGenerator;
use File::Spec;
use Logging;
use PanelConfigFile;
use MySQLServer;
use PanelDb;
use Parser;

sub getSiteId {
  my ( $domainName ) = @_;

  return _getPanelDbValue($domainName,
    "SELECT site_id FROM siteinfo WHERE domain = '$domainName'"
  );
}

sub getAdminLogin {
  my $siteName = shift;
  return _getConfigFileValue($siteName, 'siteinfo', 'admin_user');
}

sub getAdminSystemLogin {
  my $siteName = shift;
  return _getConfigFileValue($siteName, 'siteinfo', 'admin');
}

sub isEnabled {
  my $siteName = shift;
  my $siteId = getSiteId($siteName);

  Logging::trace("Getting disable status for '$siteId' site...");
  my $isEnabled = 1;
  if (-f PanelConfigFile::getSiteinfoFileName($siteId) . "/disabled") {
    $isEnabled = 0;
  }
  Logging::debug( "Site '$siteName' is " . ( ( $isEnabled == 1 ) ? 'enabled' : 'disabled' ) . '.' );
  return $isEnabled;
}

# PHP handler is determined by site security level. 'high' will get 'cgi+suexec',
# 'medium' and 'low' - 'mod_php'. The only easily visible difference between those 
# groups of security levels is a 'jail' flag in 'apache' table, which is enough.
sub getSitePhpMode {
  my $siteName = shift;

  my $sql = "SELECT CASE
        WHEN jail=1 THEN 'cgi'
        ELSE 'module'
      END
      AS phpmode
    FROM apache a, siteinfo s
    WHERE a.site_id=s.site_id AND s.domain = '$siteName'";
  return _getPanelDbValue($siteName, $sql);
}

sub isSslEnabled {
  my $siteName = shift;
  return _getConfigFileValue($siteName, 'openssl', 'enabled');
}

sub isSsiEnabled {
  my $siteName = shift;
  return _getConfigFileValue($siteName, 'ssi', 'enabled');
}

sub isPerlEnabled { 
  my $siteName = shift;
  return _getConfigFileValue($siteName, 'mod_perl', 'enabled');
}

sub isCgiEnabled { 
  my $siteName = shift;
  return _getConfigFileValue($siteName, 'cgi', 'enabled');
}

sub isMivaEnabled { 
  my $siteName = shift;
  my $miva4Enabled = _getConfigFileValue($siteName,  'mivamerchant', 'enabled');
  my $miva5Enabled = _getConfigFileValue($siteName,  'mivamerchant5', 'enabled');
  return ($miva4Enabled || $miva5Enabled);
}

sub isSendmailEnabled {
  my $siteName = shift;
  return _getConfigFileValue($siteName, 'sendmail', 'enabled');
}

sub isWebmailEnabled {
  my $siteName = shift;
  return _getConfigFileValue($siteName, 'sqmail', 'enabled');
}

sub isMailsystemEnabled {
  my $siteName = shift;
  return _getConfigFileValue($siteName, 'imap', 'enabled');
}

sub getDomainAliases {
  my ($domainName) = @_;

  my $siteId = getSiteId($domainName);
  my %info = PanelConfigFile::getContents($siteId, 'aliases');
  my $aliases = $info{'aliases'};
  # ['@li.as', 'metoo.test'] --> @li.as, metoo.test
  $aliases =~ s/[\[\]']//g;

  return split ', ', $aliases;
}

sub isDnsEnabled {
  my $siteName = shift;
  return _getConfigFileValue($siteName, 'bind', 'enabled');
}

sub isAnonymousFtpEnabled {
  my $siteName = shift;
  return _getConfigFileValue($siteName, 'anonftp', 'enabled');
}

sub hasManageDnsPermission {
  my $siteName = shift;
  my %bindInfo = _getBindInfo($siteName);
  return $bindInfo{'zone_mgmt'};
}

sub getIpInfo {
  my $siteName = shift;
  my $siteId = getSiteId($siteName);
  return PanelConfigFile::getContents($siteId, 'ipinfo', 'IP info');
}

sub isNameBased {
  my $siteName = shift;
  return _getConfigFileValue($siteName, 'ipinfo', 'namebased');
}

sub isFrontpageEnabled {
  my $siteName = shift;
  return _getConfigFileValue($siteName, 'frontpage', 'enabled');
}

sub _getBindInfo {
  my ( $siteName ) = @_;
  my %defaultSettings = (
     'dns_log'      => 0,
     'zone_mgmt'    => 0,
     'enabled'      => 0,
  );
  my $siteId = getSiteId($siteName);
  return PanelConfigFile::getContents($siteId, 'bind', 'DNS info', %defaultSettings);
}

sub getDatabaseNames {
  my $domainName = shift @_;

  my $dbPrefix = DomainDumper::getDatabasePrefix($domainName);
  return MySQLServer::getDatabases($dbPrefix);
}

sub getDatabasePrefix {
  my $domainName = shift;
  my $defaultDbPrefix = $domainName;
  $defaultDbPrefix =~ s/\./_/g;

  my $siteId = DomainDumper::getSiteId($domainName);
  # TODO GetConfigFileValue()
  my %mysqlInfo = PanelConfigFile::getContents($siteId, 'mysql');
  return exists $mysqlInfo{'dbaseprefix'} ? $mysqlInfo{'dbaseprefix'} : $defaultDbPrefix;
}

sub _getConfigFileValue {
  my ($siteName, $configName, $parameterName) = @_;
  my $siteId = getSiteId($siteName);
  my %configData = PanelConfigFile::getContents($siteId, $configName);
  my $value = $configData{$parameterName};
  _traceConfigValue($siteName, $configName, $parameterName, $value);
  return $value;
}

sub _getPanelDbValue {
  my ($domainName, $query) = @_;
  my $result = PanelDb::_getDb()->queryOneValue($query);
  _tracePanelDbValue($domainName, $query, $result);
  return $result;
}

# Provide hints for verification of migration results
sub _traceConfigValue {
  my ($siteName, $configName, $parameterName, $value) = @_;
  my $myModule = 'DomainDumper';
  my $callerName = $_ = (caller(2))[3];
  /^$myModule/ or Logging::debug("Unexpected dependency: expected to be called from module '$myModule'; actually called from '$callerName'");
  my $siteId = getSiteId($siteName);
  my $fileName = PanelConfigFile::getConfigFileName($siteId, $configName);
  Logging::trace($callerName."(domain='$siteName', file='$fileName', option='$parameterName'): $value");
}

# Provide hints for verification of migration results
sub _tracePanelDbValue {
  my ($siteName, $query, $value) = @_;
  my $myModule = 'DomainDumper';
  my $callerName = $_ = (caller(2))[3];
  $query =~ s/\s+/ /g;
  /^$myModule/ or Logging::debug("Unexpected dependency: expected to be called from module '$myModule'; actually called from '$callerName'");
  Logging::trace($callerName."(domain='$siteName', query='$query'): $value");
}

sub getEnabledWebstatEngines {
  my ($domain) = @_;

  my $siteId = getSiteId($domain);

  my @enabled = ();
  for my $engine qw(webalizer awstats analog) {
    my %info = PanelConfigFile::getContents($siteId, $engine);
    if ('1' eq $info{'enabled'}) {
      push @enabled, $engine;
    }
  }

  return @enabled;
}

sub getFilesystemRoot {
  my ($domainName) = @_;

  return "/home/virtual/$domainName";
}

sub getListOfSubdomains {
  my ($domainName) = @_;

  my $configListOfSubdomains;
  my $currentSubdomain;
  my $subcount=0;

  my $siteId = getSiteId($domainName);

  # Parser::makeSafeFileParser("/etc/virtualhosting/subdomain/site" . $siteId) not supported
  # syntax of this config. So, will read config by hand...

  my ($subdomainsConfigFile);
  open($subdomainsConfigFile, "/etc/virtualhosting/subdomain/site" . $siteId) || return undef;
  while (<$subdomainsConfigFile>) {
    chomp;
    if (/^\[(.*)\]$/) {
        $currentSubdomain = $1;
        $subcount++ if ($currentSubdomain ne "DEFAULT");
        next;
    }
    next if (!defined($currentSubdomain));

    next if (/^#/ || /^$/);
    next if (! m/^(.*?)[ \t]*=[ \t]*(.*)$/ );
    $configListOfSubdomains->{"$currentSubdomain"}{"$1"} = "$2";

  }
  close($subdomainsConfigFile);

  if (defined($configListOfSubdomains->{"DEFAULT"})) {
    return undef unless $subcount;      # check if define only DEFAULT subdomain (and not have other subdomain(s))

    # fill now default values in all subdomain(s)
    foreach my $subdomainName ( keys %{$configListOfSubdomains} ) {
        foreach my $values_subdomain (keys %{$configListOfSubdomains->{"DEFAULT"}}) {
            if (!defined($configListOfSubdomains->{"$subdomainName"}{$values_subdomain})) {
                $configListOfSubdomains->{"$subdomainName"}{$values_subdomain}
                    = $configListOfSubdomains->{DEFAULT}{$values_subdomain};
            }
        }
    }
    delete($configListOfSubdomains->{DEFAULT});
  }

  return $configListOfSubdomains;
}

sub getResellerName($) {
  my ($domainName) = @_;
  my $clientId = getSiteId($domainName);
  my $query = "SELECT reseller_info.username 
    FROM reseller, reseller_info 
    WHERE reseller.site_id = '$clientId'
        AND reseller.reseller_id = reseller_info.reseller_id";
  my $resellerName = _getPanelDbValue($domainName, $query);
  return $resellerName; 
}

sub getVendorGuid($) {
  my ($domainName) = @_;

  my $resellerName = getResellerName($domainName);

  if (!defined($resellerName) or $resellerName eq '') {
    return EnsimGuidGenerator::getAdminGuid();
  } else {
    return EnsimGuidGenerator::getResellerGuid($resellerName);
  }
}

sub getBandwidthLimit {
  my ( $domainName ) = @_; 

  my $siteId = getSiteId($domainName);
  my $result = _getPanelDbValue($domainName, 
    "SELECT threshold FROM bandwidth WHERE site_id = '$siteId'"
  );

   # for a bandwidth threshold '0' is a special value for unlimited
  if ($result eq '0') {
    return undef;
  } else {
    return $result;
  }
}

sub getSiteContactEmail {
  my ( $domain ) = @_;

  return _getPanelDbValue($domain,
    "SELECT email FROM siteinfo WHERE domain = '$domain'"
  );
}

sub getSiteAdminLogin {
  my ( $domain ) = @_;

  return _getPanelDbValue($domain,
    "SELECT admin_user FROM siteinfo WHERE domain = '$domain'"
  );
}

my %ignoredUsers = (
  'root' => '1',
  'bin' => '1',
  'daemon' => '1',
  'adm' => '1',
  'lp' => '1',
  'sync' => '1',
  'shutdown' => '1',
  'halt' => '1',
  'mail' => '1',
  'uucp' => '1',
  'operator' => '1',
  'games' => '1',
  'gopher' => '1',
  'ftp' => '1',
  'news' => '1',
  'mysql' => '1',
  'nobody' => '1',
  'zope' => '1',
  'majordomo' => '1',
  'tomcat4' => '1',
  'apache' => '1',
  'ensimrootmail' => '1',
  'sshd' => '1',
  'smmsp' => '1',
);

sub getSiteUsersInfoMap {
  my ($domainName) = @_;

  my $passwdReader = Parser::makeSafeFileParser(getFilesystemRoot($domainName) . "/etc/passwd");
  my $shadowReader = Parser::makeSafeFileParser(getFilesystemRoot($domainName) . "/etc/shadow");

  my (%passwd, %shadow);

  if (defined($passwdReader)) {
    %passwd = %{$passwdReader->{'PARSE'}->('KEYSEPARATOR' => ':', 'VALUESEPARATOR' => ':')};
  } else {
    return undef;
  }

  if (defined($shadowReader)) {
    %shadow = %{$shadowReader->{'PARSE'}->('KEYSEPARATOR' => ':', 'VALUESEPARATOR' => ':')};
  }

  my %userParams;

  while (my ($username, $paramsPtr) = each %passwd) {
    if (defined($ignoredUsers{$username})) {
      next;
    }
    my @params = @{$paramsPtr};
    my $fullname = $params[3]; #see the 'passwd' format reference
    my $id = $params[1]; #see the 'passwd' format reference
    my $password;

    if (%shadow && defined($shadow{$username})) {
      my @shadowParams = @{$shadow{$username}};
      $password = $shadowParams[0]; #see the 'shadow' format reference
    }

    $userParams{$username} = { 'fullname' => $fullname, 'password' => $password, 'id' => $id };
  }

  return \%userParams;
}

sub _getProtectedDirectoryInfo {
  my ($htaccess, $usergroups, $users) = @_;

  my %htgroups = %{$usergroups};
  my %htusers = %{$users};
  my %pdirusers;
  my %result;

  open HTACCESS, $htaccess;
  binmode HTACCESS;
  while (<HTACCESS>) {
    if (/require\s+group\s+(.*)$/) {
      my @groups = split /\s+/, $1;
      my $group;
      foreach $group (@groups) {
        if (defined($htgroups{$group})) {
          my $htuser;
          foreach $htuser (@{$htgroups{$group}}) {
            if (defined($htusers{$htuser})) {
              $pdirusers{$htuser} = $htusers{$htuser};
            } else {
              $pdirusers{$htuser} = '';
            }
          }
        }
      }
    } elsif (/AuthName [\"]?([^\"]*)[\"]?$/) {
      $result{'title'} = $1;
    }
  }
  close HTACCESS;

  if (%pdirusers) {
    $result{'users'} = \%pdirusers;
    return \%result;
  }

  # at this point, it's a FrontPage's protected directory and to warn about it is nonsense;
  # instead, will just return undef and not include such protected directory in dump.
}

sub _getRelativeDirectory {
  my ($fullPath, $prefix, $suffix) = @_;

  if ($fullPath =~ /^\Q$prefix\E(.*)$suffix$/) {
        my $infix = $1;
        if ($infix =~ /^(.*)\/$/ and $infix ne '/') {
          return $1;
        }
        return $infix;
  }

  Logging::warning("Unable to get relative directory from path $fullPath given prefix $prefix and suffix $suffix");
}

sub _getProtectedDirectoriesOfAKind {
  my ($domainName, $htgroups, $htusers, $rootPath, $isCgi) = @_;

  my $htaccessList;
  my @result;
  my $find = AgentConfig::findBin();

  my $searchedFile = '.htaccess';
  open($htaccessList, "$find '$rootPath' -name $searchedFile -type f |");
  binmode $htaccessList;
  while (<$htaccessList>) {
    my $fullPath = $_;
    my $pdirInfo_ptr = _getProtectedDirectoryInfo($fullPath, $htgroups, $htusers) or next;
    $pdirInfo_ptr->{'cgi'} = $isCgi;
    $pdirInfo_ptr->{'name'} = _getRelativeDirectory($fullPath, $rootPath, $searchedFile);
    push @result, $pdirInfo_ptr;
  }
  close $htaccessList;

  return \@result;
}

sub getProtectedDirectories {
  my ( $domainName ) = @_;

  my %htusers;
  my %htgroups;

  my $htpasswdPath = "/home/virtual/$domainName/var/www/.htpasswd";
  my $htgroupPath = "/home/virtual/$domainName/var/www/.htgroup";
  if (-e $htpasswdPath) {
    my $htpasswdReader = Parser::makeSafeFileParser("/home/virtual/$domainName/var/www/.htpasswd");
    if (defined($htpasswdReader)) {
          %htusers = %{$htpasswdReader->{'PARSE'}->('KEYSEPARATOR'  => ':')};
    }
  }

  if (-e $htgroupPath) {
    my $htgroupReader = Parser::makeSafeFileParser("/home/virtual/$domainName/var/www/.htgroup");
    if (defined($htgroupReader)) {
      %htgroups = %{$htgroupReader->{'PARSE'}->('KEYSEPARATOR' => ':')};

      my $group;
      foreach $group (keys %htgroups) {
        my @members = split(/\s+/, $htgroups{$group});
        $htgroups{$group} = \@members;
      }
    }
  }

  # htmlPDirs and cgiPDirs each have the following structure:
  # [
  #   { 'name' => $directory,
  #     'title' => $AuthName from .htaccess file,
  #     'cgi' => whether it is cgi directory or regular html directory,
  #     'users' => {
  #       $username => $password,
  #       ...
  #     }
  #   }
  # ]
  my $wwwRoot = "/home/virtual/$domainName/var/www";
  my @htmlPDirs = @{_getProtectedDirectoriesOfAKind($domainName, \%htgroups, \%htusers, $wwwRoot . "/cgi-bin", "true")};
  my @cgiPDirs = @{_getProtectedDirectoriesOfAKind($domainName, \%htgroups, \%htusers, $wwwRoot . '/html', "false")};

  return @cgiPDirs, @htmlPDirs;
}

sub getLogrotateSettings {
  my ($domainName) = @_;

  my %settings;

  open(LOGROT, "/home/virtual/$domainName/etc/logrotate.conf") or return;
  binmode LOGROT;

  my $multiplier = 1;
  $settings{'period'} = 'monthly';

  while (<LOGROT>) {
        if (/^rotate\s+(\d+)*$/) {
          $settings{'number-of-rotated-files'} = $1;
        } elsif (/^size\s+(\d+)(.?)$/) {
          if ($2 eq 'M') {
                $multiplier = 1024*1024;
          } elsif ($2 eq 'k') {
                $multiplier = 1024;
          }
          $settings{'size'} = $1 * $multiplier;
        } elsif (/^(daily|weekly|monthly)$/) {
          $settings{'period'} = $1;
        }
  }

  unless (defined $settings{'number-of-rotated-files'}) {
        Logging::warning("Log rotation configuration is not complete.");
        return;
  }
  return %settings;
}

sub getWebUsers {
  my ($domainName) = @_;

  my %result;

  my $subdomainList = getListOfSubdomains($domainName);
  my %excludedUsers;
  for my $subdomain (values %$subdomainList) {
    my $documentRoot = $subdomain->{'document_root'};
    if ($documentRoot =~ /\/home\/([^\/]+)\/public_html/) {
      $excludedUsers{$1} = 1;
    }
  }
  my $usersList = getSiteUsersInfoMap($domainName);
  while (my ($userName, $userParams) = each(%$usersList)) {
    $result{$userName} = $userParams->{'password'} unless (defined $excludedUsers{$userName});
  }

  return \%result;
}

sub getMailboxForwarding {
  my ($domainName, $userName) = @_;

  my $detectEmailRegex = qr/^[^\\\|"#\s]/;
  my @forwardAddresses = ();

  my $forwardFilePath = File::Spec->catfile(getFilesystemRoot($domainName), 'home', $userName, '.forward');
  if (-e $forwardFilePath) {
    open(my $fh, $forwardFilePath);
    my $firstLine = <$fh>;
    close($fh);
    if (defined($firstLine)) {
      chomp($firstLine);
    
      if ($firstLine =~ $detectEmailRegex) { # non-special line beginning denotes email address
        my @possibleAddressesList = split(/[ ,]+/, $firstLine);
        foreach my $possibleAddress (@possibleAddressesList) {
          if ($possibleAddress =~ $detectEmailRegex) { # no reason to implement full RFC here
            push @forwardAddresses, $possibleAddress;
          }
        }
      }
    }
  }
  return @forwardAddresses;
}

sub getMailAliases {
  my ($domainName, $ignoreAliases) = @_;

  my %ignoreAliasesHash;

  if (defined($ignoreAliases)) {
    %ignoreAliasesHash = map { $_ => 1  } @{$ignoreAliases};
  }
  
  my $configReader = Parser::makeFileParser(getFilesystemRoot($domainName) . '/etc/aliases');
  my %aliases = %{$configReader->{'PARSE'}->(
    'KEYSEPARATOR' => ':',
    'COMMENT' => '#',
    'VALUESEPARATOR' => ','
  )};

  my ($alias, $ptrActions);
  my %result;

  while (($alias, $ptrActions) = each %aliases) {

    # ignore default majordomo alias
    if ($alias eq 'majordomo') {
      next;
    }

    if (defined($ignoreAliasesHash{$alias})) {
      next;
    }

    my @actions = @{$ptrActions};
    my (@redirects, @localDelivery, @responders);

    @actions = map { trim($_); } @actions;

    foreach (@actions) {
      if (/^"\|responder\.sh\s+\S+\s+(\S+)"$/) {
        if (open(INPUT, getFilesystemRoot($domainName) . $1)) {
          binmode INPUT;
          my $subject = <INPUT>;
          chomp $subject;
          my $body;
          while (<INPUT>) {
            $body .= $_;
          }
          close(INPUT);

          push @responders, { 'subject' => $subject, 'body' => $body };
        } else {
          Logging::info("Responder is enabled for $alias, but the corresponding message can't be read ($!). Responder won't be migrated.");
        }
      } elsif (/^([\w\.-]+)$/) {
        push @localDelivery, $1;
      } elsif (/^([\w@\.-]+)$/) {
        push @redirects, $1;
      } elsif (/^"\|\/usr\/lib\/majordomo\/wrapper$"/) {
        Logging::debug("Majordomo action ignored: $_");
      } elsif (/^:include:/) {
        Logging::debug("Majordomo action ignored: $_");
      } else {
        Logging::debug("Unknown action detected: $_");
      }
    }

    $result{$alias} = {
      'responders' => \@responders,
      'localDelivery' => \@localDelivery,
      'redirects' => \@redirects
    };
  }

  return \%result;
}

sub trim {
  my ($string) = @_; 
  $string =~ s/^\s+//;
  $string =~ s/\s+$//;
  return $string;
}

sub getMaillists {
  my ($domainName) = @_;

  my $maillistsFiles = _getMaillistsFiles($domainName);
  my %result;

  while (my ($maillistName, $files) = each %{$maillistsFiles}) {
    $result{$maillistName} = {
      'adminPassword' => _getMaillistAdminPassword($files->{'configFile'}),
      'members' => _getMaillistMembers($files->{'membersListFile'}),
      'owner' => _getMaillistOwner($domainName, $maillistName)
    };
  }

  return \%result; 
}

sub getMaillistsAliases {
  my ($domainName) = @_;

  my @maillists = keys(%{_getMaillistsFiles($domainName)});
  my @aliases;

  for my $maillist (@maillists) {
    push @aliases, (
      $maillist,
      "owner-$maillist",
      "$maillist-owner",
      "$maillist-request",
      "$maillist-outgoing",
      "$maillist-list",
      "$maillist-approval"
    );
  }

  return \@aliases;
}

# returns reference to a hash with:
# - key: name of a maillist
# - value: reference to a hash with keys 'configFile' and 'userListFile' and values - full path to corresponding maillist files
sub _getMaillistsFiles {
  my ($domainName) = @_;

  my $listsDir = getFilesystemRoot($domainName) . '/var/lib/majordomo/lists';

  unless (-d $listsDir) {
    Logging::debug("Domain $domainName doesn't contain maillists directory");
    return {};
  }
  
  my %lists;

  open(LISTCONFIGS, AgentConfig::findBin() . " $listsDir -maxdepth 1 -type f -name '*.config' |");
  binmode LISTCONFIGS;

  while(<LISTCONFIGS>) {
    chomp;
    
    my $configFile = $_;
    my ($listName, $membersListFile);
    
    if (/([^\/]+)\.config/) {
      $listName = $1;
    } else {
      Logging::warning("Failed to get maillist name from config file name '$_', ignoring maillist config");
      next;
    }
    
    if (/^(.+)\.config$/) {
      $membersListFile = $1;
    } else {
      Logging::warning("Failed to get maillist members file name from config file name '$_', ignoring maillist config");
      next;
    }

    $lists{$listName} = {
      'configFile' => $configFile,
      'membersListFile' => $membersListFile
    };
	}

	close LISTCONFIGS;

  return \%lists;
}

sub _getMaillistAdminPassword {
  my ($configFile) = @_;

  open(LISTCONFIG, $configFile);
  binmode LISTCONFIG;

  while (<LISTCONFIG>) {
    if (/admin_passwd\s=\s(.*)$/) {
      close(LISTCONFIG);
      return $1;
    }
  }

  close LISTCONFIG;
  return undef;
}

sub _getMaillistMembers {
  my ($membersListFile) = @_;

  my @result;

	open(LISTMEMBERS, $membersListFile);
	binmode LISTMEMBERS;

	while (<LISTMEMBERS>) {
		chomp;
    push @result, $_;
  }

  close LISTMEMBERS;

  return \@result;
}

sub _getMaillistOwner {
  my ($domainName, $maillistName) = @_;

  my $aliases = getMailAliases($domainName);

  while (my ($alias, $aliasInfo) = each %{$aliases}) {
    if ($alias eq $maillistName . '-owner') {
      if (@{$aliasInfo->{'localDelivery'}}) {
        return $aliasInfo->{'localDelivery'}->[0] . '@' . $domainName;
      } elsif (@{$aliasInfo->{'redirects'}}) {
        return $aliasInfo->{'redirects'}->[0];
      }
    }
  }

  return undef;
}
# TODO During migration it should go to dump.xml node 'phosting / preferences / sysuser / cron'
sub getCronJobs {
  my $domain = shift @_;
  my $cronFile = "/var/spool/cron/".getAdminSystemLogin($domain);
  return DumperUtils::getFileLines($cronFile);
}

sub getMailResponders {
  my ($domain) = @_;

  my $siteId = getSiteId($domain);
  my $respondersDir = "/home/virtual/site$siteId/fst/usr/share/opcenter/responders";

  my @responders = ();
  unless (opendir (my $dir, $respondersDir)) {
    Logging::warning("Unable to open the autoresponders directory '$respondersDir'");
  } else {
    @responders = map { "$respondersDir/$_" } grep { /^responder*/ && -f "$respondersDir/$_" } readdir ($dir);
    closedir $dir;
  }
  return @responders;
}

sub getPowerToolsAvailability {
  my ($domain) = @_;

  my $siteId = getSiteId($domain);

  my $sql = "SELECT d.shortname FROM deployables d JOIN deployables_sites ds ON d.shortname = ds.shortname "
            . "WHERE ds.site_id = $siteId AND d.installed = TRUE AND ds.deployable = TRUE";
  return PanelDb::_getDb()->queryList($sql);
}

1;
