package PreMigration;

use strict;
use warnings;
use Logging;
use DumpComposer;
use PreMigrationChecks;

my @MESSAGES;

# severity levels
use constant {
  CRITICAL => 3,
  WARNING => 2,
  # use this level to show non-critical information
  INFO => 1,
  # use this level to show migration mapping
  MAPPING => 0
};

use constant SEVERITY_LEVELS => [CRITICAL, WARNING, INFO, MAPPING];

# names of severity levels
use constant SEVERITY_NAMES => {
  3 => 'CRITICAL',
  2 => 'WARNING',
  1 => 'INFO',
  0 => 'MAPPING'
};
 
# names of severity levels at overall status
use constant SEVERITY_OVERALL_NAMES => {
  3 => 'Critical',
  2 => 'Warnings',
  1 => 'Info messages',
  0 => 'Mapping messages'
};

# components
use constant {
  DNS => 'DNS',
  MAIL => 'Mail',
  WEB => 'Web', 
  DB => 'Database',
  CP => 'Control Panel',
};

use constant INDENT_LEVEL => 4; # text indentation in spaces
use constant LINE_WIDTH => 80;

use PreMigrationMessages;
use CommonPreMigrationMessages;

sub check {
  print "Checking installation for migration issues, please wait...\n";
  PreMigrationChecks::checkOverall();
  DumpComposer::makeDump(
    AgentConfig::cwd(), 
    undef, undef, undef, undef, undef, undef, undef, undef
  );

  print "=" x LINE_WIDTH . "\n";
  PreMigration::printAllMessages();
  print "=" x LINE_WIDTH . "\n";

  my $pathPrefix = 'pre-migration-checker/';
  my $briefLogFilename = 'brief.log';
  my $briefLogFilenameByType = 'brief-by-message-type.log';
  my $detailedLogFilename = 'detailed.log';
  
  print "Saving brief migration log to '$pathPrefix$briefLogFilename'...\n";
  PreMigration::writeBriefLog($briefLogFilename);
  print "Saving brief migration log sorted by message type to '$pathPrefix$briefLogFilenameByType'...\n";
  PreMigration::writeBriefLogByType($briefLogFilenameByType);
  print "Saving detailed migration log to '$pathPrefix$detailedLogFilename'...\n";
  PreMigration::writeDetailedLog($detailedLogFilename);
  printOverallStatus();
}

# add message by id from PreMigrationMessages.pm
sub message {
  my ($id, $messageContext) = @_;

  my $messageInfo = _getMessageInfo($id);
  if (!defined $messageInfo) {
    print STDERR "Pre-migration message '$id' does not exist\n";
    return;
  }

  # substitute context variables
  my $briefMessage = $messageInfo->{'briefMessage'};
  my $detailedMessage = $messageInfo->{'detailedMessage'};
  foreach my $context ($messageContext, \%PreMigrationMessages::COMMON_CONTEXT) {
    foreach my $key (keys %{$context}) {
      $briefMessage =~ s/{$key}/$context->{$key}/g;
      $detailedMessage =~ s/{$key}/$context->{$key}/g;
    }
  }

  if ($messageInfo->{'severity'} == WARNING) {
    Logging::warning("Pre-migration check: $briefMessage");
  } elsif ($messageInfo->{'severity'} == CRITICAL) {
    Logging::error("Pre-migration check: $briefMessage");
  }

  _addMessage($id, $messageInfo->{'severity'}, $messageInfo->{'component'}, $briefMessage, $detailedMessage, $messageContext);
}

# assert that some condition is true, and if so - add message
sub assert {
  my ($condition, $messageId, $context) = @_;

  if ($condition) {
    message($messageId, $context);
  }
}

sub printAllMessages {
  _printAllMessagesTo(*STDOUT);
}

sub writeBriefLogByType {
  my ($filename) = @_;
  my $fp;
  
  unless (open($fp, "> $filename")) {
    Logging::error("Failed to write brief pre-migration log to '$filename'");
    return;
  }

  my %messages_by_severity = ();
  # group messages by message type
  for my $message (@MESSAGES) {
      # my $key = "$message->{'id'} ($message->{'severityText'})";
      my $key = "$message->{'id'}";
      unless (defined $messages_by_severity{$key}) {
        $messages_by_severity{$key} = [];
      }
      push @{$messages_by_severity{$key}}, $message;
  }

  # sort message types by severity
  my @msgtypes_by_severity;
  for my $msg_type (keys %messages_by_severity) {
    my $sample_msg = $messages_by_severity{$msg_type}->[0];
    push @msgtypes_by_severity, {
        'severity' => $sample_msg->{'severity'},
        'severityText' => $sample_msg->{'severityText'},
        'id' => $msg_type};
  }
  my @sorted_keys = sort { $b->{'severity'} <=> $a->{'severity'} } @msgtypes_by_severity;

  # print the messages
  for my $msg_type (@sorted_keys) {
     print $fp "$msg_type->{'id'} ($msg_type->{'severityText'})\n";
     my $msg_list = $messages_by_severity{$msg_type->{'id'}};
     for my $msg (@$msg_list) {
         print $fp _indentLines($msg->{'briefMessage'}."\n");
     }
  }
  close($fp);
}

sub writeBriefLog {
  my ($filename) = @_;
  
  if (open(my $fp, "> $filename")) {
    _printAllMessagesTo($fp);
    close($fp);
  } else {
    Logging::error("Failed to write brief pre-migration log to '$filename'");
  }
}

sub writeDetailedLog {
  my ($filename) = @_;

  my %fields = (
    'severityText' => 'Severity', 
    'component' => 'Component',
    'briefMessage' => 'Message',
    'detailedMessage' => 'Solution',
  );
  
  if (open(my $fp, "> $filename")) {
    foreach my $message (_getAllMessagesDefaultSorted()) {
      foreach my $key ('severityText', 'component', 'briefMessage', 'detailedMessage') {
        print $fp $fields{$key} . ': ' . $message->{$key} . "\n";
      }
      print $fp "--------------------------------------------------------------------------------\n";
    }
    close($fp);
  } else {
    Logging::error("Failed to write detailed pre-migration log to '$filename'");
  }
}

sub printOverallStatus {
  print "***** Overall pre-migration status *****\n";
  for my $severityLevel (@{&SEVERITY_LEVELS}) {
    my $messagesCount = scalar(grep { $_->{'severity'} == $severityLevel } @MESSAGES);
    print SEVERITY_OVERALL_NAMES->{$severityLevel} . ': ' . $messagesCount . "\n";
  }
}

sub formatList {
  my @list = @_;

  return join(', ', map { "'" . $_ . "'" } @list);
}

# @param $severity see the list of severity levels above
# @param $component see the list of components above
# @param $briefMessage brief description of the problem
# @param $detailedMessage detailed sescription of the problem, should contain a solution 
#   that could be used by customer to avoid or solve the problem
# @param $context reference to a hash with meta-information about context where the message was issued
#   for example, for domain it could be {'type' => 'domain', 'id' => 'example.com'}, for spamassassin option
#   it could be {'type' => 'spamassassion-option', 'name' => 'OPTION_NAME', 'value' => 'option_value', 'mailbox' => 'mail.example.tld'}
#   use this for filtering and sorting of the messages
sub _addMessage {
  my ($id, $severity, $component, $briefMessage, $detailedMessage, $context) = @_;

  push @MESSAGES, {
    'severity' => $severity, 
    'severityText' => SEVERITY_NAMES->{$severity},
    'component' => $component,
    'briefMessage' => $briefMessage,
    'detailedMessage' => $detailedMessage,
    'context' => $context,
    'id' => $id
  };
}

sub _printAllMessagesTo {
  my ($fp) = @_;
    
  my $rootTextTreeNode = _makeMessagesTextTree();
  _printTextTreeNode($fp, $rootTextTreeNode, '', '');
}

sub _getAllMessagesDefaultSorted {
  return _sortBySeverity(@MESSAGES);
}

sub _indentLines {
  return map { (" " x INDENT_LEVEL).$_} @_;
}

# @return hash with keys:
#    'user' - hash with keys - user (reseller or client) names, values - message lists
#    'domain' - hash with keys - domain names, values - messages lists
#    'common' - list of common (not related to a reseller, client, or domain) messages
sub _getAllMessagesGroupedByObject {
  my %result = (
    'user' => {}, 
    'domain' => {},
    'common' => []
  );

  foreach my $message (@MESSAGES) {
    if (defined($message->{'context'}->{'domain'})) {
      my $domain = $message->{'context'}->{'domain'};

      unless (defined($result{'domain'}->{$domain})) {
        $result{'domain'}->{$domain} = [];
      }

      push @{$result{'domain'}->{$domain}}, $message;
    } elsif (defined($message->{'context'}->{'user'})) {
      my $user = $message->{'context'}->{'user'};

      unless (defined($result{'user'}->{$user})) {
        $result{'user'}->{$user} = [];
      }

      push @{$result{'user'}->{$user}}, $message;
    } else {
      push @{$result{'common'}}, $message;
    }
  }

  return %result;
}

# each element of the text tree is a hash {'text' => formatted message text, 'children' => [ child elements ]},
# @return root element of a tree
sub _makeMessagesTextTree {
  my %messages = _getAllMessagesGroupedByObject();

  return DumpComposer::_walkTree(
    DumpComposer::_createFullTree(\&Agent::getPreMigrationDomains),
    my $nodeFunctions = {
      'admin' => sub { 
        my ($id, $childResults) = @_; 
        return {
          'text' => 'All',
          'children' => [ _makeObjectNode('Common', [], $messages{'common'}) , @$childResults ]
        };
      },
      'reseller' => sub { 
        my ($id, $childResults) = @_; 
        return _makeObjectNode("Reseller '$id'", $childResults, $messages{'user'}->{$id});
      },
      'client' =>   sub { 
        my ($id, $childResults) = @_; 
        return _makeObjectNode("User '$id'", $childResults, $messages{'user'}->{$id});
      },
      'domain' => sub { 
        my ($id, $childResults) = @_; 
        return _makeObjectNode("Domain '$id'", $childResults, $messages{'domain'}->{$id});
      }
    }
  );
}

sub _makeObjectNode {
  my ($objectName, $childNodes, $messages) = @_;

  my @children = (
    defined($messages) ? (map { _messageToLeafNode($_) } _sortBySeverity(@$messages)) : (),
    @$childNodes
  );

  if (scalar(@children) == 0) {
    # skip (do not display) nodes that have no warnings
    return ();
  }
  
  return { 'text' => $objectName, 'children' => \@children }; 
}

sub _messageToLeafNode {
  my ($message) = @_;

  return {
    'text' => $_->{'severityText'} . "\n" . $_->{'briefMessage'}, 
    'children' => [] # message node is always a leaf
  };
}

sub _sortBySeverity {
  my @messages = @_;

  return sort { $b->{'severity'} <=> $a->{'severity'} } @messages;
}

# print text tree node recursively with all the child nodes
# @see _makeMessagesTextTree for more details about text tree
sub _printTextTreeNode {
  my ($fp, $node, $firstLineIndent, $otherLinesIndent) = @_;

  my $text = $node->{'text'} . "\n";
  my $wrapLength = LINE_WIDTH - length($firstLineIndent);
  $text =~ s/([^\n]{0,$wrapLength})(?:\b\s*|\n)/$1\n/gi; # wrap text
  my @lines = split("\n", $text);

  my ($firstLine, @otherLines) = @lines;
  print $fp join("\n", ( 
      $firstLineIndent . $firstLine, 
      (map { $otherLinesIndent . $_ } @otherLines),
      '' 
    )
  );

  my $lastChild = pop @{$node->{'children'}};
  foreach my $child (@{$node->{'children'}}) {
    _printTextTreeNode($fp, $child, $otherLinesIndent . '|---', $otherLinesIndent . '|   ');
  }

  if (defined($lastChild)) {
    _printTextTreeNode($fp, $lastChild, $otherLinesIndent . '`---', $otherLinesIndent . '    ');
  }
}

# Search for a message with a given ID. If one is found in agent-specific messages,
# return that one, otherwise, search in common messages as well.
# @param message ID
# @return hash of message attributes
sub _getMessageInfo {
  my ($id) = @_;
  my $message;

  if (defined $PreMigrationMessages::MESSAGES{$id}) {
    return $PreMigrationMessages::MESSAGES{$id};
  } elsif (defined $CommonPreMigrationMessages::MESSAGES{$id}) {
    return $CommonPreMigrationMessages::MESSAGES{$id};
  }
  return $message;
}

1;
