package PreMigrationChecks;

use strict;
use warnings;

use PreMigration;
use Dumper;
use ConfixxConfig;

# put here all checks that could be performed independently from agent

sub checkOverall {
  PreMigration::message('PERSONAL_INFO_NOT_FULLY_MIGRATED', {});
  PreMigration::message('COMPLEX_PASSWORDS_NOT_MIGRATED', {});
  PreMigration::message('STATISTICS_MODEL', {});
  PreMigration::assert(Dumper::hasIpRestrictions(), 'IP_RESTRICTIONS', {});
  PreMigration::assert(Dumper::hasBlackListedDomains(), 'DOMAIN_BLACKLIST', {});

  my $adminInfo  = Dumper::getAdminInfo();
  my @nsTemplateRecords = grep { $_->[3] eq 'NS' } Dumper::parseDnsRecords(split "\n", $adminInfo->{'dnstemplate'});
  PreMigration::assert(scalar(@nsTemplateRecords) > 0, 'NS_TEMPLATE_RECORDS', {});
  
  my @supportedMaildrops = ("HOMEDIR/Maildir/", "HOMEDIR/Mailbox", "/var/spool/mail/USER");
  my $currentMaildrop = ConfixxConfig::getValue('maildrop');
  PreMigration::assert(
    !(grep { $_ eq  $currentMaildrop } @supportedMaildrops),
    'UNKNOWN_MAILDROP', {'maildrop' => $currentMaildrop}
  );
}

sub checkClient {
  my ($client, $clientInfo) = @_;

  my @sharedDomains = Dumper::getSharedDomains($client);
  PreMigration::assert(scalar(@sharedDomains) > 0, 'SHARED_DOMAINS', {'user' => $client, 'domains' => PreMigration::formatList(@sharedDomains)});

  my @atDomains = Dumper::getAtDomains($client);
  PreMigration::assert(scalar(@atDomains) > 0, 'AT_DOMAINS', {'user' => $client, 'atdomains' => PreMigration::formatList(@atDomains)});

  PreMigration::assert(Dumper::hasClientCustomErrorDocs($client), 'CLIENT_CUSTOM_ERROR_DOCS', {'user' => $client});
  PreMigration::assert(scalar(Dumper::getClientDatabaseNames($client)) > 1, 'MULTIPLE_DATABASES', {'client' => $client});

  my @emailsWithPercent = Dumper::getEmailsWithPercentSign($client);
  PreMigration::assert(scalar(@emailsWithPercent) > 0, 'EMAILS_WITH_PERCENT_SIGN', {'user'=> $client, 'emails' => PreMigration::formatList(@emailsWithPercent)});

  PreMigration::assert($clientInfo->{'shell'} == 0 && $clientInfo->{'scponly'} == 1, 'SCPONLY_NOT_SUPPORTED', {'user' => $client});
  PreMigration::assert($clientInfo->{'awstats'} == 1 && $clientInfo->{'statistik'} == 1, 'BOTH_AWSTATS_AND_WEBALIZER_ENABLED', {'user' => $client});

  PreMigration::assert(Dumper::hasClientCronJobs($client), 'CRON_JOBS', {'user' => $client});
  PreMigration::assert(Dumper::hasClientAddressbookEntries($client), 'ADDRESSBOOK', {'user' => $client});

  _checkPersonalInfo($client, $clientInfo);
}

sub checkReseller {
  my ($reseller, $resellerInfo) = @_;

  PreMigration::assert($resellerInfo->{'dnsspezial'} ne '', 'RESELLER_DNS_SPECIALS', {'user' => $reseller}); 
  PreMigration::assert(scalar(Dumper::getClientTemplates($reseller)) > 0, 'NO_CUSTOMER_TEMPLATES', {'user' => $reseller}); 
  
  my @exclusiveIPs = Dumper::getResellerExclusiveIPs($reseller);
  PreMigration::assert(
    scalar(@exclusiveIPs) > 0, 
    'RESELLER_EXCLUSIVE_IPS', {'user' => $reseller, 'ips' => PreMigration::formatList(map { $_->{'ip'} } @exclusiveIPs)},
  );

  PreMigration::assert(scalar($resellerInfo->{'indexcode'} =~ /##user##/), 'RESELLER_INDEX_PAGE_WITH_USER_VARIABLE', {'user' => $reseller});
  PreMigration::assert($resellerInfo->{'hilfecode'} ne '', 'RESELLER_HELP_MESSAGE', {'user' => $reseller});

  _checkPersonalInfo($reseller, $resellerInfo);
}

sub _checkPersonalInfo {
  my ($user, $info) = @_;

  PreMigration::assert(
    scalar($info->{'telefon'} && $info->{'telefon'} =~ /\//), 
    'PHONE_CONTAINS_SLASHES', {'user' => $user, 'phone' => $info->{'telefon'}},
    $info); 
  PreMigration::assert(
    scalar($info->{'fax'} && $info->{'fax'}  =~ /\//),
    'FAX_CONTAINS_SLASHES', {'user' => $user, 'fax' => $info->{'fax'}},
    $info); 

  _checkCustomPersonalInfo($user, $info);
    
  if ($info->{'land'} ne '') {
    my $countryCode = FormatConverter::convertCountry($info->{'land'});
    PreMigration::assert(!defined($countryCode), 'CAN_NOT_CONVERT_COUNTRY_CODE', {'user' => $user, 'country' => $info->{'land'}});
  }
    
  my $locale = FormatConverter::convertLocale($info->{'language'});

  # we don't know if it is really possible as there is no free-form language field
  PreMigration::assert(!defined($locale), 'CAN_NOT_CONVERT_LOCALE', {'user' => $user, 'language' => $info->{'language'}});
}

sub _checkCustomPersonalInfo {
  my ($user, $info) = @_;

  for (my $i = 1; $i <= 3; $i++) {
    if ($info->{"freifeldname$i"} ne '' || $info->{"freifeldname$i"} ne '') {
      PreMigration::message('PERSONAL_INFO_CUSTOM_FIELDS', {'user' => $user});
      return;
    }
  }
}

1;
