# visiGene.sql was originally generated by the autoSql program, which also 
# generated visiGene.c and visiGene.h.  This creates the database representation of
# an object which can be loaded and saved from RAM in a fairly 
# automatic way.

#Location of image, typically a file directory
CREATE TABLE fileLocation (
    id int auto_increment not null,	# ID of location
    name longblob not null,	# Directory path usually
              #Indices
    PRIMARY KEY(id)
);

#Name of strain (eg C57BL for a mouse)
CREATE TABLE strain (
    id int auto_increment not null,	# ID of strain
    taxon int not null,	# NCBI taxon of organism
    name varchar(255) not null,	# Name of strain
              #Indices
    PRIMARY KEY(id),
    INDEX(taxon),
    INDEX(name(8))
);

#Brain, eye, kidney, etc.  Use 'whole' for whole body
CREATE TABLE bodyPart (
    id int auto_increment not null,	# ID of body part
    name varchar(255) not null,	# Name of body part
              #Indices
    PRIMARY KEY(id),
    INDEX(name(8))
);

# Neuron, glia, etc.
CREATE TABLE cellType (
    id int auto_increment not null,	# ID of cell type
    name varchar(255),			# Name of cell type
              #Indices
    PRIMARY KEY(id),
    INDEX(name(8))
);

# A more detailed splitting up of cell types.  What type of neuron, etc.
CREATE TABLE cellSubtype (
    id int auto_increment not null,	# ID of cell subtype
    name varchar(255),			# Name of cell subtype
              #Indices
    PRIMARY KEY(id),
    INDEX(name(8))
);

#Sex of a specimen
CREATE TABLE sex (
    id int auto_increment not null,	# Sex ID
    name varchar(255) not null,	# Name of sex - male, female, hermaphrodite, mixed
              #Indices
    PRIMARY KEY(id),
    INDEX(name(8))
);


#Horizontal, coronal, whole mount, etc.
CREATE TABLE sliceType (
    id int auto_increment not null,	# ID of section
    name varchar(255) not null,	# Name of horizontal/whole mount, etc
              #Indices
    PRIMARY KEY(id),
    INDEX(name(8))
);

#Fixation conditions - 3% formaldehyde or the like
CREATE TABLE fixation (
    id int auto_increment not null,	# ID of fixation
    description varchar(255) not null,	# Text string describing fixation
              #Indices
    PRIMARY KEY(id),
    INDEX(description(8))
);

#Embedding media for slices - paraffin, etc.
CREATE TABLE embedding (
    id int auto_increment not null,	# ID of embedding
    description varchar(255) not null,	# Text string describing embedding
              #Indices
    PRIMARY KEY(id),
    INDEX(description(8))
);

#Permeablization conditions
CREATE TABLE permeablization (
    id int auto_increment not null,	# ID of treatment
    description varchar(255) not null,	# Text string describing conditions
              #Indices
    PRIMARY KEY(id),
    INDEX(description(8))
);

#Info on contributor
CREATE TABLE contributor (
    id int auto_increment not null,	# ID of contributor
    name varchar(255) not null,	# Name in format like Kent W.J.
              #Indices
    PRIMARY KEY(id),
    INDEX(name(8))
);

#Information on a journal
CREATE TABLE journal (
    id int auto_increment not null,	# ID of journal
    name varchar(255) not null,	# Name of journal
    url varchar(255) not null,	# Journal's main url
              #Indices
    PRIMARY KEY(id),
    INDEX(name(8))
);

#Copyright information
CREATE TABLE copyright (
    id int auto_increment not null,	# ID of copyright
    notice longblob not null,	# Text of copyright notice
              #Indices
    PRIMARY KEY(id),
    INDEX(notice(26))
);

# Source of data - an external database, a contributor, etc.
CREATE TABLE submissionSource (
    id int auto_increment not null, # ID of submission source"
    name varchar(255) not null,     # Short name: Jackson Lab, Paul Gray, etc.
    acknowledgement longtext not null, # Something extra to put in the caption after copyright.
    setUrl longblob not null,	# URL for whole set
    itemUrl longblob not null,	# URL for item.  Put %s where image.submitId should go
    abUrl longblob not null,	# URL for antibody.  Put %s where antibody.abSubmitId should go
              #Indices
    PRIMARY KEY(id),
    UNIQUE(name(32))
    );


#Info on a batch of images submitted at once
CREATE TABLE submissionSet (
    id int auto_increment not null,	# ID of submission set
    name varchar(255) not null,  # Name of submission set
    contributors longblob not null,	# Comma separated list of contributors in format Kent W.J., Wu F.Y.
    year int not null,			# Year of publication or submission
    publication longblob not null,	# Name of publication
    pubUrl longblob not null,	# Publication URL
    journal int not null,	# Journal for publication
    copyright int not null,     # Copyright notice
    submissionSource int not null, # Source of this submission
    privateUser int not null,	# ID of user allowed to view. If 0 all can see.
              #Indices
    PRIMARY KEY(id),
    UNIQUE(name(32))
);

#Association between contributors and submissionSets
CREATE TABLE submissionContributor (
    submissionSet int not null,	# ID in submissionSet table
    contributor int not null,	# ID in contributor table
              #Indices
    INDEX(submissionSet),
    INDEX(contributor)
);

#Info on a bunch of sections through same sample
CREATE TABLE sectionSet (
    id int auto_increment not null,	# Section id
              #Indices
    PRIMARY KEY(id)
);

#Information on an antibody
CREATE TABLE antibody (
    id int auto_increment not null,	# Antibody ID
    name varchar(255) not null,	# Name of antibody
    description longblob not null,	# Description of antibody
    taxon int not null,	# Taxon of animal antibody is from
              #Indices
    PRIMARY KEY(id),
    INDEX(name(8)),
    INDEX(taxon)
);

#Information on an antibodySource
CREATE TABLE antibodySource (
    antibody int not null,	# Antibody ID
    submissionSource int not null, # Source of this submission
    abSubmitId varchar(255) not null,	# ID within submission source
              #Indices
    INDEX(antibody)
);

#Color - what color probe is in
CREATE TABLE bac (
    id int auto_increment not null,	# Id of bac
    name varchar(255) not null,	# BAC name
              #Indices
    PRIMARY KEY(id),
    INDEX(name(8))
);

#Info on a gene
CREATE TABLE gene (
    id int auto_increment not null,	# ID of gene
    name varchar(255) not null,	# Gene symbol (HUGO if available)
    locusLink varchar(255) not null,	# NCBI locus link ID or blank if none
    refSeq varchar(255) not null,	# RefSeq ID or blank if none
    genbank varchar(255) not null,	# Genbank/EMBL accession or blank if none
    uniProt varchar(255) not null,	# SwissProt/Uniprot accession or blank if none
    taxon int not null,	# NCBI taxon ID of organism
              #Indices
    PRIMARY KEY(id),
    INDEX(name(8)),
    INDEX(locusLink(8)),
    INDEX(refSeq(10)),
    INDEX(genbank(10)),
    INDEX(uniProt(10)),
    INDEX(taxon)
);

#A synonym for a gene
CREATE TABLE geneSynonym (
    gene int not null,	# ID in gene table
    name varchar(255) not null,	# Synonymous name for gene
              #Indices
    INDEX(gene),
    INDEX(name(8))
);

#Name of a gene allele
CREATE TABLE allele (
    id int auto_increment not null,	# ID of allele
    gene int not null,	# ID of gene
    name varchar(255) not null,	# Allele name
              #Indices
    PRIMARY KEY(id),
    INDEX(name(8))
);

#How different from wild type.  Associated with genotypeAllele table
CREATE TABLE genotype (
    id int auto_increment not null,	# Genotype id
    taxon int not null,	# Taxon of organism
    strain int not null,	# Strain of organism
    alleles longblob not null,	# Comma separated list of gene:allele in alphabetical order
              #Indices
    PRIMARY KEY(id),
    INDEX(taxon),
    INDEX(strain),
    INDEX(alleles(16))
);

#Association between genotype and alleles
CREATE TABLE genotypeAllele (
    genotype int not null,	# Associated genotype
    allele int not null,	# Associated allele
              #Indices
    INDEX(genotype),
    INDEX(allele)
);

#A biological specimen - something mounted, possibly sliced up
CREATE TABLE specimen (
    id int auto_increment not null,	# Specimen ID
    name varchar(255) not null,	# Name of specimen, frequently blank
    taxon int not null,		# NCBI Taxon of specimen
    genotype int not null,	# Genotype of specimen
    bodyPart int not null,	# Body part of specimen
    sex int not null,		# Sex of specimen
    age float not null,	# Age in days since birth or conception depending on isEmbryo
    minAge float not null,	# Minimum age.
    maxAge float not null,	# Maximum age.  May differ from minAge if uncertain of age
    notes longblob not null,	# Any notes on specimen
              #Indices
    PRIMARY KEY(id),
    INDEX(name(8)),
    INDEX(taxon),
    INDEX(genotype),
    INDEX(bodyPart),
    INDEX(age)

);

#How a specimen is prepared
CREATE TABLE preparation (
    id int auto_increment not null,	# Preparation ID
    fixation int not null,	# How fixed
    embedding int not null,	# How embedded
    permeablization int not null,	# How permeablized
    sliceType int not null,	# How it was sliced
    notes longblob not null,	# Any other notes on preparation
              #Indices
    PRIMARY KEY(id),
    INDEX(fixation,embedding,permeablization,sliceType)
);

#Type of probe - RNA, antibody, etc.
CREATE TABLE probeType (
    id int auto_increment not null,	# ID of probe type
    name varchar(255) not null,	# Name of probe type
              #Indices
    PRIMARY KEY(id)
);

#Info on a probe
CREATE TABLE probe (
    id int auto_increment not null,	# ID of probe
    gene int not null,	# Associated gene if any
    antibody int not null,	# Associated antibody if any
    probeType int not null,	# Type of probe - antibody, RNA, etc..
    fPrimer varchar(255) not null,	# Forward PCR primer if any
    rPrimer varchar(255) not null,	# Reverse PCR primer if any
    seq longblob not null,	# Associated sequence if any
    bac int not null,	#Associated BAC if any
              #Indices
    PRIMARY KEY(id),
    INDEX(gene),
    INDEX(antibody),
    INDEX(bac)
);

#Color - what color probe is in
CREATE TABLE probeColor (
    id int auto_increment not null,	# Id of color
    name varchar(255) not null,	# Color name
              #Indices
    PRIMARY KEY(id),
    INDEX(name(8))
);

# An image caption.  Does not contain tabs or newlines, may have html tags
CREATE TABLE caption (
    id int auto_increment not null,	# Id of caption
    caption longtext not null,	# Caption text
              #Indices
    PRIMARY KEY(id),
    INDEX(caption(24))
);
    

#A biological image file
CREATE TABLE imageFile (
    id int auto_increment not null,	# ID of imageFile
    fileName varchar(255) not null,	# Image file name not including directory
    priority float not null,	# Lower priorities are displayed first
    imageWidth int not null,	# width of image in pixels
    imageHeight int not null,	# height of image in pixels
    fullLocation int not null,	# Location of full-size image
    thumbLocation int not null,	# Location of thumbnail-sized image
    submissionSet int not null,	# Submission set this is part of
    submitId varchar(255) not null,	# ID within submission set
    caption int not null,	# Pointer to caption or 0 for none
              #Indices
    PRIMARY KEY(id),
    INDEX(fullLocation,fileName(12)),
    INDEX(submissionSet),
    INDEX(submitId(12))
);

#An image.  There may be multiple images within an imageFile
CREATE TABLE image (
    id int auto_increment not null,	# ID of image
    submissionSet int not null, # Submission set this is part of
    imageFile int not null,	# ID of image file
    imagePos int not null,	# Position in image file, starting with 0
    paneLabel varchar(255) not null, # Label of this pane in image file
    sectionSet int not null,	# Set of sections this is part of or 0 if none
    sectionIx int not null,	# Position (0 based) within set of sections
    specimen int not null,	# Pointer to info on specimen
    preparation int not null,	# Pointer to info on how specimen prepared
              #Indices
    PRIMARY KEY(id),
    INDEX(imageFile),
    INDEX(submissionSet),
    INDEX(specimen)
);

# make imageFile forwarding table - map a Mahoney imageFile to JaxMahoney 
CREATE TABLE imageFileFwd (
    fromIf int not null,      # From imageFile
    toIf   int not null,      #   To imageFile
        #Indices
    INDEX(fromIf),
    INDEX(toIf)
);

#Associate probe and image
CREATE TABLE imageProbe (
    id int auto_increment not null, # ID of imageProbe combination
    image int not null,	# ID of image
    probe int not null,	# ID of probe
    probeColor int not null,	# ID of probeColor
              #Indices
    PRIMARY KEY(id),
    INDEX(image),
    INDEX(probe)
);

#Things like 'scattered' 'regional' 'widely expressed'
CREATE TABLE expressionPattern (
    id int auto_increment not null,	# ID of expression pattern
    description varchar(255) not null,	# Short description of pattern
              #Indices
    PRIMARY KEY(id),
    INDEX(description(8))
);

#Annotated expression level if any
CREATE TABLE expressionLevel (
    imageProbe int not null,	# Image and probe
    bodyPart int not null,	# Location of expression
    level float not null,	# Expression level (0.0 to 1.0)
    cellType int not null,      # Cell type expression seen in
    cellSubtype int not null,   # Cell subtype expression seen in
    expressionPattern int not null, # Things like scattered, regional, etc.
              #Indices
    INDEX(imageProbe),
    INDEX(bodyPart),
    INDEX(cellType),
    INDEX(cellSubtype)
);

#Information of ages critical points in life cycle
CREATE TABLE lifeTime (
    taxon int not null,	# NCBI taxon
    birth float not null,	# Typical number of days from conception to birth/hatching
    adult float not null,	# Typical number of days from conception to adulthood
    death float not null,	# Typical number of days from conception to death
              #Indices
    PRIMARY KEY(taxon)
);

# http://www.koolpages.com/hokuspokus/lifespans.html is a good source for gestation age
# and lifeSpans.  Doesn't have age of adulthood though.  
# Mouse  lifeTime
insert into lifeTime values(10090, 19, 61, 1000);
# Human lifeTime.
insert into lifeTime values(9606,  266, 5840, 28470);

#List of schemes for developmental stages
CREATE TABLE lifeStageScheme (
    id int auto_increment not null,	# ID of scheme
    taxon int not null,	# NCBI taxon
    name varchar(255) not null,	# Theiler, or whatever
              #Indices
    PRIMARY KEY(id)
);

insert into lifeStageScheme values(default, 10090, "Theiler");
insert into lifeStageScheme values(default, 8355, "Nieuwkoop/Faber");

#List of life stages according to a particular scheme
CREATE TABLE lifeStage (
    lifeStageScheme int not null,	# Which staging scheme this is
    name varchar(255) not null,	# Name of this stage
    age float not null,	# Start age of this stage measured in days since conception
    description varchar(255) not null, # One sentence description of stage
              #Indices
    INDEX(lifeStageScheme),
    INDEX(name(8)),
    INDEX(age)
);

insert into lifeStage values(1, "1", 0.0, "One-cell egg");
insert into lifeStage values(1, "2", 1.0, "Dividing egg");
insert into lifeStage values(1, "3", 2.0, "Morula");
insert into lifeStage values(1, "4", 3.0, "Blastocyst");
insert into lifeStage values(1, "5", 4.0, "Hatched blastocyst");
insert into lifeStage values(1, "6", 4.5, "Attached blastocyst");
insert into lifeStage values(1, "7", 5, "Implantation and egg cylinder");
insert into lifeStage values(1, "8", 6, "Differentiation of egg cylinder");
insert into lifeStage values(1, "9a", 6.5, "Early streak");
insert into lifeStage values(1, "10", 7, "Mid streak");
insert into lifeStage values(1, "10a", 7.17, "Late streak, no bud");
insert into lifeStage values(1, "10b", 7.33, "Late streak, early bud");
insert into lifeStage values(1, "11", 7.5, "Neural plate");
insert into lifeStage values(1, "11a", 7.63, "Late neural plate");
insert into lifeStage values(1, "11b", 7.75, "Early head fold");
insert into lifeStage values(1, "11c", 7.88, "Late head fold");
insert into lifeStage values(1, "12", 8, "1-4 somites");
insert into lifeStage values(1, "12a", 8.25, "5-7 somites");
insert into lifeStage values(1, "13", 8.5, "Embryo turns");
insert into lifeStage values(1, "14", 9, "Anterior neuropore forms and closes");
insert into lifeStage values(1, "15", 9.5, "Posterior neuropore forms");
insert into lifeStage values(1, "16", 10, "Posterior neuropore closes");
insert into lifeStage values(1, "17", 10.5, "Deep lens indentation");
insert into lifeStage values(1, "18", 11, "Lense vesicle closure");
insert into lifeStage values(1, "19", 11.5, "Lense vesicle separates");
insert into lifeStage values(1, "20", 12, "First sign of fingers");
insert into lifeStage values(1, "21", 13, "Fore footplate indented");
insert into lifeStage values(1, "22", 14, "Fingertips separate");
insert into lifeStage values(1, "23", 15, "Fingers and toes separate");
insert into lifeStage values(1, "24", 16, "Reposition of umbilical hernia");
insert into lifeStage values(1, "25", 17, "Wrinkled skin");
insert into lifeStage values(1, "26", 18, "Long whiskers");
insert into lifeStage values(1, "27", 19, "Newborn");
insert into lifeStage values(1, "28", 20, "Pup");
insert into lifeStage values(1, "adult", 61, "Adult");
insert into lifeStage values(1, "elder", 1000, "Elder");

insert into lifeStage values(2, "St1", 0.00, "One-cell egg");
insert into lifeStage values(2, "St2", 0.06, "Two-cell");
insert into lifeStage values(2, "St3", 0.08, "Four-cell");
insert into lifeStage values(2, "St4", 0.09, "Eight-cell");
insert into lifeStage values(2, "St5", 0.11, "16-cell");
insert into lifeStage values(2, "St6", 0.12, "32-cell");
insert into lifeStage values(2, "St6.5", 0.15, "Blastula");
insert into lifeStage values(2, "St7", 0.17, "Blastula");
insert into lifeStage values(2, "St8", 0.21, "Blastula");
insert into lifeStage values(2, "St9", 0.29, "Blastula");
insert into lifeStage values(2, "St10", 0.38, "Early gastrula");
insert into lifeStage values(2, "St10.5", 0.46, "Early gastrula");
insert into lifeStage values(2, "St11", 0.49, "Middle gastrula");
insert into lifeStage values(2, "St11.5", 0.52, "Late gastrula");
insert into lifeStage values(2, "St12", 0.55, "Formation of neural plate");
insert into lifeStage values(2, "St12.5", 0.59, "Definite neural plate");
insert into lifeStage values(2, "St13", 0.61, "Early neurula");
insert into lifeStage values(2, "St14", 0.68, "Neurula");
insert into lifeStage values(2, "St15", 0.73, "Neurula");
insert into lifeStage values(2, "St16", 0.76, "");
insert into lifeStage values(2, "St17", 0.78, "");
insert into lifeStage values(2, "St18", 0.82, "");
insert into lifeStage values(2, "St19", 0.86, "");
insert into lifeStage values(2, "St20", 0.91, "");
insert into lifeStage values(2, "St21", 0.94, "");
insert into lifeStage values(2, "St22", 1.00, "");
insert into lifeStage values(2, "St23", 1.03, "");
insert into lifeStage values(2, "St24", 1.09, "");
insert into lifeStage values(2, "St25", 1.15, "");
insert into lifeStage values(2, "St26", 1.23, "");
insert into lifeStage values(2, "St27", 1.30, "");
insert into lifeStage values(2, "St28", 1.35, "");
insert into lifeStage values(2, "St29", 1.42, "");
insert into lifeStage values(2, "St30", 1.50, "");
insert into lifeStage values(2, "St31", 1.56, "");
insert into lifeStage values(2, "St32", 1.67, "");
insert into lifeStage values(2, "St33", 1.79, "");
insert into lifeStage values(2, "St34", 1.88, "");
insert into lifeStage values(2, "St35", 2.04, "");
insert into lifeStage values(2, "St36", 2.12, "");
insert into lifeStage values(2, "St37", 2.19, "");
insert into lifeStage values(2, "St38", 2.27, "");
insert into lifeStage values(2, "St39", 2.35, "");
insert into lifeStage values(2, "St40", 2.75, "");
insert into lifeStage values(2, "St41", 3.17, "");
insert into lifeStage values(2, "St42", 3.33, "");
insert into lifeStage values(2, "St43", 3.62, "");
insert into lifeStage values(2, "St44", 3.85, "");
insert into lifeStage values(2, "St45", 4.08, "");
insert into lifeStage values(2, "St46", 4.42, "");
