/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.gui.desktop;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.regex.Pattern;
import javax.swing.JOptionPane;
import org.jabref.Globals;
import org.jabref.JabRefGUI;
import org.jabref.gui.ClipBoardManager;
import org.jabref.gui.IconTheme;
import org.jabref.gui.JabRefFrame;
import org.jabref.gui.desktop.os.DefaultDesktop;
import org.jabref.gui.desktop.os.Linux;
import org.jabref.gui.desktop.os.NativeDesktop;
import org.jabref.gui.desktop.os.OSX;
import org.jabref.gui.desktop.os.Windows;
import org.jabref.gui.externalfiletype.ExternalFileType;
import org.jabref.gui.externalfiletype.ExternalFileTypeEntryEditor;
import org.jabref.gui.externalfiletype.ExternalFileTypes;
import org.jabref.gui.externalfiletype.UnknownExternalFileType;
import org.jabref.gui.filelist.FileListEntry;
import org.jabref.gui.filelist.FileListEntryEditor;
import org.jabref.gui.filelist.FileListTableModel;
import org.jabref.gui.undo.UndoableFieldChange;
import org.jabref.logic.l10n.Localization;
import org.jabref.logic.util.OS;
import org.jabref.model.database.BibDatabaseContext;
import org.jabref.model.entry.BibEntry;
import org.jabref.model.entry.identifier.DOI;
import org.jabref.model.entry.identifier.Eprint;
import org.jabref.model.util.FileHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JabRefDesktop {
    private static final Logger LOGGER = LoggerFactory.getLogger(JabRefDesktop.class);
    private static final NativeDesktop NATIVE_DESKTOP = JabRefDesktop.getNativeDesktop();
    private static final Pattern REMOTE_LINK_PATTERN = Pattern.compile("[a-z]+://.*");

    private JabRefDesktop() {
    }

    public static void openExternalViewer(BibDatabaseContext databaseContext, String initialLink, String initialFieldName) throws IOException {
        String link = initialLink;
        String fieldName = initialFieldName;
        if ("ps".equals(fieldName) || "pdf".equals(fieldName)) {
            List<String> dir = databaseContext.getFileDirectories(fieldName, Globals.prefs.getFileDirectoryPreferences());
            Optional<Path> file = FileHelper.expandFilename(link, dir);
            if (!file.isPresent() || !Files.exists(file.get(), new LinkOption[0])) {
                throw new IOException("File not found (" + fieldName + "): '" + link + "'.");
            }
            link = file.get().toAbsolutePath().toString();
            String[] split = file.get().getFileName().toString().split("\\.");
            if (split.length >= 2) {
                if ("pdf".equalsIgnoreCase(split[split.length - 1])) {
                    fieldName = "pdf";
                } else if ("ps".equalsIgnoreCase(split[split.length - 1]) || split.length >= 3 && "ps".equalsIgnoreCase(split[split.length - 2])) {
                    fieldName = "ps";
                }
            }
        } else {
            if ("doi".equals(fieldName)) {
                JabRefDesktop.openDoi(link);
                return;
            }
            if ("eprint".equals(fieldName)) {
                link = Eprint.build(link).map(Eprint::getURIAsASCIIString).orElse(link);
                fieldName = "url";
            }
        }
        if ("url".equals(fieldName)) {
            JabRefDesktop.openBrowser(link);
        } else if ("ps".equals(fieldName)) {
            try {
                NATIVE_DESKTOP.openFile(link, "ps");
            }
            catch (IOException e) {
                LOGGER.error("An error occurred on the command: " + link, e);
            }
        } else if ("pdf".equals(fieldName)) {
            try {
                NATIVE_DESKTOP.openFile(link, "pdf");
            }
            catch (IOException e) {
                LOGGER.error("An error occurred on the command: " + link, e);
            }
        } else {
            LOGGER.info("Message: currently only PDF, PS and HTML files can be opened by double clicking");
        }
    }

    private static void openDoi(String doi) throws IOException {
        String link = DOI.parse(doi).map(DOI::getURIAsASCIIString).orElse(doi);
        JabRefDesktop.openBrowser(link);
    }

    public static boolean openExternalFileAnyFormat(BibDatabaseContext databaseContext, String link, Optional<ExternalFileType> type) throws IOException {
        Optional<Path> tmp;
        boolean httpLink = false;
        if (REMOTE_LINK_PATTERN.matcher(link.toLowerCase(Locale.ROOT)).matches()) {
            httpLink = true;
        }
        Path file = null;
        if (!httpLink && (tmp = FileHelper.expandFilename(databaseContext, link, Globals.prefs.getFileDirectoryPreferences())).isPresent()) {
            file = tmp.get();
        }
        if (httpLink || file != null && Files.exists(file, new LinkOption[0]) && type.isPresent()) {
            String filePath = httpLink ? link : file.toString();
            JabRefDesktop.openExternalFilePlatformIndependent(type, filePath);
            return true;
        }
        return false;
    }

    public static boolean openExternalFileAnyFormat(Path file, BibDatabaseContext databaseContext, Optional<ExternalFileType> type) throws IOException {
        return JabRefDesktop.openExternalFileAnyFormat(databaseContext, file.toString(), type);
    }

    private static void openExternalFilePlatformIndependent(Optional<ExternalFileType> fileType, String filePath) throws IOException {
        if (fileType.isPresent()) {
            String application = fileType.get().getOpenWithApplication();
            if (application.isEmpty()) {
                NATIVE_DESKTOP.openFile(filePath, fileType.get().getExtension());
            } else {
                NATIVE_DESKTOP.openFileWithApplication(filePath, application);
            }
        }
    }

    public static boolean openExternalFileUnknown(JabRefFrame frame, BibEntry entry, BibDatabaseContext databaseContext, String link, UnknownExternalFileType fileType) throws IOException {
        String cancelMessage = Localization.lang("Unable to open file.", new String[0]);
        Object[] options = new String[]{Localization.lang("Define '%0'", fileType.getName()), Localization.lang("Change file type", new String[0]), Localization.lang("Cancel", new String[0])};
        String defOption = options[0];
        int answer = JOptionPane.showOptionDialog(frame, Localization.lang("This external link is of the type '%0', which is undefined. What do you want to do?", fileType.getName()), Localization.lang("Undefined file type", new String[0]), 1, 3, null, options, defOption);
        if (answer == 2) {
            frame.output(cancelMessage);
            return false;
        }
        if (answer == 0) {
            ExternalFileType newType = new ExternalFileType(fileType.getName(), fileType.getExtension(), "", "", "new", IconTheme.JabRefIcon.FILE.getSmallIcon());
            ExternalFileTypeEntryEditor editor = new ExternalFileTypeEntryEditor(frame, newType);
            editor.setVisible(true);
            if (editor.okPressed()) {
                ArrayList<ExternalFileType> fileTypes = new ArrayList<ExternalFileType>(ExternalFileTypes.getInstance().getExternalFileTypeSelection());
                fileTypes.add(newType);
                Collections.sort(fileTypes);
                ExternalFileTypes.getInstance().setExternalFileTypes(fileTypes);
                return JabRefDesktop.openExternalFileAnyFormat(databaseContext, link, Optional.of(newType));
            }
            frame.output(cancelMessage);
            return false;
        }
        FileListTableModel tModel = new FileListTableModel();
        Optional<String> oldValue = entry.getField("file");
        oldValue.ifPresent(tModel::setContent);
        FileListEntry flEntry = null;
        for (int i = 0; i < tModel.getRowCount(); ++i) {
            FileListEntry iEntry = tModel.getEntry(i);
            if (!iEntry.getLink().equals(link)) continue;
            flEntry = iEntry;
            break;
        }
        if (flEntry == null) {
            throw new RuntimeException("Could not find the file list entry " + link + " in " + entry);
        }
        FileListEntryEditor editor = new FileListEntryEditor(flEntry.toParsedFileField(), false, true, databaseContext);
        editor.setVisible(true, false);
        if (editor.okPressed()) {
            String newValue = tModel.getStringRepresentation();
            UndoableFieldChange ce = new UndoableFieldChange(entry, "file", oldValue.orElse(null), newValue);
            entry.setField("file", newValue);
            frame.getCurrentBasePanel().getUndoManager().addEdit(ce);
            frame.getCurrentBasePanel().markBaseChanged();
            return JabRefDesktop.openExternalFileAnyFormat(databaseContext, flEntry.getLink(), flEntry.getType());
        }
        frame.output(cancelMessage);
        return false;
    }

    public static void openFolderAndSelectFile(Path fileLink) throws IOException {
        NATIVE_DESKTOP.openFolderAndSelectFile(fileLink);
    }

    public static void openBrowser(String url) throws IOException {
        Optional<ExternalFileType> fileType = ExternalFileTypes.getInstance().getExternalFileTypeByExt("html");
        JabRefDesktop.openExternalFilePlatformIndependent(fileType, url);
    }

    public static void openBrowser(URI url) throws IOException {
        JabRefDesktop.openBrowser(url.toASCIIString());
    }

    public static void openBrowserShowPopup(String url) {
        try {
            JabRefDesktop.openBrowser(url);
        }
        catch (IOException exception) {
            new ClipBoardManager().setClipboardContents(url);
            LOGGER.error("Could not open browser", exception);
            String couldNotOpenBrowser = Localization.lang("Could not open browser.", new String[0]);
            String openManually = Localization.lang("Please open %0 manually.", url);
            String copiedToClipboard = Localization.lang("The link has been copied to the clipboard.", new String[0]);
            JabRefGUI.getMainFrame().output(couldNotOpenBrowser);
            JOptionPane.showMessageDialog(JabRefGUI.getMainFrame(), couldNotOpenBrowser + "\n" + openManually + "\n" + copiedToClipboard, couldNotOpenBrowser, 0);
        }
    }

    public static void openConsole(File file) throws IOException {
        if (file == null) {
            return;
        }
        String absolutePath = file.toPath().toAbsolutePath().getParent().toString();
        boolean usingDefault = Globals.prefs.getBoolean("useDefaultConsoleApplication");
        if (usingDefault) {
            NATIVE_DESKTOP.openConsole(absolutePath);
        } else {
            String command = Globals.prefs.get("consoleCommand");
            if (!(command = command.trim()).isEmpty()) {
                command = command.replaceAll("\\s+", " ");
                String[] subcommands = command.split(" ");
                String commandLoggingText = command.replace("%DIR", absolutePath);
                JabRefGUI.getMainFrame().output(Localization.lang("Executing command \"%0\"...", commandLoggingText));
                LOGGER.info("Executing command \"" + commandLoggingText + "\"...");
                try {
                    new ProcessBuilder(subcommands).start();
                }
                catch (IOException exception) {
                    LOGGER.error("Open console", exception);
                    JOptionPane.showMessageDialog(JabRefGUI.getMainFrame(), Localization.lang("Error occured while executing the command \"%0\".", commandLoggingText), Localization.lang("Open console", new String[0]) + " - " + Localization.lang("Error", new String[0]), 0);
                    JabRefGUI.getMainFrame().output(null);
                }
            }
        }
    }

    public static NativeDesktop getNativeDesktop() {
        if (OS.WINDOWS) {
            return new Windows();
        }
        if (OS.OS_X) {
            return new OSX();
        }
        if (OS.LINUX) {
            return new Linux();
        }
        return new DefaultDesktop();
    }
}

